﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Net;
using System.Text;
using System.Xml;

using SystemNeo.Collections;
using SystemNeo.Net;

namespace SystemNeo.Net
{
	/// <summary>
	/// XML-RPC を利用してネットワーク上のサービスを呼び出します。
	/// </summary>
	public class XmlRpc
	{
		// public 定数 //

		public const string ArrayTagName = "array";
		public const string BooleanTagName = "boolean";
		public const string DataTagName = "data";
		public const string DoubleTagName = "double";
		public const string MemberTagName = "member";
		public const string MethodCallTagName = "methodCall";
		public const string MethodNameTagName = "methodName";
		public const string I4TagName = "i4";
		public const string IntTagName = "int";
		public const string NameTagName = "name";
		public const string NilTagName = "nil";
		public const string ParamsTagName = "params";
		public const string ParamTagName = "param";
		public const string ResponseValuePath = "/methodResponse/params/param/value";
		public const string StringTagName = "string";
		public const string StructTagName = "struct";
		public const string ValueTagName = "value";

		// public プロパティ //

		/// <summary>
		/// 
		/// </summary>
		public string Uri { get; set; }

		// public コンストラクタ //

		/// <summary>
		/// 
		/// </summary>
		public XmlRpc() {}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="uri"></param>
		public XmlRpc(string uri)
		{
			this.Uri = uri;
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="uri"></param>
		public XmlRpc(Uri uri) : this(uri.ToString()) {}

		// public メソッド //

		/// <summary>
		/// 
		/// </summary>
		/// <param name="methodName"></param>
		/// <param name="params"></param>
		/// <returns></returns>
		public object Call(string methodName, params object[] @params)
		{
			XmlDocument requestDoc = CreateRequestDocument(methodName, @params);

			var request = (HttpWebRequest)WebRequest.Create(this.Uri);
			request.ContentType = MimeTypes.Xml;
			var wce = new WebClientEx(request);
			using (Stream stream = wce.GetPostStream()) {
				requestDoc.Save(stream);
			}
			var responseDoc = new XmlDocument();
			WebHeaderCollection headers;
			using (var ms = new MemoryStream()) {
				bool result = wce.DownloadData(ms, false, out headers);
				if (! result) {
					throw new WebException();
				}
				ms.Position = 0;
				responseDoc.Load(ms);
			}
			var valueElem = (XmlElement)responseDoc.SelectSingleNode(ResponseValuePath);
			return ParseValueElement(valueElem);
		}

		// private static メソッド //

		/// <summary>
		/// 
		/// </summary>
		/// <param name="methodName"></param>
		/// <param name="params"></param>
		/// <returns></returns>
		private static XmlDocument CreateRequestDocument(string methodName, params object[] @params)
		{
			var doc = new XmlDocument();
			XmlNode rootElem = doc.AppendChild(doc.CreateElement(MethodCallTagName));

			XmlElement methodNameElem = doc.CreateElement(MethodNameTagName);
			rootElem.AppendChild(methodNameElem);
			methodNameElem.AppendChild(doc.CreateTextNode(methodName));

			XmlElement paramsElem = doc.CreateElement(ParamsTagName);
			rootElem.AppendChild(paramsElem);
			foreach (object param in @params) {
				XmlElement paramElem = doc.CreateElement(ParamTagName);
				paramsElem.AppendChild(paramElem);
				paramElem.AppendChild(CreateValueElement(doc, param));
			}
			
			return doc;
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="doc"></param>
		/// <param name="value"></param>
		/// <returns></returns>
		private static XmlElement CreateValueElement(XmlDocument doc, object value)
		{
			XmlElement valueElem = doc.CreateElement(ValueTagName);
			XmlElement childElem = null;
			if (value == null) {
				childElem = doc.CreateElement(NilTagName);
			} else if (value is bool) {
				childElem = doc.CreateElement(BooleanTagName);
				childElem.AppendChild(doc.CreateTextNode((bool)value ? "1" : "0"));
			} else if (value is int) {
				childElem = doc.CreateElement(I4TagName);
				childElem.AppendChild(doc.CreateTextNode(value.ToString()));
			} else if (value is double) {
				childElem = doc.CreateElement(DoubleTagName);
				childElem.AppendChild(doc.CreateTextNode(value.ToString()));
			} else if (value is string || value is Enum) {
				childElem = doc.CreateElement(StringTagName);
				childElem.AppendChild(doc.CreateTextNode(value.ToString()));
			} else if (value is IDictionary) {
				childElem = doc.CreateElement(StructTagName);
				foreach (DictionaryEntry entry in (IDictionary)value) {
					XmlElement memberElem = doc.CreateElement(MemberTagName);
					XmlElement nameElem = doc.CreateElement(NameTagName);
					nameElem.AppendChild(doc.CreateTextNode(Convert.ToString(entry.Key)));
					memberElem.AppendChild(nameElem);
					memberElem.AppendChild(CreateValueElement(doc, entry.Value));
					childElem.AppendChild(memberElem);
				}
			}
			valueElem.AppendChild(childElem);
			return valueElem;
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="valueElem"></param>
		/// <returns></returns>
		private static object ParseValueElement(XmlElement valueElem)
		{
			if (valueElem.Name != ValueTagName) {
				throw new ArgumentException();
			}
			XmlNodeList children = valueElem.SelectNodes("*");
			switch (children.Count) {
			case 0:
				return valueElem.InnerText;
			case 1:
				break;
			default:
				throw new ArgumentException();
			}
			string text = children[0].InnerText;
			switch (children[0].Name) {
			case NilTagName:
				return null;
			case BooleanTagName:
				return (text != "0");
			case I4TagName:
			case IntTagName:
				return int.Parse(text);
			case DoubleTagName:
				return double.Parse(text);
			case StringTagName:
				return text;
			case ArrayTagName:
				XmlNodeList valueNodes = children[0].SelectNodes(DataTagName + "/" + ValueTagName);
				return valueNodes.CastSelect<XmlElement, object>(ParseValueElement).ToArray();
			case StructTagName:
				var dic = new Hashtable();
				foreach (XmlElement memberElem in children[0].ChildNodes) {
					var nameElem = (XmlElement)memberElem.SelectSingleNode(NameTagName);
					XmlNode valueNode = memberElem.SelectSingleNode(ValueTagName);
					dic[nameElem.InnerText] = ParseValueElement((XmlElement)valueNode);
				}
				return dic;
			}
			throw new ArgumentException();
		}
	}
}
