package jp.osoite.tomu.itree.util;
////////////////////////////////////////////////////////////////////////
// DirEntry

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import jp.osoite.tomu.itree.core.ITDataNode;
import jp.osoite.tomu.itree.core.ITDirNode;
import jp.osoite.tomu.itree.core.ITNode;
import jp.osoite.tomu.itree.core.ITreeCore;

////////////////////////////////////////////////////////////////////////

/**
 * DirEntry implements the entries of a directory node (ITDirNode)

 * the info of the DirEntry in a ITDirNode block is organised as follows:
 * +-------------+-----+------------------------+-----+-------------+
 * | bounces[0]  | ... | bounces[2*dimension-1] | son | num_of_data |
 * +-------------+-----+------------------------+-----+-------------+
 */


public class DirEntry {

    ITreeCore my_tree;                      // pointer to my R-tree
    private int son;                            // block # of son
    private ITNode son_ptr;              		// pointer to son if in main mem.
    private boolean son_is_data;                // TRUE, if son is a data page
    int dimension;                      // dimension of the box
    private float bounces[];                    // the mbr of the box
    int son_level;                      // level of the node pointed to
    private int num_of_data;                    // amount of data entries behind the
    // son of this entry

    public DirEntry(int _dimension, boolean son_is_data, ITreeCore rt) {
        dimension = _dimension;
        this.setSon_is_data(son_is_data);
        my_tree = rt;
        setBounces(new float[2 * dimension]);
        setSon_ptr(null);
        setNum_of_data(0);
    }

    /**
     * copy the contents of this to another Direntry object
     */
    void copyTo(DirEntry target) {
        target.dimension = dimension;
        target.setSon(son);
        target.setSon_ptr(son_ptr);
        target.son_level = son_level;
        target.setSon_is_data(son_is_data);
        target.setNum_of_data(num_of_data);
        System.arraycopy(getBounces(), 0, target.getBounces(), 0, 2 * dimension);
    }

    /**
     * Checks if point v is inside the entry's MBR
     */
    public boolean is_inside(float v[]) {
        int i;

        for (i = 0; i < dimension; i++) {
            if (v[i] < getBounces()[2 * i] || // upper limit
                    v[i] > getBounces()[2 * i + 1]) // lower limit
            {
                return false;
            }
        }
        return true;
    }

    /**
     * Tests if the parameter mbr is inside or overlaps the MBR of the entry
     */
    public int section(float mbr[]) {
        boolean inside;
        boolean overlap;
        int i;

        overlap = true;
        inside = true;
        for (i = 0; i < dimension; i++) {
            if (mbr[2 * i] > getBounces()[2 * i + 1]
                    || mbr[2 * i + 1] < getBounces()[2 * i]) {
                overlap = false;
            }
            if (mbr[2 * i] < getBounces()[2 * i]
                    || mbr[2 * i + 1] > getBounces()[2 * i + 1]) {
                inside = false;
            }
        }
        if (inside) {
            return Constants.INSIDE;
        } else if (overlap) {
            return Constants.OVERLAP;
        } else {
            return Constants.S_NONE;
        }
    }

    /**
     * reads from the input stream the object's info
     * used by RTDirNode.read_from_buffer()
     */
    public void read_from_buffer(DataInputStream in) throws IOException {
        for (int i = 0; i < 2 * dimension; ++i) {
            getBounces()[i] = in.readFloat();
        }
        setSon(in.readInt());
        setNum_of_data(in.readInt());
    }

    /**
     * writes to the output stream the object's info
     * used by RTDirNode.write_to_buffer()
     */
    public void write_to_buffer(DataOutputStream out) throws IOException {
        for (int i = 0; i < 2 * dimension; ++i) {
            out.writeFloat(getBounces()[i]);
        }
        out.writeInt(getSon());
        out.writeInt(getNum_of_data());
    }

    public int get_size() {
        return 2 * dimension * Constants.SIZEOF_FLOAT
                + Constants.SIZEOF_INT
                + Constants.SIZEOF_INT;
    }

    /**
     * returns the son_ptr (the node this entry points to
     * if the node is not in main memory, it it read
     * from disk (see RTDirNode/RTDataNode constructor)
     */
    public ITNode get_son() {
        if (getSon_ptr() == null) {
            if (isSon_is_data()) {
                setSon_ptr(new ITDataNode(my_tree, getSon()));
            } else {
                setSon_ptr(new ITDirNode(my_tree, getSon()));
            }
        }
        return getSon_ptr();
    }

    /**
     * returns true if the entry intersects the circle
     */
    public boolean section_circle(PPoint center, float radius) {
        return Constants.section_c(dimension, getBounces(), center, radius);
    }

    /**
     * returns true if the entry intersects the ring
     */
    public boolean section_ring(PPoint center, float radius1, float radius2) {
        return Constants.section_ring(dimension, getBounces(), center, radius1, radius2);
    }

    public void delete() {
        if (getSon_ptr() != null) {
            ((Node) getSon_ptr()).delete();
        }
    }

    public void setBounces(float bounces[]) {
        this.bounces = bounces;
    }

    public float[] getBounces() {
        return bounces;
    }

    public void setSon(int son) {
        this.son = son;
    }

    public int getSon() {
        return son;
    }

    public void setSon_ptr(ITNode son_ptr) {
        this.son_ptr = son_ptr;
    }

    public ITNode getSon_ptr() {
        return son_ptr;
    }

    public void setSon_is_data(boolean son_is_data) {
        this.son_is_data = son_is_data;
    }

    public boolean isSon_is_data() {
        return son_is_data;
    }

    public void setNum_of_data(int num_of_data) {
        this.num_of_data = num_of_data;
    }

    public int getNum_of_data() {
        return num_of_data;
    }
}
