#pragma once

namespace sf { 

class Win32ErrorException : std::exception 
{
public:
	Win32ErrorException(boost::uint32_t hr);
	Win32ErrorException();

	virtual ~Win32ErrorException() {};
	boost::uint32_t getHresult() {return hresult_;}
	std::wstring& error() {return error_;}
private:
	boost::uint32_t hresult_;
	std::wstring error_;
};

class WASAPI:public juce::AudioIODevice
{
public:
	virtual ~WASAPI();
	WASAPI(const juce::String & deviceName,int outputDeviceIndex,const juce::String& outputDeviceId,int inputDeviceIndex,const juce::String& inputDeviceId);
	//==============================================================================
    /** Returns the names of all the available output channels on this device.
        To find out which of these are currently in use, call getActiveOutputChannels().
    */
	virtual const juce::StringArray getOutputChannelNames();

    /** Returns the names of all the available input channels on this device.
        To find out which of these are currently in use, call getActiveInputChannels().
    */
    virtual const juce::StringArray getInputChannelNames();

    //==============================================================================
    /** Returns the number of sample-rates this device supports.

        To find out which rates are available on this device, use this method to
        find out how many there are, and getSampleRate() to get the rates.

        @see getSampleRate
    */
    virtual int getNumSampleRates();

    /** Returns one of the sample-rates this device supports.

        To find out which rates are available on this device, use getNumSampleRates() to
        find out how many there are, and getSampleRate() to get the individual rates.

        The sample rate is set by the open() method.

        (Note that for DirectSound some rates might not work, depending on combinations
        of i/o channels that are being opened).

        @see getNumSampleRates
    */
    virtual double getSampleRate (int index);

    /** Returns the number of sizes of buffer that are available.

        @see getBufferSizeSamples, getDefaultBufferSize
    */
    virtual int getNumBufferSizesAvailable();

    /** Returns one of the possible buffer-sizes.

        @param index    the index of the buffer-size to use, from 0 to getNumBufferSizesAvailable() - 1
        @returns a number of samples
        @see getNumBufferSizesAvailable, getDefaultBufferSize
    */
    virtual int getBufferSizeSamples (int index);

    /** Returns the default buffer-size to use.

        @returns a number of samples
        @see getNumBufferSizesAvailable, getBufferSizeSamples
    */
    virtual int getDefaultBufferSize();

    //==============================================================================
    /** Tries to open the device ready to play.

        @param inputChannels        a BitArray in which a set bit indicates that the corresponding
                                    input channel should be enabled
        @param outputChannels       a BitArray in which a set bit indicates that the corresponding
                                    output channel should be enabled
        @param sampleRate           the sample rate to try to use - to find out which rates are
                                    available, see getNumSampleRates() and getSampleRate()
        @param bufferSizeSamples    the size of i/o buffer to use - to find out the available buffer
                                    sizes, see getNumBufferSizesAvailable() and getBufferSizeSamples()
        @returns    an error description if there's a problem, or an empty string if it succeeds in
                    opening the device
        @see close
    */
	virtual const juce::String open (const juce::BitArray& inputChannels,
								const juce::BitArray& outputChannels,
                               double sampleRate,
                               int bufferSizeSamples);

    /** Closes and releases the device if it's open. */
    virtual void close();

    /** Returns true if the device is still open.

        A device might spontaneously close itself if something goes wrong, so this checks if
        it's still open.
    */
    virtual bool isOpen();

    /** Starts the device actually playing.

        This must be called after the device has been opened.

        @param callback     the callback to use for streaming the data.
        @see AudioIODeviceCallback, open
    */
    virtual void start (juce::AudioIODeviceCallback* callback);

    /** Stops the device playing.

        Once a device has been started, this will stop it. Any pending calls to the
        callback class will be flushed before this method returns.
    */
    virtual void stop();

    /** Returns true if the device is still calling back.

        The device might mysteriously stop, so this checks whether it's
        still playing.
    */
    virtual bool isPlaying();

    /** Returns the last error that happened if anything went wrong. */
    virtual const juce::String getLastError() ;

    //==============================================================================
    /** Returns the buffer size that the device is currently using.

        If the device isn't actually open, this value doesn't really mean much.
    */
    virtual int getCurrentBufferSizeSamples() ;

    /** Returns the sample rate that the device is currently using.

        If the device isn't actually open, this value doesn't really mean much.
    */
    virtual double getCurrentSampleRate() ;

    /** Returns the device's current physical bit-depth.

        If the device isn't actually open, this value doesn't really mean much.
    */
    virtual int getCurrentBitDepth() ;

    /** Returns a mask showing which of the available output channels are currently
        enabled.
        @see getOutputChannelNames
    */
    virtual const juce::BitArray getActiveOutputChannels() const ;

    /** Returns a mask showing which of the available input channels are currently
        enabled.
        @see getInputChannelNames
    */
    virtual const juce::BitArray getActiveInputChannels() const ;

    /** Returns the device's output latency.

        This is the delay in samples between a callback getting a block of data, and
        that data actually getting played.
    */
    virtual int getOutputLatencyInSamples() ;

    /** Returns the device's input latency.

        This is the delay in samples between some audio actually arriving at the soundcard,
        and the callback getting passed this block of data.
    */
    virtual int getInputLatencyInSamples() ;


    //==============================================================================
    /** True if this device can show a pop-up control panel for editing its settings.

        This is generally just true of ASIO devices. If true, you can call showControlPanel()
        to display it.
    */
    virtual bool hasControlPanel() const;

    /** Shows a device-specific control panel if there is one.

        This should only be called for devices which return true from hasControlPanel().
    */
    virtual bool showControlPanel();
	int getInputDeviceIndex();
	int getOutputDeviceIndex();
protected:
private:
	struct impl;
	boost::shared_ptr<impl> impl_;
};

enum EDataFlow
    {	eRender	= 0,
	eCapture	= ( eRender + 1 ) ,
	eAll	= ( eCapture + 1 ) ,
	EDataFlow_enum_count	= ( eAll + 1 ) 
};

// Enumeration: audio sources
enum EAudioSourceType
{
    eNullSource = 0,
    eToneGenerator,
    eWaveFile,
    eCaptureEndpoint

};

enum ERole
{	eConsole	= 0,
eMultimedia	= ( eConsole + 1 ) ,
eCommunications	= ( eMultimedia + 1 ) ,
ERole_enum_count	= ( eCommunications + 1 ) 
};

enum AVRT_PRIORITY
{
    AVRT_PRIORITY_LOW = -1,
    AVRT_PRIORITY_NORMAL,
    AVRT_PRIORITY_HIGH,
    AVRT_PRIORITY_CRITICAL
};
// Notification callbacks from Player to client.
// The client implements the callback methods.
class PlayerCallbacks
{
public:
    // Player calls this method when playback volume level changes.
    virtual void VolumeChangeCallback(float volume, bool mute) = 0;

    // Player calls this method when playback stream stops.
    virtual void PlayerStopCallback() = 0;

    // Player calls this method when capture device is unplugged.
    virtual void CaptureDisconnectCallback() = 0;

    // Player calls this method when rendering device is unplugged.
    virtual void RenderDisconnectCallback() = 0;
};


struct WASAPIInterface
{
    virtual void setExclusiveMode(bool enable) = 0;
    virtual const bool getExclusiveMode() const = 0;
	virtual const int getDefaultPeriod() const = 0;
	virtual const int getMinPeriod() const = 0;
    virtual void setBufferLatency(int bufferLatency) = 0;
    virtual const int getBufferLatency() const = 0;
    virtual void enableMMCSS(bool enable) = 0;
    virtual const bool isMMCSSEnabled() const = 0;
    virtual const bool isHwVolumeSupported() const = 0;

	virtual void setMmThreadPriority(AVRT_PRIORITY priority) = 0;
    virtual const AVRT_PRIORITY getMmThreadPriority() const = 0;

	virtual void setVolume(float volume) = 0;
	virtual const float getVolume() const = 0;
};

class AudioDeviceManager : public juce::AudioDeviceManager
{
public:
	AudioDeviceManager() : juce::AudioDeviceManager(){}
	virtual void createAudioDeviceTypes (juce::OwnedArray <juce::AudioIODeviceType>& types);
};

}

namespace boost {
	namespace serialization{
		template<class Archive> inline void save(Archive& ar, const sf::WASAPIInterface& data, const unsigned int /*nVersion*/)
		{
			const bool dt = data.getExclusiveMode();
			ar & data.getExclusiveMode();
			ar & data.getBufferLatency();
			ar & data.isMMCSSEnabled(); 
			ar & data.getMmThreadPriority();
			ar & data.getVolume();
		};

		template<class Archive> inline void load(Archive& ar, sf::WASAPIInterface& data, const unsigned int /*nVersion*/)
		{
			bool exclusiveMode = false;;
			int bufferLatency = 0;
			bool enableMMCSS;
			int MmThreadPriority;
			float volume;
			ar &  exclusiveMode;
			data.setExclusiveMode(exclusiveMode);
			ar & bufferLatency;
			data.setBufferLatency(bufferLatency);
			ar & enableMMCSS;
			data.enableMMCSS(enableMMCSS);
			ar & MmThreadPriority;
			data.setMmThreadPriority((sf::AVRT_PRIORITY)MmThreadPriority);
			ar & volume;
			data.setVolume(volume);
		};

	}
}

BOOST_SERIALIZATION_SPLIT_FREE(sf::WASAPIInterface);
