﻿#pragma once
/*
  ==============================================================================

   This file is part of the S.F.Tracker
   Copyright 2005-7 by Satoshi Fujiwara.

   S.F.Tracker can be redistributed and/or modified under the terms of the
   GNU General Public License, as published by the Free Software Foundation;
   either version 2 of the License, or (at your option) any later version.

   S.F.Tracker is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with S.F.Tracker; if not, visit www.gnu.org/licenses or write to the
   Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
   Boston, MA 02111-1307 USA

  ==============================================================================
*/
#include "aeffectx.h"
#include "MidiMultiOutPlugin.h"
#include "FilterGraph.h"
#define INIT_TEMPO 120.0
#define INIT_NUMBERS 4.0
#define INIT_DIVDENOMINATOR 4.0
#define INIT_NUMERATOR 4.0
#define INIT_TIMEBASE 480.0
#define INIT_SAMPLERATE 44100.0
#define INIT_RIGHTLOCATOR 10000000000.0
#define INIT_TRACK 24



namespace sf
{
	class SequenceFilter : public MidiMultiOutPlugin,juce::Timer
	{
	public:
		SequenceFilter();

		/** Returns the parent graph to which this processor belongs, or 0 if it
            hasn't yet been added to one. */
		juce::FilterGraph* getParentGraph() const throw()         { return graph; }

        /** True if this is an audio or midi input. */
		bool isInput() const throw(){return true;};
        /** True if this is an audio or midi output. */
		bool isOutput() const throw(){return true;};

		const juce::String getName() const {return L"Sequencer";};

		void fillInPluginDescription (juce::PluginDescription& d) const;

		void prepareToPlay (double sampleRate, int estimatedSamplesPerBlock);
		void releaseResources();
//		void processBlock (juce::AudioSampleBuffer& buffer, juce::MidiBuffer& midiMessages);	
		void processBlock (juce::AudioSampleBuffer& buffer, juce::Array <juce::MidiBuffer*>& midiBuffers);

		const juce::String getInputChannelName (const int channelIndex) const {return juce::String::empty;};
		const juce::String getOutputChannelName (const int channelIndex) const{return juce::String::empty;};
		bool isInputChannelStereoPair (int index) const { return false;};
		bool isOutputChannelStereoPair (int index) const { return false;};
		bool acceptsMidi() const {return true;};
		bool producesMidi() const{return true;};

		juce::AudioProcessorEditor* createEditor();

		int getNumParameters(){return 0;};
		const juce::String getParameterName (int){return juce::String::empty;};
		float getParameter (int){return 0.0f;};
		const juce::String getParameterText (int){return juce::String::empty;};
		void setParameter (int, float){};

		int getNumPrograms(){return 0;};
		int getCurrentProgram(){return 0;};
		void setCurrentProgram (int){};
		const juce::String getProgramName (int){return juce::String::empty;};
		void changeProgramName (int, const juce::String&){};

		void getStateInformation (JUCE_NAMESPACE::MemoryBlock& destData);

		void setStateInformation (const void* data, int sizeInBytes);

        /** @internal */
		void setParentGraph (juce::FilterGraph* const graph_) throw() 
		{
			graph = graph_;
		};

		virtual ~SequenceFilter() {};

		//==============================================================================
		void timerCallback (){updateHostDisplay();};

		//==============================================================================
		bool timeSignatureChanged (const double barsCount,
								   const double timeDenominator);


		bool playingPositionChanged (const double absolutePosition);

		bool noteAdded (const int noteNumber,
						const double beatNumber,
						const double noteLength);

		bool noteRemoved (const int noteNumber,
						  const double beatNumber,
						  const double noteLength);

		bool noteMoved (const int oldNote,
						const double oldBeat,
						const int noteNumber,
						const double beatNumber,
						const double noteLength);

		bool noteResized (const int noteNumber,
						  const double beatNumber,
						  const double noteLength);

		bool allNotesRemoved ();

 
 		bool readSequenceDataFromFile(const juce::File& inputFile);

	   //==============================================================================
		/** Return the specified note on

			This is typically used in GUI, when components need to rebuild internal
			note structure (typically a piano grid)
		*/
		void getNoteOnIndexed (const int index, int& note, double& beat, double& length);

		/** Counts the available note on */
		int getNumNoteOn () const;

		//==============================================================================
		/** Serialize internal properties to an Xml element */
		void savePropertiesToXml (juce::XmlElement* element);

		/** Deserialize internal properties from an Xml element */
		void loadPropertiesFromXml (juce::XmlElement* element);

		//==============================================================================
		/** Serialize midi file to a memory block */
		void getChunk (juce::MemoryBlock& mb);

		/** Deserialize midi file from a memory block */
		void setChunk (const juce::MemoryBlock& mb);
		juce_UseDebuggingNewOperator

		// Sequence Control

	    void play ();/// 再生開始

		void stop ();/// 停止

	    void record ();/// 録音

		void rewind ();///　巻き戻し

	    void allNotesOff ();

		void resetRecording ();

		bool isPlaying () const                          { return playing; }
		bool isRecording () const                        { return recording; }
		bool isLooping () const                          { return looping; }

		void setLooping (const bool newValue)            { looping = newValue; }

		bool willStopRecord () const                     { return doStopRecord; }
		bool willSendAllNotesOff () const                { return doAllNotesOff; }
		bool willRewind () const                         { return doRewind; }

		void setTimeSignature (const double bpmTempo,
							   const double barsCount,
							   const double timeDenominator,const double timeBase);

		double getTickCounter() const {return tickCounter;}

		double getNumBars () const                          { return numBars; }
		double getTimeDenominator () const                  { return divDenominator; }

		void setTempo (const double newTempo);
		double getTempo () const                            { return bpmTempo; }

		double getNumerator(){return numerator;}

		void setLeftLocator (const double beatNumber);
		double getLeftLocator () const                    { return leftLocator; }

		void setRightLocator (const double beatNumber);
		double getRightLocator () const                   { return rightLocator; }

		double getFramesPerBeat () const                    { return framesPerBeat; }

		void setPosition (const double newTick)      
		{ 
			const juce::ScopedLock sl (graph->getGraph().getCallbackLock());
			jassert(newTick <= lastTick);

			tickCounter = newTick;
			convertTickToSamplePosition();
			seekMidiEvent();
		}

		double getSamplePosition () const { return samplePositionCounter; }
		double getLastTick() const {return lastTick;}

		double getTimeBase() const { return timeBase;}
		double getSamplesPerTick() const { return samplesPerTick;}

		/**
			TickをSamplePositionに変換する
		*/
		void convertTickToSamplePosition();
		void seekMidiEvent();

		bool isEmptyData(){return !file || file->getNumTracks() == 0;}
		const juce::File& getMIDIFileInfo() const {return readFile;}

		///void loadFromXml (juce::XmlElement* xml);
		///void saveToXml (juce::XmlElement* xml);    

	private:
	    friend class boost::serialization::access;
		template<class Archive>
		void save(Archive & ar, const unsigned int version) const;
		template<class Archive>
		void load(Archive & ar, const unsigned int version);
		
		BOOST_SERIALIZATION_SPLIT_MEMBER();

		void processIncomingMidi (juce::MidiBuffer& midiMessages);

		juce::ScopedPointer<juce::MidiFile> file;
        juce::FilterGraph* graph;
		juce::MidiMessageSequence recordingSequence,tempoSig;
		int currentTrackNo,numOfTracks;
		juce::MidiMessage allNotesOffMessage;
		juce::Array<int> currentSequenceIndexs;
		juce::File readFile;
//		juce::SequenceEditor* activeEditor;
		
		// Sequence Time Control
		double sampleRateDelta;
	    double samplePositionCounter;
//	    double endTick;
		double framesPerBeat;

		double leftLocator;///
		double rightLocator;

		double bpmTempo;/// beat Per miniutes
		double numBars;/// number of bars 
		double divDenominator;///
		double numerator;
		double timeBase;/// SMF Time Base 
		double samplesPerTick;/// Samples / 1 Time Base  
		double tickCounter;/// tick count
		double lastTick;

		// internal state bitflags
		bool playing,        
			 looping,        
			 recording,      
			 doStopRecord,   
			 doRewind,       
			 doAllNotesOff;  

		#if JUCE_PLUGINHOST_VST
			// vst internal timeinfo
		VstTimeInfo timeInfo;
		#endif

		const SequenceFilter& operator= (const SequenceFilter&);

	};
}
