/*
  ==============================================================================

   This file is part of the JUCE library - "Jules' Utility Class Extensions"
   Copyright 2004-10 by Raw Material Software Ltd.

  ------------------------------------------------------------------------------

   JUCE can be redistributed and/or modified under the terms of the GNU General
   Public License (Version 2), as published by the Free Software Foundation.
   A copy of the license is included in the JUCE distribution, or can be found
   online at www.gnu.org/licenses.

   JUCE is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
   A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

  ------------------------------------------------------------------------------

   To release a closed-source product which uses JUCE, commercial licenses are
   available: visit www.rawmaterialsoftware.com/juce for more information.

  ==============================================================================
*/

#ifndef __SF_AUDIOPROCESSORGRAPH_JUCEHEADER__
#define __SF_AUDIOPROCESSORGRAPH_JUCEHEADER__

//#include "juce_AudioProcessor.h"
//#include "../plugins/juce_AudioPluginFormatManager.h"
//#include "../plugins/juce_KnownPluginList.h"
//#include "../../containers/juce_NamedValueSet.h"
//#include "../../containers/juce_ReferenceCountedArray.h"
//==============================================================================
/**
    A type of juce::AudioProcessor which plays back a graph of other AudioProcessors.

    Use one of these objects if you want to wire-up a set of AudioProcessors
    and play back the result.

    Processors can be added to the graph as "nodes" using addNode(), and once
    added, you can connect any of their input or output channels to other
    nodes using addConnection().

    To play back a graph through an audio device, you might want to use an
    AudioProcessorPlayer object.
*/
namespace sf {

class AddConnectionCommand;
class RemoveConnectionCommand;
class RemoveConnectionAtCommand;
class RemoveNodeCommand;
class AddNodeCommand;

class JUCE_API  AudioProcessorGraph   : public juce::AudioProcessor,
                                        public juce::AsyncUpdater
{
public:
	friend class AddConnectionCommand;
	friend class RemoveConnectionCommand;
	friend class RemoveConnectionAtCommand;
	friend class RemoveNodeCommand;
	friend class AddNodeCommand;

	//==============================================================================
    /** Creates an empty graph.
    */
    AudioProcessorGraph();

    /** Destructor.

        Any processor objects that have been added to the graph will also be deleted.
    */
    ~AudioProcessorGraph();

    //==============================================================================
    /** Represents one of the nodes, or processors, in an AudioProcessorGraph.

        To create a node, call AudioProcessorGraph::addNode().
    */
    class JUCE_API  Node   : public juce::ReferenceCountedObject
    {
    public:
        /** Destructor.
        */
        ~Node();

        //==============================================================================
        /** The ID number assigned to this node.

            This is assigned by the graph that owns it, and can't be changed.
        */
        const juce::uint32 id;

        /** The actual processor object that this node represents.
        */
        juce::AudioProcessor* const processor;

        /** A set of user-definable properties that are associated with this node.

            This can be used to attach values to the node for whatever purpose seems
            useful. For example, you might store an x and y position if your application
            is displaying the nodes on-screen.
        */
        juce::NamedValueSet properties;

        //==============================================================================
        /** A convenient typedef for referring to a pointer to a node object.
        */
        typedef juce::ReferenceCountedObjectPtr <Node> Ptr;

        //==============================================================================
        juce_UseDebuggingNewOperator

    private:
        friend class AudioProcessorGraph;

        bool isPrepared;

        Node (juce::uint32 id, juce::AudioProcessor* processor);

        void prepare (double sampleRate, int blockSize, AudioProcessorGraph* graph);
        void unprepare();

        Node (const Node&);
        Node& operator= (const Node&);
    };

    //==============================================================================
    /** Represents a connection between two channels of two nodes in an AudioProcessorGraph.

        To create a connection, use AudioProcessorGraph::addConnection().
    */
    struct JUCE_API  Connection
    {
    public:
        //==============================================================================
        /** The ID number of the node which is the input source for this connection.
            @see AudioProcessorGraph::getNodeForId
        */
        juce::uint32 sourceNodeId;

        /** The index of the output channel of the source node from which this
            connection takes its data.

            If this value is the special number AudioProcessorGraph::midiChannelIndex, then
            it is referring to the source node's midi output. Otherwise, it is the zero-based
            index of an audio output channel in the source node.
        */
        int sourceChannelIndex;

        /** The ID number of the node which is the destination for this connection.
            @see AudioProcessorGraph::getNodeForId
        */
        juce::uint32 destNodeId;

        /** The index of the input channel of the destination node to which this
            connection delivers its data.

            If this value is the special number AudioProcessorGraph::midiChannelIndex, then
            it is referring to the destination node's midi input. Otherwise, it is the zero-based
            index of an audio input channel in the destination node.
        */
        int destChannelIndex;

        //==============================================================================
        juce_UseDebuggingNewOperator

    private:
    };

    //==============================================================================
    /** Deletes all nodes and connections from this graph.

        Any processor objects in the graph will be deleted.
    */
    void clear();

    /** Returns the number of nodes in the graph. */
    int getNumNodes() const                                         { return nodes.size(); }

    /** Returns a pointer to one of the nodes in the graph.

        This will return 0 if the index is out of range.
        @see getNodeForId
    */
    Node* getNode (const int index) const                           { return nodes [index]; }

    /** Searches the graph for a node with the given ID number and returns it.

        If no such node was found, this returns 0.
        @see getNode
    */
    Node* getNodeForId (const juce::uint32 nodeId) const;

    /** Adds a node to the graph.

        This creates a new node in the graph, for the specified processor. Once you have
        added a processor to the graph, the graph owns it and will delete it later when
        it is no longer needed.

        The optional nodeId parameter lets you specify an ID to use for the node, but
        if the value is already in use, this new node will overwrite the old one.

        If this succeeds, it returns a pointer to the newly-created node.
    */
    Node* addNode (juce::AudioProcessor* newProcessor, juce::uint32 nodeId = 0);


    /** Deletes a node within the graph which has the specified ID.

        This will also delete any connections that are attached to this node.
    */
    bool removeNode (juce::uint32 nodeId); // Undoable
    bool removeNode_ (juce::uint32 nodeId);// Not Undoable

    //==============================================================================
    /** Returns the number of connections in the graph. */
    int getNumConnections() const                                       { return connections.size(); }

    /** Returns a pointer to one of the connections in the graph. */
    const Connection* getConnection (int index) const                   { return connections [index]; }

    /** Searches for a connection between some specified channels.

        If no such connection is found, this returns 0.
    */
    const Connection* getConnectionBetween (juce::uint32 sourceNodeId,
                                            int sourceChannelIndex,
                                            juce::uint32 destNodeId,
                                            int destChannelIndex) const;

    /** Returns true if there is a connection between any of the channels of
        two specified nodes.
    */
    bool isConnected (juce::uint32 possibleSourceNodeId,
                      juce::uint32 possibleDestNodeId) const;

    /** Returns true if it would be legal to connect the specified points.
    */
    bool canConnect (juce::uint32 sourceNodeId, int sourceChannelIndex,
                     juce::uint32 destNodeId, int destChannelIndex) const;

    /** Attempts to connect two specified channels of two nodes.

        If this isn't allowed (e.g. because you're trying to connect a midi channel
        to an audio one or other such nonsense), then it'll return false.
    */
    bool addConnection (juce::uint32 sourceNodeId, int sourceChannelIndex,
                        juce::uint32 destNodeId, int destChannelIndex);

    /** Deletes the connection with the specified index.

        Returns true if a connection was actually deleted.
    */
    void removeConnection (int index);

    /** Deletes any connection between two specified points.

        Returns true if a connection was actually deleted.
    */

    bool removeConnection (juce::uint32 sourceNodeId, int sourceChannelIndex,
                           juce::uint32 destNodeId, int destChannelIndex);

    /** Removes all connections from the specified node.
    */
    bool disconnectNode (juce::uint32 nodeId);

    /** Performs a sanity checks of all the connections.

        This might be useful if some of the processors are doing things like changing
        their channel counts, which could render some connections obsolete.
    */
    bool removeIllegalConnections();

    //==============================================================================
    /** A special number that represents the midi channel of a node.

        This is used as a channel index value if you want to refer to the midi input
        or output instead of an audio channel.
    */
    static const int midiChannelIndex;


    //==============================================================================
    /** A special type of juce::AudioProcessor that can live inside an AudioProcessorGraph
        in order to use the audio that comes into and out of the graph itself.

        If you create an AudioGraphIOProcessor in "input" mode, it will act as a
        node in the graph which delivers the audio that is coming into the parent
        graph. This allows you to stream the data to other nodes and process the
        incoming audio.

        Likewise, one of these in "output" mode can be sent data which it will add to
        the sum of data being sent to the graph's output.

        @see AudioProcessorGraph
    */
    class JUCE_API  AudioGraphIOProcessor     : public juce::AudioPluginInstance
    {
    public:
        /** Specifies the mode in which this processor will operate.
        */
        enum IODeviceType
        {
            audioInputNode,     /**< In this mode, the processor has output channels
                                     representing all the audio input channels that are
                                     coming into its parent audio graph. */
            audioOutputNode,    /**< In this mode, the processor has input channels
                                     representing all the audio output channels that are
                                     going out of its parent audio graph. */
            midiInputNode,      /**< In this mode, the processor has a midi output which
                                     delivers the same midi data that is arriving at its
                                     parent graph. */
            midiOutputNode      /**< In this mode, the processor has a midi input and
                                     any data sent to it will be passed out of the parent
                                     graph. */
        };

        //==============================================================================
        /** Returns the mode of this processor. */
        IODeviceType getType() const                                { return type; }

        /** Returns the parent graph to which this processor belongs, or 0 if it
            hasn't yet been added to one. */
        AudioProcessorGraph* getParentGraph() const                 { return graph; }

        /** True if this is an audio or midi input. */
        bool isInput() const;
        /** True if this is an audio or midi output. */
        bool isOutput() const;

        //==============================================================================
        AudioGraphIOProcessor (const IODeviceType type);
        ~AudioGraphIOProcessor();

        const juce::String getName() const;
        void fillInPluginDescription (juce::PluginDescription& d) const;

        void prepareToPlay (double sampleRate, int estimatedSamplesPerBlock);
        void releaseResources();
        void processBlock (juce::AudioSampleBuffer& buffer, juce::MidiBuffer& midiMessages);

        const juce::String getInputChannelName (const int channelIndex) const;
        const juce::String getOutputChannelName (const int channelIndex) const;
        bool isInputChannelStereoPair (int index) const;
        bool isOutputChannelStereoPair (int index) const;
        bool acceptsMidi() const;
        bool producesMidi() const;

        juce::AudioProcessorEditor* createEditor();

        int getNumParameters();
        const juce::String getParameterName (int);
        float getParameter (int);
        const juce::String getParameterText (int);
        void setParameter (int, float);

        int getNumPrograms();
        int getCurrentProgram();
        void setCurrentProgram (int);
        const juce::String getProgramName (int);
        void changeProgramName (int, const juce::String&);

        void getStateInformation (juce::MemoryBlock& destData);
        void setStateInformation (const void* data, int sizeInBytes);

        /** @internal */
        void setParentGraph (AudioProcessorGraph* graph);

        juce_UseDebuggingNewOperator

    private:
        const IODeviceType type;
        AudioProcessorGraph* graph;

        AudioGraphIOProcessor (const AudioGraphIOProcessor&);
        AudioGraphIOProcessor& operator= (const AudioGraphIOProcessor&);
    };

    //==============================================================================
    // juce::AudioProcessor methods:

    const juce::String getName() const;

    void prepareToPlay (double sampleRate, int estimatedSamplesPerBlock);
    void releaseResources();
    void processBlock (juce::AudioSampleBuffer& buffer, juce::MidiBuffer& midiMessages);

    const juce::String getInputChannelName (const int channelIndex) const;
    const juce::String getOutputChannelName (const int channelIndex) const;
    bool isInputChannelStereoPair (int index) const;
    bool isOutputChannelStereoPair (int index) const;

    bool acceptsMidi() const;
    bool producesMidi() const;

    juce::AudioProcessorEditor* createEditor()            { return 0; }

    int getNumParameters()                          { return 0; }
    const juce::String getParameterName (int)             { return juce::String::empty; }
    float getParameter (int)                        { return 0; }
    const juce::String getParameterText (int)             { return juce::String::empty; }
    void setParameter (int, float)                  { }

    int getNumPrograms()                            { return 0; }
    int getCurrentProgram()                         { return 0; }
    void setCurrentProgram (int)                    { }
    const juce::String getProgramName (int)               { return juce::String::empty; }
    void changeProgramName (int, const juce::String&)     { }

    void getStateInformation (juce::MemoryBlock& destData);
    void setStateInformation (const void* data, int sizeInBytes);

    /** @internal */
    void handleAsyncUpdate();

    //==============================================================================
    juce_UseDebuggingNewOperator

private:
	Node* addNode_ (juce::AudioProcessor* newProcessor, juce::uint32 nodeId = 0);
    void removeConnection_ (int index);
    bool addConnection_ (juce::uint32 sourceNodeId, int sourceChannelIndex,
                        juce::uint32 destNodeId, int destChannelIndex);
	bool removeConnection_ (juce::uint32 sourceNodeId, int sourceChannelIndex,
                        juce::uint32 destNodeId, int destChannelIndex);

    juce::ReferenceCountedArray <Node> nodes;
    juce::OwnedArray <Connection> connections;
    int lastNodeId;
    juce::AudioSampleBuffer renderingBuffers;
    juce::OwnedArray <juce::MidiBuffer> midiBuffers;

//    juce::CriticalSection renderLock;
    juce::VoidArray renderingOps;

    friend class AudioGraphIOProcessor;
    juce::AudioSampleBuffer* currentAudioInputBuffer;
    juce::AudioSampleBuffer currentAudioOutputBuffer;
    juce::MidiBuffer* currentMidiInputBuffer;
    juce::MidiBuffer currentMidiOutputBuffer;

    void clearRenderingSequence();
    void buildRenderingSequence();

    bool isAnInputTo (juce::uint32 possibleInputId, juce::uint32 possibleDestinationId, int recursionCheck) const;

    AudioProcessorGraph (const AudioProcessorGraph&);
    AudioProcessorGraph& operator= (const AudioProcessorGraph&);
};



class AddConnectionCommand : public juce::UndoableAction
{
public:
	AddConnectionCommand(AudioProcessorGraph& graph,const juce::uint32 sourceNodeId_,
                                         const int sourceChannelIndex_,
                                         const juce::uint32 destNodeId_,
                                         const int destChannelIndex_) 
										 : targetGraph(graph),
											sourceNodeId(sourceNodeId_),sourceChannelIndex(sourceChannelIndex_),
											destNodeId(destNodeId_),destChannelIndex(destChannelIndex_)
	{
	}
	virtual bool perform();
	virtual bool undo(); 
    virtual int getSizeInUnits()    { return (int)(sizeof(AddConnectionCommand)); }
private:
	AudioProcessorGraph& targetGraph;
	const juce::uint32 sourceNodeId;
    const int sourceChannelIndex;
    const juce::uint32 destNodeId;
    const int destChannelIndex;
};

class RemoveConnectionCommand : public juce::UndoableAction
{
public:
	RemoveConnectionCommand(AudioProcessorGraph& graph,const juce::uint32 sourceNodeId_,
                                         const int sourceChannelIndex_,
                                         const juce::uint32 destNodeId_,
                                         const int destChannelIndex_) 
										 : targetGraph(graph),
											sourceNodeId(sourceNodeId_),sourceChannelIndex(sourceChannelIndex_),
											destNodeId(destNodeId_),destChannelIndex(destChannelIndex_)
	{
	}
	virtual bool perform();
	virtual bool undo(); 
    virtual int getSizeInUnits()    { return (int)(sizeof(RemoveConnectionCommand)); }
private:
	AudioProcessorGraph& targetGraph;
	const juce::uint32 sourceNodeId;
    const int sourceChannelIndex;
    const juce::uint32 destNodeId;
    const int destChannelIndex;
};

class RemoveConnectionAtCommand : public juce::UndoableAction
{
public:
	RemoveConnectionAtCommand(AudioProcessorGraph& graph,const int targetIndex_)
										 : targetGraph(graph),
											targetIndex(targetIndex_)

	{
		const sf::AudioProcessorGraph::Connection* c = targetGraph.getConnection(targetIndex_);
		sourceNodeId = c->sourceNodeId;
		sourceChannelIndex = c->sourceChannelIndex;
		destChannelIndex = c->destChannelIndex;
		destNodeId = c->destNodeId;
	}
	virtual bool perform();
	virtual bool undo(); 
    virtual int getSizeInUnits()    { return (int)(sizeof(RemoveConnectionAtCommand)); }
private:
	AudioProcessorGraph& targetGraph;
    const int targetIndex;
	juce::uint32 sourceNodeId;
    int sourceChannelIndex;
    juce::uint32 destNodeId;
    int destChannelIndex;

};

class RemoveNodeCommand : public juce::UndoableAction
{
public:
	RemoveNodeCommand(AudioProcessorGraph& graph,const juce::uint32 targetId_) : targetGraph(graph),targetId(targetId_){};
	virtual bool perform();
	virtual bool undo(); 
    virtual int getSizeInUnits()    { return (int)(sizeof(RemoveNodeCommand)); }
private:
	AudioProcessorGraph& targetGraph;
	const juce::uint32 targetId;
	int lastNodeId;
	juce::PluginDescription targetPd;
	juce::MemoryBlock targetState;
	juce::NamedValueSet targetProperties;
};

class AddNodeCommand : public juce::UndoableAction
{
public:
	AddNodeCommand(AudioProcessorGraph& target,juce::AudioProcessor* newProcessor_, juce::uint32 nodeId_ = 0) 
		: targetGraph(target),newProcessor(newProcessor_),addedNode(0),nodeId(nodeId_) {};

	virtual bool perform();
	virtual bool undo(); 
	virtual int getSizeInUnits()    { return (int)(sizeof(AddNodeCommand)); }
private:
	int lastNodeId;
	const juce::uint32 nodeId;
	AudioProcessorGraph& targetGraph;
	juce::AudioProcessor* newProcessor;
	sf::AudioProcessorGraph::Node* addedNode;
	
};

} // namespace sf

// serialize function //
namespace boost {
namespace serialization {

	template <class Archive>
		void serialize(Archive& ar, sf::AudioProcessorGraph::Connection& con, const unsigned int version)
		{
			ar & make_nvp("sourceChannelIndex",con.sourceChannelIndex);
			ar & make_nvp("sourceNodeId",con.sourceNodeId);
			ar & make_nvp("destChannelIndex",con.destChannelIndex);
			ar & make_nvp("destChannelId",con.destChannelId);
		}

//	template <class Archive>
//		void save(Archive& ar,const sf::AudioProcessorGraph::Node& node, const unsigned int version)
//		{
//			juce::AudioPluginInstance* plugin = dynamic_cast <AudioPluginInstance*> (node.processor);
//			juce::PluginDescription pd;
//
//			plugin->fillInPluginDescription (pd);
//			ar & make_nvp("PluginDescription",pd);
//			ar & make_nvp("id",node.id);
////			ar & make_nvp("x",(double)(node.properties ["x"]));
////			ar & make_nvp("y",(double)(node.properties ["y"]));
//			juce::MemoryBlock m;
//			node.processor->getStateInformation (m);
//			ar & make_nvp("State",m);
//
//		}
//
//	template <class Archive>
//		void load(Archive& ar,sf::AudioProcessorGraph::Node& node, const unsigned int version)
//		{
//		    juce::PluginDescription pd;
//			ar & make_nvp("PluginDescription",pd);
//			juce::String errorMessage;
//		    juce::AudioPluginInstance* instance
//			= juce::AudioPluginFormatManager::getInstance()->createPluginInstance (pd, errorMessage);
//			node.processor = instance;
//			ar & make_nvp("id",node.id);
//
//
//
//		}

}
}

//BOOST_SERIALIZATION_SPLIT_FREE(sf::AudioProcessorGraph::Node);


#endif   // __SF_AUDIOPROCESSORGRAPH_JUCEHEADER__
