/*
  sted_screen.h
  base class of screen driver

  Made by Studio Breeze. 2002

  Permission is hereby granted, free of charge, to any person obtaining a copy
  of this software and associated documentation files (the "Software"), to deal
  in the Software without restriction, including without limitation the rights
  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software, and to permit persons to whom the Software is
  furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in
  all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
  THE SOFTWARE.
 */

#ifndef STED_SCREEN_H
#define STED_SCREEN_H

#include <stdio.h>

#ifndef __dosmode_t_defined__
#define __dosmode_t_defined__
typedef int dosmode_t;
#endif

enum {
  EButtonLeft = 0,
  EButtonRight
};

enum {
  EButtonOff = 0,
  EButtonOn
};

class CSTedScreenBase {
 protected:
  CSTedScreenBase();
  void Construct(void);

 public:
  static CSTedScreenBase* New(void);
  virtual ~CSTedScreenBase();
  virtual void Tidy(void);

  // window
  virtual void InitWindow(void);
  virtual void CloseWindow(void);
  virtual void RepaintWindow(void);
  virtual void UpdateWindow(void);
  virtual void DisableUpdateWindow(void);

  // color
  virtual int SetGraphicsPalette(int in_pal, int in_color);
  virtual int SetTextPalette(int in_pal, int in_color);
  virtual int SetGraphicsColor(int in_color);
  virtual int SetTextColor(int in_color);
  
  // cursor
  virtual void SetCursor(int in_onoff);

  // text
  virtual void SetTextWindow(int in_y, int in_h);
  virtual void Locate(int in_x, int in_y);
  virtual void Puts(const char* in_str);
  virtual void ClsEol(void);
  virtual void ClsEd(void);
  virtual void GraphicsPuts(int in_x, int in_y, const char* in_str);

  // graphics
  virtual void TextXBox(int in_x0, int in_y0, int in_x1, int in_y1, int in_page);
  virtual void TextXXLine(int in_page, int in_x0, int in_y, int in_x1, int in_linestyle);
  virtual void TextXYLine(int in_page, int in_x, int in_y0, int in_y1, int in_linestyle);
  virtual void TextReverse(int in_x, int in_y, int in_width, int in_page);
  virtual void TextReverseArea(int in_r_ad, int r_ln, int in_edit_scr);
  virtual void TextFill(int in_page, int in_x, int in_y, int in_w, int in_h, int in_linestyle);
  virtual void TextRasterCopy(int in_dst, int in_src, int in_line, int in_mode);
  virtual void TextScroll(int in_x0, int in_y0, int in_xs, int in_ys, int in_x1, int in_y1);

  virtual void GraphicsBox(int in_x0, int in_y0, int in_x1, int in_y1, unsigned int in_color, unsigned int in_linestyle);
  virtual int GraphicsPoint(int in_x, int in_y);
  virtual void GraphicsLine(int in_x0, int in_y0, int in_x1, int in_y1, int in_color, int in_linestyle);
  virtual void GraphicsFill(int in_x0, int in_y0, int in_x1, int in_y1, int in_color);

  virtual void CopyTextToGraphics(int in_editscreen);
  virtual void CopyTextToGraphics2(int in_editscreen);

  virtual void GraphicsHome(int in_currenthome);
  virtual void ClsAll(void);
  virtual void GraphicsClear(void);

  // key
  virtual int KeyInit(void);
  virtual int KeyIn(int in_code);
  virtual int KeyInp(void);
  virtual int SftSense(void);
  virtual int BitSense(int in_group);
  virtual int KeySense(void);
  virtual void KeyWait(void);
  virtual void LedMode(int in_code, int in_onoff);
  virtual void ClearKeyBuffer(void);
  void SetFncKey(const char in_fnckey[12][6]);
  void SetX68ScanCode(const int* in_codes);

  // pointer
  virtual void MouseInit(void);
  virtual void SetMouseCursorVisible(int in_onoff);
  virtual int GetMouseButtonState(void);
  virtual int SetMouseMoveLimit(int in_x0, int in_y0, int in_x1, int in_y1);
  virtual int GetMousePos(int *out_x, int *out_y);
  virtual void MouseWait(void);

  // midi devices
  virtual int OpenMidiDevice(void);
  virtual void CloseMidiDevice(void);
  virtual int GetMidiData(void);
  virtual int CanMidiRead(void);
  virtual void PutMidiData(int in_data);
  virtual void MidiWait(void);
  virtual void MidiPlayerWait(void);

  // midi players
  virtual int StartMidiPlayer(unsigned char* in_data, int in_len);
  virtual int StopMidiPlayer(void);
  virtual int CloseMidiPlayer(int in_sig);

  // file related
  virtual dosmode_t FileChmod(const char* in_name, dosmode_t in_atr);
  virtual int FileChdir(const char* in_name);
  virtual int FileCurdir(int in_drive, char* out_buff);
  virtual int FileCurdrv(void);
  virtual int FileDelete(const char* in_name);
  virtual int FileRename(const char* in_name, const char* in_newname);
  virtual int FileFiles(struct _x68_filbuf *in_buff, const char *in_name, int in_atr);
  virtual int FileNfiles(struct _x68_filbuf *in_buff);
  virtual FILE* FileFopen(const char* in_name, const char* in_mode);

  // misc functions
  virtual int OnTime(void);
  virtual int USleep(long in_usec);

 protected:
  bool fConstructed;
  int fFncKey[12][6];
  const int *fX68ScanCode;
};

/* accessor */
extern CSTedScreenBase* CreateScreenDriver(int in_size);
extern void DestroyScreenDriver(CSTedScreenBase* in_driver);

extern CSTedScreenBase* GetDriver(void);

#endif /* STED_SCREEN_H */
