/*
  MIDI device handling codes

  Copyright 1999 by Daisuke Nagano <breeze.nagano@nifty.ne.jp>
  Dec.18.1999
  Oct.16.2002


  Permission is hereby granted, free of charge, to any person obtaining a copy
  of this software and associated documentation files (the "Software"), to deal
  in the Software without restriction, including without limitation the rights
  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software, and to permit persons to whom the Software is
  furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in
  all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
  THE SOFTWARE.
*/

/* ------------------------------------------------------------------- */

#ifdef HAVE_CONFIG_H
#include "../../config.h"
#endif /* HAVE_CONFIG_H */


#include "../drivers/sted_screen.h"

extern "C" {

#include "rcp.h"
#include "mididev.h"

extern int reset_mode;
extern int sted_sleep(long in_usec);

/* MIDI reset messages */

static const unsigned char gs_reset[] = {
  0xf0, 0x41, 0x10, 0x42, 0x12, 0x40, 0x00, 
  0x7f, 0x00, 0x41, 0xf7,
  0xff }; /* GS Reset */

static const unsigned char gm_reset[] = {
  0xf0, 0x7e, 0x7f, 0x09, 0x01, 0xf7,
  0xff };

static const unsigned char S8_reset[] = {
  0xf0, 0x41, 0x10, 0x42, 0x12, 0x00, 0x00,
  0x7f, 0x00, 0x01, 0xf7,
  0xff }; /* Single module mode */

static const unsigned char XG_reset[] = {
  0xf0, 0x43, 0x10, 0x4c, 0x00, 0x00,
  0x7e, 0x00, 0xf7,
  0xff };

static const unsigned char all_reset[] = {
  0xb0, 0x7b, 0x00, 0xb0, 0x78, 0x00, 0xb0, 0x79, 0x00,
  0xb1, 0x7b, 0x00, 0xb1, 0x78, 0x00, 0xb1, 0x79, 0x00,
  0xb2, 0x7b, 0x00, 0xb2, 0x78, 0x00, 0xb2, 0x79, 0x00,
  0xb3, 0x7b, 0x00, 0xb3, 0x78, 0x00, 0xb3, 0x79, 0x00,
  0xb4, 0x7b, 0x00, 0xb4, 0x78, 0x00, 0xb4, 0x79, 0x00,
  0xb5, 0x7b, 0x00, 0xb5, 0x78, 0x00, 0xb5, 0x79, 0x00,
  0xb6, 0x7b, 0x00, 0xb6, 0x78, 0x00, 0xb6, 0x79, 0x00,
  0xb7, 0x7b, 0x00, 0xb7, 0x78, 0x00, 0xb7, 0x79, 0x00,
  0xb8, 0x7b, 0x00, 0xb8, 0x78, 0x00, 0xb8, 0x79, 0x00,
  0xb9, 0x7b, 0x00, 0xb9, 0x78, 0x00, 0xb9, 0x79, 0x00,
  0xba, 0x7b, 0x00, 0xba, 0x78, 0x00, 0xba, 0x79, 0x00,
  0xbb, 0x7b, 0x00, 0xbb, 0x78, 0x00, 0xbb, 0x79, 0x00,
  0xbc, 0x7b, 0x00, 0xbc, 0x78, 0x00, 0xbc, 0x79, 0x00,
  0xbd, 0x7b, 0x00, 0xbd, 0x78, 0x00, 0xbd, 0x79, 0x00,
  0xbe, 0x7b, 0x00, 0xbe, 0x78, 0x00, 0xbe, 0x79, 0x00,
  0xbf, 0x7b, 0x00, 0xbf, 0x78, 0x00, 0xbf, 0x79, 0x00,
  0xff };

static const unsigned char *reset_exclusives[]={
  gm_reset, gs_reset, S8_reset, XG_reset
};

/* ------------------------------------------------------------------- */

static int           block_write = FLAG_TRUE;
static unsigned char midibuf[1024];
static int           midibuf_ptr = 0;

/* ------------------------------------------------------------------- */

int
sayleen_put_midi( int data ) {

  if ( data < 0 || data > 255 ) { return 1; }

  if ( midibuf_ptr >= 1023 ) {
    sayleen_flush_midi();
  }
  midibuf[midibuf_ptr++] = (unsigned char)data;

  return 0;
}

int
sayleen_flush_midi( void ) {
	int i;
	CSTedScreenBase* scr = GetDriver();
	if (!scr) return -1;

	if ( midibuf_ptr > 0 ) {
		for (i=0; i<midibuf_ptr; i++) {
			scr->PutMidiData(midibuf[i]);
		}
		midibuf_ptr = 0;
	}

	return 0;
}

int
sayleen_change_midi_port( int port ) {

	sayleen_put_midi( 0xf5 );
	sayleen_put_midi( port+1 );

	return 0;
}

int
sayleen_send_midi_reset( void ) {

	unsigned char *p;
	int port;

	if ( reset_mode >= sizeof(reset_exclusives) ) {
		return 1;
	}

	sayleen_flush_midi();
	for ( port=0 ; port<1 ; port++ ) {

		/* reset all notes */
		p = (unsigned char *)all_reset;
		while ( *p != 0xff ) {
		sayleen_put_midi( (int)*p++ );
		}
		sayleen_flush_midi();
		sayleen_myusleep(60000);

		/* send reset message */
		p = (unsigned char *)reset_exclusives[reset_mode];
		while ( *p != 0xff ) {
			sayleen_put_midi( (int)*p++ );
		}
		sayleen_flush_midi();
		sayleen_myusleep(60000);
	}

	return 0;
}

/* ------------------------------------------------------------------- */

int 
sayleen_send_rtm_start( void ) {
	sayleen_put_midi( 0xfa );
	return 0;
}

int 
sayleen_send_rtm_stop( void ) {
	sayleen_put_midi( 0xfc );
	return 0;
}

int
sayleen_send_rtm_continue( void ) {
	sayleen_put_midi( 0xfb );
	return 0;
}

int 
sayleen_send_rtm_timingclock( void ) {
	sayleen_put_midi( 0xf8 );
	return 0;
}

void
sayleen_myusleep( long usec ) {
	sted_sleep(usec);
}

}
