/*
  plm : tiny MIDI de-multiplex-a daemon

    usage: % plm <midi_device> <in_device> <out_device>

  plm receives any midi-event from <midi_device> and through it to
  (in_device).
  plm also receives any midi-event from (out_device) and through it to
  <midi_device>.
  So you can specify same <midi_device> to a midi-player and a midi-recorder.

  It doesn't require any root permission to perform.

  Copyright 2000 by Daisuke Nagano <breeze.nagano@nifty.ne.jp>
  Jan.28.1999

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2, or (at your option)
  any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#define HAVE_SERIAL_SUPPORT 1

#include <stdio.h>
#include <stdlib.h>

#include <signal.h>
#include <termios.h>

#include <string.h>

#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/stat.h>

#ifdef _POSIX_PRIORITY_SCHEDULING
# include <sched.h>
#endif
#ifdef _POSIX_MEMLOCK
# include <sys/mman.h>
#endif

#ifdef HAVE_GETTEXT_SUPPORT
#include "gettext_wrapper.h"
#else
#define _(string) string
#endif

static char *IN_FIFO;
static char *OUT_FIFO;

#ifdef HAVE_SERIAL_SUPPORT
#define SERIAL_SPEED  B38400
#endif

static int midi_dev=-1, in_fifo=-1, out_fifo=-1;

static void all_close( void );
static void set_signals( void );
static void priority_init( void );

/* main */

int main( int argc, char **argv ) {

  int sel;
  struct stat stt;
#ifdef HAVE_SERIAL_SUPPORT
  struct termios tio;
#endif

  if ( argc != 4 ) {
    fprintf(stderr,_("usage: %s <midi device> <in_device> <out_device>\n"), argv[0]);
    exit(1);
  }

  IN_FIFO = strdup(argv[2]);
  if ( IN_FIFO == NULL ) exit(1);
  OUT_FIFO = strdup(argv[3]);
  if ( OUT_FIFO == NULL ) exit(1);

  /* midi device */

  if ( stat(argv[1], &stt ) !=0 ) {
    fprintf(stderr,_("%s: midi device %s not available\n"), argv[0], argv[1]);
    exit(1);
  }
  if ( ! S_ISCHR(stt.st_mode ) {
    fprintf(stderr,_("%s: midi device %s not available\n"), argv[0], argv[1]);
    exit(1);
  }

  midi_dev = open( argv[1], O_RDWR|O_NONBLOCK );
  if ( midi_dev < 0 ) {
    fprintf(stderr,_("%s: midi device %s not available\n"), argv[0], argv[1]);
    exit(1);
  }
#ifdef HAVE_SERIAL_SUPPORT
  if ( tcgetattr(midi_dev, &tio) >= 0 ) {  /* It will be a serial device */

    tio.c_iflag = 0;
    tio.c_oflag = 0;
    tio.c_cflag = CS8 | CREAD | CLOCAL;
    tio.c_lflag = 0;
    cfsetispeed(&tio, SERIAL_SPEED);
    cfsetospeed(&tio, SERIAL_SPEED);
    tcsetattr(midi_dev, TCSANOW, &tio);
  }
#endif

  /* input fifo */

  if ( mkfifo( IN_FIFO, 0666 ) != 0 ) {
    fprintf(stderr,_("%s: Cannot create in-fifo %s\n"), argv[0], IN_FIFO);
    all_close();
    exit(1);
  }
  in_fifo = open( IN_FIFO, O_RDWR|O_NONBLOCK );
  if ( in_fifo < 0 ) {
    fprintf(stderr,_("%s: input-fifo %s is not available\n"), argv[0], IN_FIFO);
    all_close();
    exit(1);
  }

  /* output fifo */

  if ( mkfifo( OUT_FIFO, 0666 ) != 0 ) {
    fprintf(stderr,_("%s: Cannot create out-fifo %s\n"), argv[0], OUT_FIFO);
    all_close();
    exit(1);
  }
  out_fifo = open( OUT_FIFO, O_RDWR|O_NONBLOCK );
  if ( out_fifo < 0 ) {
    fprintf(stderr,_("%s: output-fifo %s is not available\n"), argv[0], OUT_FIFO);
    all_close();
    exit(1);
  }

  set_signals();
  priority_init();

  /* main loop */

  sel = midi_dev > in_fifo ? midi_dev+1 : in_fifo+1;

  fprintf(stderr, _("physical device:\t%s\n"), argv[1] );
  fprintf(stderr, _("pseudo MIDI-in:\t%s\n"), IN_FIFO);
  fprintf(stderr, _("pseudo MIDI-out:\t%s\n"), OUT_FIFO);
  fprintf(stderr, "\n");

  while(1) {
    fd_set fd;
    struct timeval tv;
    char buf[16];

    FD_ZERO(&fd);
    FD_SET(in_fifo, &fd);
    FD_SET(midi_dev, &fd);
    tv.tv_usec = 10;
    tv.tv_sec = 0;
    select( sel, &fd, NULL, NULL, &tv );

    /* checks for in-fifo */
    if ( FD_ISSET( in_fifo, &fd ) ) {
      read( in_fifo, buf, 1 );
      write( midi_dev, buf, 1 );
    }

    /* checks for MIDI_DEV */
    if ( FD_ISSET( midi_dev, &fd ) ) {
      read( midi_dev, buf, 1 );
      write( out_fifo, buf, 1 );
    }
  }

  all_close();

#ifdef _POSIX_PRIORITY_SCHEDULING
  sched_yield();
#endif
#ifdef _POSIX_MEMLOCK
  munlockall();
#endif

  exit(0);
}

static void all_close(void) {

  struct stat b;

  if ( midi_dev >= 0 ) close( midi_dev );
  if ( in_fifo >= 0 ) close( in_fifo );
  if ( out_fifo >= 0 ) close( out_fifo );
  midi_dev = in_fifo = out_fifo = -1;

  if ( stat( IN_FIFO, &b ) == 0 ) unlink( IN_FIFO );
  if ( stat( OUT_FIFO, &b ) == 0 ) unlink( OUT_FIFO );

  return;
}

/* */

static void sigexit( int num )
{
  all_close();

#ifdef _POSIX_PRIORITY_SCHEDULING
  sched_yield();
#endif
#ifdef _POSIX_MEMLOCK
  munlockall();
#endif

  fprintf(stderr,_("Signal caught : %d\n"), num);
  exit(1);
}

static const int signals[]={SIGHUP,SIGQUIT,SIGILL,SIGABRT,SIGFPE,
                              SIGBUS,SIGSEGV,SIGPIPE,SIGTERM,SIGINT, 0};

static void set_signals( void ) {

  int i;

  for ( i=0 ; signals[i]!=0 ; i++ )
    signal( signals[i], sigexit );

  return;
}

static void priority_init( void ) {

#ifdef _POSIX_PRIORITY_SCHEDULING
  struct sched_param ptmp, *priority_param;
  int i;

  priority_param=&ptmp;
  i=sched_get_priority_max( SCHED_FIFO );
  priority_param->sched_priority = i/2; /* no means */
  sched_setscheduler( 0, SCHED_FIFO, priority_param );
#endif
#ifdef _POSIX_MEMLOCK
  mlockall(MCL_CURRENT);
#endif

  return;
}
