package jp.xet.eclipse.activitytracker.views;

import jp.xet.eclipse.activitytracker.Activator;
import jp.xet.eclipse.activitytracker.Mode;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.part.ViewPart;

/**
 * Shows log of user activity.
 */
public class UserActivityView extends ViewPart {
	
	private Mode mode = Mode.LETTER;
	
	/**
	 * Shows the user activity log.
	 */
	private Text logText;
	
	/**
	 * The action to turn on/off user activity logging.
	 */
	private IAction toggleLoggingAction;
	
	private IAction toggleMouseAction;
	
	/**
	 * The action to clear activity log.
	 */
	private IAction clearLogAction;
	
	/**
	 * Indicator that user activity logging is turned on.
	 */
	private boolean logging;
	

	/**
	 * Stop logging user activity.
	 */
	public void clearLog() {
		logText.setText("");
	}
	
	@Override
	public void createPartControl(Composite parent) {
		logText = new Text(parent, SWT.MULTI | SWT.READ_ONLY | SWT.H_SCROLL | SWT.V_SCROLL);
		logText.setFont(new Font(parent.getDisplay(), "Courier New", 12, SWT.NORMAL));
		makeActions();
		contributeToActionBars();
	}
	
	@Override
	public void dispose() {
		stopLogging();
		super.dispose();
	}
	
	public void printControl(String text) {
		if (mode == Mode.LETTER) {
			logText.append("\n");
		}
		logText.append(text);
		logText.append("\n");
		mode = Mode.CONTROL;
	}
	
	public void printLetter(String text) {
		logText.append(text);
		mode = Mode.LETTER;
	}
	
	@Override
	public void setFocus() {
		logText.setFocus();
	}
	
	/**
	 * Start logging user activity.
	 */
	public void startLogging() {
		logging = true;
		Activator.getDefault().getListener().setOutput(this);
	}
	
	/**
	 * Start logging user activity.
	 */
	public void startTrackMouse() {
		Activator.getDefault().getListener().setTrackMouse(true);
	}
	
	/**
	 * Stop logging user activity.
	 */
	public void stopLogging() {
		if (logging) {
			Activator.getDefault().getListener().setOutput(null);
			logging = false;
		}
	}
	
	/**
	 * Stop logging user activity.
	 */
	public void stopTrackMouse() {
		Activator.getDefault().getListener().setTrackMouse(false);
	}
	
	/**
	 * Contributes to the view action bars.
	 */
	private void contributeToActionBars() {
		fillLocalToolBar(getViewSite().getActionBars().getToolBarManager());
	}
	
	/**
	 * Creates {@link #clearLogAction}.
	 */
	private void createCleaLogAction() {
		clearLogAction = new Action("CLEAR TEXT", IAction.AS_PUSH_BUTTON) {
			
			@Override
			public void run() {
				clearLog();
			}
		};
		clearLogAction.setText("Clear");
		clearLogAction.setToolTipText("Clear User Activity Log");
	}
	
	/**
	 * Creates {@link #toggleLoggingAction}.
	 */
	private void createToggleLoggingAction() {
		toggleLoggingAction = new Action("RUN TEXT", IAction.AS_CHECK_BOX) {
			
			@Override
			public void run() {
				if (isChecked()) {
					startLogging();
				} else {
					stopLogging();
				}
			}
		};
		toggleLoggingAction.setText("Log");
		toggleLoggingAction.setToolTipText("Start/Stop User Activity Logging");
	}
	
	/**
	 * Creates {@link #toggleLoggingAction}.
	 */
	private void createToggleMouseAction() {
		toggleMouseAction = new Action("TRACK MOUSE", IAction.AS_CHECK_BOX) {
			
			@Override
			public void run() {
				if (isChecked()) {
					startTrackMouse();
				} else {
					stopTrackMouse();
				}
			}
		};
		toggleMouseAction.setText("Mouse");
		toggleMouseAction.setToolTipText("Start/Stop Tracking Mouse");
	}
	
	/**
	 * Initializes local toolbar of the view.
	 * @param manager the toolbar manager. Assumed not <code>null</code>.
	 */
	private void fillLocalToolBar(IToolBarManager manager) {
		manager.add(toggleLoggingAction);
		manager.add(toggleMouseAction);
		manager.add(clearLogAction);
	}
	
	/**
	 * Creates view actions.
	 */
	private void makeActions() {
		createToggleLoggingAction();
		createToggleMouseAction();
		createCleaLogAction();
	}
}
