package jp.xet.eclipse.activitytracker.views;

import jp.xet.eclipse.activitytracker.Activator;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.part.ViewPart;

/**
 * Shows log of user activity.
 *
 * @author Andriy Palamarchuk
 */
public class UserActivityView extends ViewPart {
	
	/**
	 * Shows the user activity log.
	 */
	private Text logText;
	
	/**
	 * The action to turn on/off user activity logging.
	 */
	private IAction toggleLoggingAction;
	
	/**
	 * The action to clear activity log.
	 */
	private IAction clearLogAction;
	
	/**
	 * Indicator that user activity logging is turned on.
	 */
	private boolean logging;
	

	/**
	 * Stop logging user activity.
	 */
	public void clearLog() {
		logText.setText("");
	}
	
	@Override
	public void createPartControl(Composite parent) {
		logText = new Text(parent, SWT.MULTI | SWT.READ_ONLY | SWT.H_SCROLL | SWT.V_SCROLL);
		logText.setFont(new Font(parent.getDisplay(), "Courier New", 12, SWT.NORMAL));
		makeActions();
		contributeToActionBars();
	}
	
	@Override
	public void dispose() {
		stopLogging();
		super.dispose();
	}
	
	/**
	 * Adds the provided text to the view. 
	 * @param text the text to add to the view. Not null.
	 */
	public void print(String text) {
		logText.append(text);
	}
	
	/**
	 * Adds the provided text with an additional new line to the view. 
	 * @param text the text to add to the view. Not null.
	 */
	public void println(String text) {
		logText.append(text);
		logText.append("\n");
	}
	
	@Override
	public void setFocus() {
		logText.setFocus();
	}
	
	/**
	 * Start logging user activity.
	 */
	public void startLogging() {
		logging = true;
		Activator.getDefault().getListener().setOutput(this);
	}
	
	/**
	 * Stop logging user activity.
	 */
	public void stopLogging() {
		if (logging) {
			Activator.getDefault().getListener().setOutput(null);
			logging = false;
		}
	}
	
	/**
	 * Contributes to the view action bars.
	 */
	private void contributeToActionBars() {
		fillLocalToolBar(getViewSite().getActionBars().getToolBarManager());
	}
	
	/**
	 * Creates {@link #clearLogAction}.
	 */
	private void createCleaLogAction() {
		clearLogAction = new Action("CLEAR TEXT", IAction.AS_PUSH_BUTTON) {
			
			@Override
			public void run() {
				clearLog();
			}
		};
		clearLogAction.setText("Clear");
		clearLogAction.setToolTipText("Clear User Activity Log");
	}
	
	/**
	 * Creates {@link #toggleLoggingAction}.
	 */
	private void createToggleLoggingAction() {
		toggleLoggingAction = new Action("RUN TEXT", IAction.AS_CHECK_BOX) {
			
			@Override
			public void run() {
				if (isChecked()) {
					startLogging();
				} else {
					stopLogging();
				}
			}
		};
		toggleLoggingAction.setText("Log");
		toggleLoggingAction.setToolTipText("Start/Stop User Activity Logging");
	}
	
	/**
	 * Initializes local toolbar of the view.
	 * @param manager the toolbar manager. Assumed not <code>null</code>.
	 */
	private void fillLocalToolBar(IToolBarManager manager) {
		manager.add(toggleLoggingAction);
		manager.add(clearLogAction);
	}
	
	/**
	 * Creates view actions.
	 */
	private void makeActions() {
		createToggleLoggingAction();
		createCleaLogAction();
	}
}
