package sdl4gcj.video;

import sdl4gcj.SDLException;

import java.io.File;
import java.io.InputStream;
import java.io.IOException;

public class Surface implements VideoConstants
{
	// instance variables
	protected gnu.gcj.RawData implementation = null;
	protected PixelFormat format = null;
	protected byte[] pixelArray = null;
	private Object sourceArray = null;

	// class methods
	/**
	* Obtain the name of the video driver.
	* SDL_VideoDriverName() wrapper.
	**/
	public static native String getVideoDriverName();

	/**
	* Load a Windows BMP file into an SDL_Surface.
	* SDL_LoadBMP(const char* filePath) wrapper.
	**/
	public static native Surface loadBMP(String filePath);

	/**
	* Load a Windows BMP file into an SDL_Surface.
	* SDL_LoadBMP(const char* filePath) wrapper.
	**/
	public static Surface loadBMP(File file)
	{
		return loadBMP(file.getAbsolutePath());
	}

	/**
	* Load a Windows BMP from byte array into an SDL_Surface.
	* SDL_LoadBMP_RW() wrapper.
	**/
	public native static Surface loadBMP(byte[] bytes, int offset, int length);

	/**
	* Load a Windows BMP from byte array into an SDL_Surface.
	* This is same as loadBMP(bytes, 0, bytes.length);
	* SDL_LoadBMP_RW() wrapper.
	**/
	public static Surface loadBMP(byte[] bytes)
	{
		return loadBMP(bytes, 0, bytes.length);
	}

	/**
	* Load a Windows BMP from inputStream array into an SDL_Surface.
	* SDL_LoadBMP_RW() wrapper.
	**/
	public static Surface loadBMP(InputStream inputStream)
	{
		Surface result = null;
		byte[] bytes = new byte[4096];
		int index = 0;
		int data = -1;
		
		try
		{
			while ((data = inputStream.read()) != -1)
			{
				if (index >= bytes.length)
					bytes = new byte[bytes.length*2];
				bytes[index++] = (byte)data;
			}
			result = loadBMP(bytes, 0, index);
		}
		catch (IOException ioe)
		{
			ioe.printStackTrace();
		}

		return result;
	}

	/**
	* Create an empty SDL_Surface.
	* SDL_CreateRGBSurface() wrapper.
	**/
	public static Surface createRGBSurface(int flags, 
		int width, int height, int bpp,
		int rmask, int gmask, int bmask, int amask)
	{
		Surface surface = new Surface();
		surface.initRGBSurface(flags,
			width, height, bpp, 
			rmask, gmask, bmask, amask);
		return surface;
	}

	public static Surface createRGBSurfaceFrom(byte[] pixels,
		int width, int height, int bpp, int pitch,
		int rmask, int gmask, int bmask, int amask)
	{
		Surface surface = new Surface();
		surface.initRGBSurfaceFrom(pixels,
			width, height, bpp, pitch,
			rmask, gmask, bmask, amask);
		return surface;
	}

	public static Surface createRGBSurfaceFrom(short[] pixels,
		int width, int height, int bpp, int pitch,
		int rmask, int gmask, int bmask, int amask)
	{
		Surface surface = new Surface();
		surface.initRGBSurfaceFrom(pixels,
			width, height, bpp, pitch,
			rmask, gmask, bmask, amask);
		return surface;
	}

	public static Surface createRGBSurfaceFrom(int[] pixels,
		int width, int height, int bpp, int pitch,
		int rmask, int gmask, int bmask, int amask)
	{
		Surface surface = new Surface();
		surface.initRGBSurfaceFrom(pixels,
			width, height, bpp, pitch,
			rmask, gmask, bmask, amask);
		return surface;
	}

	// constructors
	protected Surface() 
	{ 
	}


	// instance methods
	/**
	* gets video flags.
	**/
	public native int getFlags();

	/**
	* gets width.
	**/
	public native int getW();

	/**
	* gets height.
	**/
	public native int getH();

	/**
	* gets byte per line.
	**/
	public native int getPitch();

	/**
	* Save an SDL_Surface as a Windows BMP file.
	* SDL_SaveBMP() wrapper.
	**/
	public native void saveBMP(String filePath);

	/**
	* Save an SDL_Surface as a Windows BMP file.
	* SDL_SaveBMP() wrapper.
	**/
	public void saveBMP(File file)
	{
		this.saveBMP(file.getAbsolutePath());
	}

	/**
	* This performs a fast blit from the surface to 
	* the reciever surface.
	* SDL_BlitSurface() wrapper.
	* @param surface source surface
	* @param sx source rectangle's x
	* @param sy source rectangle's y
	* @param sw source rectangle's width
	* @param sh source rectangle's height
	* @param dx destination rectangle's x
	* @param dy destination rectangle's y
	**/
	public native int blitSurface(Surface surface, 
		int sx, int sy, int sw, int sh, int dx, int dy);

	/**
	* This performs a fast blit from the surface to 
	* the reciever surface.
	* The entire surface is copied.
	* SDL_BlitSurface() wrapper.
	* @param surface source surface
	* @param dx destination rectangle's x
	* @param dy destination rectangle's y
	**/
	public native int blitSurface(Surface surface, int dx, int dy);

	/**
	* This performs a fast blit from the source
	* surface to the destination surface.
	* The entire surface is copied.
	* SDL_BlitSurface() wrapper.
	* @param surface source surface
	**/
	public native int blitSurface(Surface surface);

	/**
	* Repeat surface blitting.
	* SDL_BlitSurface() wrapper.
	**/
	public int repeatSurface(Surface surface)
	{
		int targetW = this.getW();
		int targetH = this.getH();
		int srcW = surface.getW();
		int srcH = surface.getH();
		for (int y = 0; y <= targetH; y += srcH)
		{
			for (int x = 0;x <= targetW; x += srcW)
			{
				int result = this.blitSurface(surface, x, y);
				if (result != 0) return result;
			}
		}
		return 0;
	}

	/**
	* This  function  performs a fast fill of the given 
	* rectangle with some color.
	* SDL_FillRect() wrapper.
	**/
	public native void fillRect(
		int sx, int sy, int sw, int sh, int color);

	/**
	* This  function  performs a fast fill of the given 
	* rectangle with some color.
	* SDL_FillRect() wrapper.
	**/
	public void fillRect(Rect rect, int color)
	{
		this.fillRect(rect.getX(), rect.getY(),
			rect.getW(), rect.getH(), color);
	}

	/**
	* This  function  performs a fast fill of the entire
	* suface with some color.
	* SDL_FillRect() wrapper.
	**/
	public native void fillRect(int color);

	/**
	* Sets the color key (transparent pixel) in a blittable surface 
	* and RLE acceleration.
	* SDL_SetColorKey() wrapper.
	**/
	public native void setColorKey(int flags, int colorkey);

	/**
	* Sets the color key (transparent pixel) in a blittable surface 
	* and RLE acceleration.
	* SDL_SetColorKey() wrapper.
	**/
	public native void setColorKey(int flags);

	/**
	* Sets the color key (transparent pixel) in a blittable surface 
	* and RLE acceleration.
	* SDL_SetColorKey() wrapper.
	**/
	public void setColorKey()
	{
		this.setColorKey(SDL_SRCCOLORKEY|SDL_RLEACCEL);
	}

	/**
	* Sets the clipping rectangle for a surface.
	* SDL_SetClipRect() wrapper.
	**/
	public native boolean setClipRect(int x, int y, int w, int h);

	/**
	* Sets the clipping rectangle to the full size of the surface.
	* SDL_SetClipRect() wrapper.
	**/
	public native boolean setClipRect();

	/**
	* Gets the clipping rectangle for a surface.
	* SDL_GetClipRect() wrapper.
	**/
	public native int[] getClipRect();

	/**
	* Adjust the alpha properties of a surface.
	* SDL_SetAlpha() wrapper.
	**/
	public native int setAlpha(int flags, int alpha);

	/**
	* Convert a surface to the display format.
	* SDL_DisplayFormat() wrapper.
	**/
	public native void displayFormat();

	/**
	* Convert a surface to the display format.
	* SDL_DisplayFormatAlpha() wrapper.
	**/
	public native void displayFormatAlpha();

	/**
	* Converts a surface to the same format as another surface.
	* SDL_ConvertSurface() wrapper.
	**/
	public native void convertSurface(PixelFormat format, int flags);

	/**
	* Clone surface.
	* SDL_ConvertSurface() wrapper.
	**/
	public native Surface cloneSurface();

	/**
	* Get pixel format.
	**/
	public native PixelFormat getPixelFormat();

	/**
	* Sets the colors in the palette of an 8-bit surface.
	* SDL_SetPalette() wrapper.
	**/
	public native void setPalette(Color[] palette);

	/**
	* Sets the colors in the palette of an 8-bit surface.
	* SDL_SetPalette() wrapper.
	**/
	public void setPalette(Palette palette)
	{
		this.setPalette(palette, 0, 0, palette.getSize());
	}

	/**
	* Sets the colors in the palette of an 8-bit surface.
	* SDL_SetPalette() wrapper.
	**/
	public void setPalette(Palette palette, int index)
	{
		this.setPalette(palette, index, 0, palette.getSize());
	}

	/**
	* Sets the colors in the palette of an 8-bit surface.
	* SDL_SetPalette() wrapper.
	**/
	public native void setPalette(Palette palette, int index, int firstcolor, int ncolors);

	/**
	* Gets palette.
	**/
	public Palette getPalette()
	{
		return this.getPixelFormat().getPalette();
	}

	/**
	* Map a RGB color value to a pixel format.
	* SDL_MapRGB()
	**/
	public int mapRGB(Color color)
	{
		return this.getPixelFormat().mapRGB(color);
	}

	/**
	* Map a RGB color value to a pixel format.
	* SDL_MapRGB()
	**/
	public int mapRGB(int red, int green, int blue)
	{
		return this.getPixelFormat().mapRGB(red, green, blue);
	}

	/**
	* Map a RGBA color value to a pixel format.
	* SDL_MapRGBA() wrapper.
	**/
	public int mapRGBA(Color color)
	{
		return this.getPixelFormat().mapRGBA(color);
	}

	/**
	* Map a RGBA color value to a pixel format.
	* SDL_MapRGBA() wrapper.
	**/
	public int mapRGBA(int red, int green, int blue, int alpha)
	{
		return this.getPixelFormat().mapRGBA(red, green, blue, alpha);
	}

	/**
	* Get RGB values from a pixel in the specified pixel format.
	* SDL_GetRGB() wrapper.
	**/
	public int[] getRGB(int pixel)
	{
		return this.getPixelFormat().getRGB(pixel);
	}

	/**
	* Get RGBA values from a pixel in the specified pixel format.
	* SDL_GetRGBA() wrapper.
	**/
	public int[] getRGBA(int pixel)
	{
		return this.getPixelFormat().getRGBA(pixel);
	}

	/**
	* Gets pixels.
	* Copy surface's pixels, and return it.
	**/
	public native byte[] getPixels();

	/**
	* Sets pixels.
	**/
	public native void setPixels(byte[] pixels);

	/**
	* Free surface.
	* SDL_FreeSurface() wrapper.
	**/
	public native void freeSurface();

	protected void finalize() throws Throwable
	{
		this.freeSurface();
		super.finalize();
	}

	protected native void initRGBSurface(int flags, 
		int width, int height, int bpp,
		int rmask, int gmask, int bmask, int amask);

	protected native void initRGBSurfaceFrom(byte[] pixels,
		int width, int height, int bpp, int pitch,
		int rmask, int gmask, int bmask, int amask);

	protected native void initRGBSurfaceFrom(short[] pixels,
		int width, int height, int bpp, int pitch,
		int rmask, int gmask, int bmask, int amask);

	protected native void initRGBSurfaceFrom(int[] pixels,
		int width, int height, int bpp, int pitch,
		int rmask, int gmask, int bmask, int amask);

}

