/************************************************************
Copyright (C) 2005 Masahiko SAWAI All Rights Reserved. 
************************************************************/
#include "sdl4gcj/mixer/Mixer.h"

#include "sdl4gcj/mixer/MixerConstants.h"
#include "sdl4gcj/mixer/Music.h"
#include "sdl4gcj/mixer/Chunk.h"
#include "sdl4gcj/SDLException.h"

#include <gcj/cni.h>
#include <gcj/javaprims.h>
#include <stdlib.h>
#include <string.h>
#include <SDL_mixer.h>


using namespace sdl4gcj::mixer;

// function
static
inline
sdl4gcj::SDLException*
createFreedMusicException()
{
	return new sdl4gcj::SDLException(JvNewStringLatin1("This Music was already freed."));
}

static
inline
sdl4gcj::SDLException*
createFreedChunkException()
{
	return new sdl4gcj::SDLException(JvNewStringLatin1("This Chunk was already freed."));
}


// class methods

Mixer*
Mixer::openAudio(jint frequency, jint format, jint channels, jint chunksize)
{
	JvInitClass(&Mixer::class$);
	Mixer* mixer = NULL;

	int resultCode = ::Mix_OpenAudio(frequency, format, channels, chunksize);
	if (resultCode < 0) throw new sdl4gcj::SDLException();

	mixer = new Mixer();

	return mixer;
}


// instance methods

void
Mixer::closeAudio()
{
	::Mix_CloseAudio();
}

jint
Mixer::getFrequency()
{
	int frequency;
	Mix_QuerySpec(&frequency, NULL, NULL);
	return frequency;
}

jint
Mixer::getFormat()
{
	Uint16 format;
	Mix_QuerySpec(NULL, &format, NULL);
	return (jint)format;
}

jint
Mixer::getChannels()
{
	int channels;
	Mix_QuerySpec(NULL, NULL, &channels);
	return channels;
}

jint
Mixer::groupChannel(jint channel, jint tag)
{
	return ::Mix_GroupChannel(channel, tag);
}

jint
Mixer::groupChannels(jint fromChannel, jint toChannel, jint tag)
{
	return ::Mix_GroupChannels(fromChannel, toChannel, tag);
}

jint
Mixer::groupAvailable(jint tag)
{
	return ::Mix_GroupAvailable(tag);
}

jint
Mixer::groupCount(jint tag)
{
	return ::Mix_GroupCount(tag);
}

jint
Mixer::groupOldest(jint tag)
{
	return ::Mix_GroupOldest(tag);
}

jint
Mixer::groupNewer(jint tag)
{
	return ::Mix_GroupNewer(tag);
}

jint
Mixer::allocateChannels(jint numberOfChannels)
{
	return ::Mix_AllocateChannels(numberOfChannels);
}

jboolean
Mixer::playing(jint channel)
{
	return (::Mix_Playing(channel) != 0) ? true : false;
}

jboolean
Mixer::playingMusic()
{
	return (::Mix_PlayingMusic() != 0) ? true : false;
}

jint
Mixer::playChannel(jint channel, Chunk* chunk, jint loops, jint ticks)
{
	if (chunk->implementation == NULL) throw createFreedChunkException();
	Mix_Chunk* nativeChunk = (Mix_Chunk*)chunk->implementation;
	return ::Mix_PlayChannelTimed(channel, nativeChunk, loops, ticks);
}

jint
Mixer::fadeInChannel(jint channel, Chunk* chunk, jint loops, jint fadeTime, jint ticks)
{
	if (chunk->implementation == NULL) throw createFreedChunkException();
	Mix_Chunk* nativeChunk = (Mix_Chunk*)chunk->implementation;
	return ::Mix_FadeInChannelTimed(channel, nativeChunk, loops, fadeTime, ticks);
}

void
Mixer::playMusic(Music* music, jint loops)
{
	if (music->implementation == NULL) throw createFreedMusicException();
	Mix_Music* nativeMusic = (Mix_Music*)music->implementation;
	int resultCode = ::Mix_PlayMusic(nativeMusic, loops);
	if (resultCode == -1) throw new sdl4gcj::SDLException();
}

void
Mixer::fadeInMusic(Music* music, jint loops, jint ms)
{
	if (music->implementation == NULL) throw createFreedMusicException();
	Mix_Music* nativeMusic = (Mix_Music*)music->implementation;
	int resultCode = ::Mix_FadeInMusic(nativeMusic, loops, ms);
	if (resultCode == -1) throw new sdl4gcj::SDLException();
}

void
Mixer::fadeInMusicPos(Music* music, jint loops, jint ms, jdouble position)
{
	if (music->implementation == NULL) throw createFreedMusicException();
	Mix_Music* nativeMusic = (Mix_Music*)music->implementation;
	int resultCode = ::Mix_FadeInMusicPos(nativeMusic, loops, ms, position);
	if (resultCode == -1) throw new sdl4gcj::SDLException();
}

void 
Mixer::pause(jint channel)
{
	::Mix_Pause(channel);
}

void 
Mixer::resume(jint channel)
{
	::Mix_Resume(channel);
}

jboolean
Mixer::paused(jint channel)
{
	return (::Mix_Paused(channel) != 0) ? true : false;
}

void 
Mixer::pauseMusic()
{
	::Mix_PauseMusic();
}

void 
Mixer::resumeMusic()
{
	::Mix_ResumeMusic();
}

void 
Mixer::rewindMusic()
{
	::Mix_RewindMusic();
}

jboolean
Mixer::pausedMusic()
{
	return (::Mix_PausedMusic() != 0) ? true : false;
}

void
Mixer::setMusicPosition(jdouble position)
{
	int resultCode = ::Mix_SetMusicPosition(position);
	if (resultCode != 0) throw new sdl4gcj::SDLException();
}

jint
Mixer::volume(jint channel, jint volume)
{
	return ::Mix_Volume(channel, volume);
}

jint
Mixer::volumeMusic(jint volume)
{
	return ::Mix_VolumeMusic(volume);
}

jint
Mixer::haltChannel(jint channel)
{
	return ::Mix_HaltChannel(channel);
}

jint
Mixer::haltGroup(jint tag)
{
	return ::Mix_HaltGroup(tag);
}

jint
Mixer::haltMusic()
{
	return ::Mix_HaltMusic();
}

jint
Mixer::expireChannel(jint channel, jint ticks)
{
	return ::Mix_ExpireChannel(channel, ticks);
}

jint
Mixer::fadeOutChannel(jint channel, jint ms)
{
	return ::Mix_FadeOutChannel(channel, ms);
}

jint
Mixer::fadeOutGroup(jint tag, jint ms)
{
	return ::Mix_FadeOutGroup(tag, ms);
}

jint
Mixer::fadeOutMusic(jint ms)
{
	return ::Mix_FadeOutMusic(ms);
}

jint
Mixer::fadingMusic()
{
	return ::Mix_FadingMusic();
}

jint
Mixer::fadingChannel(jint channel)
{
	return ::Mix_FadingChannel(channel);
}

void
Mixer::setPanning(jint channel, jshort left, jshort right)
{
	int resultCode = ::Mix_SetPanning(channel, (Uint8)left, (Uint8)right);
	if (resultCode == 0) throw new sdl4gcj::SDLException();
}

void
Mixer::setPosition(jint channel, jshort angle, jshort distance)
{
	int resultCode = ::Mix_SetPosition(channel, (Sint16)angle, (Uint8)distance);
	if (resultCode == 0) throw new sdl4gcj::SDLException();
}

void
Mixer::setDistance(jint channel, jshort distance)
{
	int resultCode = ::Mix_SetDistance(channel, (Uint8)distance);
	if (resultCode == 0) throw new sdl4gcj::SDLException();
}

void
Mixer::setReverseStereo(jint channel, jboolean flip)
{
	int resultCode = ::Mix_SetReverseStereo(channel, flip);
	if (resultCode == 0) throw new sdl4gcj::SDLException();
}

jint 
Mixer::reserveChannels(jint num)
{
	return ::Mix_ReserveChannels(num);
}


