package org.seasar.extension.jdbc.impl;

import java.lang.reflect.Field;
import java.sql.DatabaseMetaData;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.StringTokenizer;

import org.seasar.extension.jdbc.BeanMetaData;
import org.seasar.extension.jdbc.ColumnNotFoundRuntimeException;
import org.seasar.extension.jdbc.PropertyType;
import org.seasar.extension.jdbc.RelationPropertyType;
import org.seasar.extension.jdbc.ValueType;
import org.seasar.extension.jdbc.types.ValueTypes;
import org.seasar.framework.beans.BeanDesc;
import org.seasar.framework.beans.PropertyDesc;
import org.seasar.framework.beans.PropertyNotFoundRuntimeException;
import org.seasar.framework.beans.factory.BeanDescFactory;
import org.seasar.framework.util.CaseInsensitiveMap;
import org.seasar.framework.util.ClassUtil;
import org.seasar.framework.util.DatabaseMetaDataUtil;
import org.seasar.framework.util.FieldUtil;

/**
 * @author higa
 *  
 */
public class BeanMetaDataImpl implements BeanMetaData {

	private Class beanClass_;

	private String tableName_;

	private CaseInsensitiveMap propertyTypes_ = new CaseInsensitiveMap();

	private Map propertyTypesByColumnName_ = new CaseInsensitiveMap();

	private List relationPropertyTypes_ = new ArrayList();
	
	private boolean hasDatabaseMetaData_ = false;
	
	private boolean persistent_ = true;
	
	private String[] primaryKeys_ = new String[0];

	public BeanMetaDataImpl(Class beanClass) {
		beanClass_ = beanClass;
		BeanDesc beanDesc = BeanDescFactory.getBeanDesc(beanClass);
		setupTableName(beanDesc);
		setupPropertyType(beanDesc);
	}

	/**
	 * @see org.seasar.extension.jdbc.BeanMetaData#getBeanClass()
	 */
	public Class getBeanClass() {
		return beanClass_;
	}

	/**
	 * @see org.seasar.extension.jdbc.BeanMetaData#getTableName()
	 */
	public String getTableName() {
		return tableName_;
	}

	/**
	 * @see org.seasar.extension.jdbc.BeanMetaData#getPropertyTypeSize()
	 */
	public int getPropertyTypeSize() {
		return propertyTypes_.size();
	}

	/**
	 * @see org.seasar.extension.jdbc.BeanMetaData#getPropertyType(int)
	 */
	public PropertyType getPropertyType(int index) {
		return (PropertyType) propertyTypes_.get(index);
	}

	/**
	 * @see org.seasar.extension.jdbc.BeanMetaData#getPropertyType(java.lang.String)
	 */
	public PropertyType getPropertyType(String propertyName)
			throws PropertyNotFoundRuntimeException {

		PropertyType propertyType = (PropertyType) propertyTypes_
				.get(propertyName);
		if (propertyType == null) {
			throw new PropertyNotFoundRuntimeException(beanClass_, propertyName);
		}
		return propertyType;
	}

	/**
	 * @see org.seasar.extension.jdbc.BeanMetaData#getPropertyTypeByColumnName(java.lang.String)
	 */
	public PropertyType getPropertyTypeByColumnName(String columnName)
			throws ColumnNotFoundRuntimeException {

		PropertyType propertyType = (PropertyType) propertyTypesByColumnName_
				.get(columnName);
		if (propertyType == null) {
			throw new ColumnNotFoundRuntimeException(tableName_, columnName);
		}
		return propertyType;
	}

	/**
	 * @see org.seasar.extension.jdbc.BeanMetaData#getRelationPropertyTypeSize()
	 */
	public int getRelationPropertyTypeSize() {
		return relationPropertyTypes_.size();
	}

	/**
	 * @see org.seasar.extension.jdbc.BeanMetaData#getRelationPropertyType(int)
	 */
	public RelationPropertyType getRelationPropertyType(int index) {
		return (RelationPropertyType) relationPropertyTypes_.get(index);
	}

	/**
	 * @see org.seasar.extension.jdbc.BeanMetaData#getRelationPropertyType(java.lang.String)
	 */
	public RelationPropertyType getRelationPropertyType(String propertyName)
			throws PropertyNotFoundRuntimeException {

		for (int i = 0; i < getRelationPropertyTypeSize(); i++) {
			RelationPropertyType rpt = (RelationPropertyType) relationPropertyTypes_
					.get(i);
			if (rpt != null
					&& rpt.getPropertyName().equalsIgnoreCase(propertyName)) {
				return rpt;
			}
		}
		throw new PropertyNotFoundRuntimeException(beanClass_, propertyName);
	}

	private void setupTableName(BeanDesc beanDesc) {
		if (beanDesc.hasField(TABLE_KEY)) {
			Field field = beanDesc.getField(TABLE_KEY);
			tableName_ = (String) FieldUtil.get(field, null);
		} else {
			tableName_ = ClassUtil.getShortClassName(beanClass_);
		}
	}

	private void setupPropertyType(BeanDesc beanDesc) {
		for (int i = 0; i < beanDesc.getPropertyDescSize(); ++i) {
			PropertyDesc pd = beanDesc.getPropertyDesc(i);
			String relnoKey = pd.getPropertyName() + RELNO_KEY_SUFFIX;
			if (beanDesc.hasField(relnoKey)) {
				RelationPropertyType rpt = createRelationPropertyType(beanDesc,
						pd, relnoKey);
				addRelationPropertyType(rpt);
			} else {
				PropertyType pt = createPropertyType(beanDesc, pd);
				addPropertyType(pt);
			}
		}
	}

	private RelationPropertyType createRelationPropertyType(BeanDesc beanDesc,
			PropertyDesc propertyDesc, String relnoKey) {

		Field field = beanDesc.getField(relnoKey);
		String[] myKeys = new String[0];
		String[] yourKeys = new String[0];
		int relno = FieldUtil.getInt(field, null);
		String relkeysKey = propertyDesc.getPropertyName() + RELKEYS_KEY_SUFFIX;
		if (beanDesc.hasField(relkeysKey)) {
			Field field2 = beanDesc.getField(relkeysKey);
			String relkeys = (String) FieldUtil.get(field2, null);
			StringTokenizer st = new StringTokenizer(relkeys, " \t\n\r\f,");
			List myKeyList = new ArrayList();
			List yourKeyList = new ArrayList();
			while (st.hasMoreTokens()) {
				String token = st.nextToken();
				int index = token.indexOf(':');
				if (index > 0) {
					myKeyList.add(token.substring(0, index));
					yourKeyList.add(token.substring(index + 1));
				} else {
					myKeyList.add(token);
					yourKeyList.add(token);
				}
			}
			myKeys = (String[]) myKeyList.toArray(new String[myKeyList
					.size()]);
			yourKeys = (String[]) yourKeyList
					.toArray(new String[yourKeyList.size()]);
		}
		RelationPropertyType rpt = new RelationPropertyTypeImpl(propertyDesc,
				relno, myKeys, yourKeys);
		return rpt;
	}

	private void addRelationPropertyType(RelationPropertyType rpt) {
		for (int i = relationPropertyTypes_.size(); i <= rpt.getRelationNo(); ++i) {
			relationPropertyTypes_.add(null);
		}
		relationPropertyTypes_.set(rpt.getRelationNo(), rpt);
	}

	private PropertyType createPropertyType(BeanDesc beanDesc,
			PropertyDesc propertyDesc) {

		String columnNameKey = propertyDesc.getPropertyName()
				+ COLUMN_KEY_SUFFIX;
		String columnName = propertyDesc.getPropertyName();
		if (beanDesc.hasField(columnNameKey)) {
			Field field = beanDesc.getField(columnNameKey);
			columnName = (String) FieldUtil.get(field, null);
		}
		ValueType valueType = ValueTypes.getValueType(propertyDesc
				.getPropertyType());
		PropertyType pt = new PropertyTypeImpl(propertyDesc, valueType,
				columnName);
		return pt;
	}

	private void addPropertyType(PropertyType propertyType) {
		propertyTypes_.put(propertyType.getPropertyName(), propertyType);
		propertyTypesByColumnName_.put(propertyType.getColumnName(),
				propertyType);
	}
	
	/**
	 * @see org.seasar.extension.jdbc.BeanMetaData#isPersistent()
	 */
	public boolean isPersistent() {
		return persistent_;
	}
	
	/**
	 * @see org.seasar.extension.jdbc.BeanMetaData#setupDatabaseMetaData(java.sql.DatabaseMetaData)
	 */
	public synchronized void setupDatabaseMetaData(DatabaseMetaData dbMetaData) {
		if (hasDatabaseMetaData_) {
			return;
		}
		Set primaryKeySet = DatabaseMetaDataUtil.getPrimaryKeySet(dbMetaData,
				tableName_);
		Set columnSet = DatabaseMetaDataUtil.getColumnSet(dbMetaData,
				tableName_);
		for (int i = 0; i < getPropertyTypeSize(); ++i) {
			PropertyType pt = getPropertyType(i);
			if (primaryKeySet.contains(pt.getColumnName())) {
				pt.setPrimaryKey(true);
			} else {
				pt.setPrimaryKey(false);
			}
			if (columnSet.contains(pt.getColumnName())) {
				pt.setPersistent(true);
			} else {
				pt.setPersistent(false);
			}
		}
		persistent_ = columnSet.size() > 0;
		primaryKeys_ = (String[]) primaryKeySet.toArray(
				new String[primaryKeySet.size()]);
		hasDatabaseMetaData_ = true;
	}
	
	/**
	 * @see org.seasar.extension.jdbc.BeanMetaData#getPrimaryKeySize()
	 */
	public int getPrimaryKeySize() {
		return primaryKeys_.length;
	}
	
	/**
	 * @see org.seasar.extension.jdbc.BeanMetaData#getPrimaryKey(int)
	 */
	public String getPrimaryKey(int index) {
		return primaryKeys_[index];
	}
}