package org.seasar.dao.impl;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.sql.DataSource;

import org.seasar.dao.BeanMetaData;
import org.seasar.dao.DaoMetaData;
import org.seasar.dao.Dbms;
import org.seasar.dao.IllegalSignatureRuntimeException;
import org.seasar.dao.SqlCommand;
import org.seasar.extension.jdbc.ResultSetHandler;
import org.seasar.extension.jdbc.impl.ObjectResultSetHandler;
import org.seasar.framework.beans.BeanDesc;
import org.seasar.framework.beans.MethodNotFoundRuntimeException;
import org.seasar.framework.beans.factory.BeanDescFactory;
import org.seasar.framework.util.ClassUtil;
import org.seasar.framework.util.ConnectionUtil;
import org.seasar.framework.util.DataSourceUtil;
import org.seasar.framework.util.FieldUtil;
import org.seasar.framework.util.MethodUtil;
import org.seasar.framework.util.ResourceUtil;
import org.seasar.framework.util.StringUtil;
import org.seasar.framework.util.TextUtil;

/**
 * @author higa
 *  
 */
public class DaoMetaDataImpl implements DaoMetaData {

	private static final String[] INSERT_NAMES = new String[] { "insert",
			"create", "add" };

	private static final String[] UPDATE_NAMES = new String[] { "update",
			"modify", "store" };

	private static final String[] DELETE_NAMES = new String[] { "delete",
			"remove" };

	private Class daoClass_;

	private BeanDesc daoBeanDesc_;

	private Dbms dbms_;

	private Class beanClass_;

	private BeanMetaData beanMetaData_;

	private Map sqlCommands_ = new HashMap();

	public DaoMetaDataImpl(Class daoClass, Dbms dbms, DataSource ds) {
		daoClass_ = daoClass;
		dbms_ = dbms;
		daoBeanDesc_ = BeanDescFactory.getBeanDesc(daoClass);
		Field beanField = daoBeanDesc_.getField(BEAN_KEY);
		beanClass_ = (Class) FieldUtil.get(beanField, null);
		beanMetaData_ = BeanMetaDataFactory.getBeanMetaData(beanClass_);
		setupDatabaseMetaData(ds);
		setupSqlCommand();
	}
	
	private void setupDatabaseMetaData(DataSource ds) {
		if (!beanMetaData_.setupDoneDatabaseMetaData()) {
			Connection con = DataSourceUtil.getConnection(ds);
			try {
				DatabaseMetaData dbMetaData = ConnectionUtil
						.getMetaData(con);
				beanMetaData_.setupDatabaseMetaData(dbMetaData);
			} finally {
				ConnectionUtil.close(con);
			}
		}
	}

	private void setupSqlCommand() {
		String[] names = daoBeanDesc_.getMethodNames();
		for (int i = 0; i < names.length; ++i) {
			Method[] methods = daoBeanDesc_.getMethods(names[i]);
			if (methods.length == 1 && MethodUtil.isAbstract(methods[0])) {
				setupMethod(methods[0]);
			}
		}
	}

	private void setupMethod(Method method) {
		String base = daoClass_.getName().replace('.', '/') + "_"
				+ method.getName();
		String dbmsPath = base + dbms_.getSuffix() + ".sql";
		String standardPath = base + ".sql";
		if (ResourceUtil.isExist(dbmsPath)) {
			String sql = TextUtil.readText(dbmsPath);
			setupMethodByManual(method, sql);
		} else if (ResourceUtil.isExist(standardPath)) {
			String sql = TextUtil.readText(standardPath);
			setupMethodByManual(method, sql);
		} else {
			setupMethodByAuto(method);
		}
	}

	private void setupMethodByManual(Method method, String sql) {
		if (isSelect(method)) {
			setupSelectMethodByManual(method, sql);
		} else {
			setupUpdateMethodByManual(method, sql);
		}
	}

	private void setupMethodByAuto(Method method) {
		if (isInsert(method.getName())) {
			setupInsertMethodByAuto(method);
		} else if (isUpdate(method.getName())) {
			setupUpdateMethodByAuto(method);
		} else if (isDelete(method.getName())) {
			setupDeleteMethodByAuto(method);
		} else {
			setupSelectMethodByAuto(method);
		}
	}

	private void setupSelectMethodByManual(Method method, String sql) {
		SelectDynamicCommand cmd = new SelectDynamicCommand(beanMetaData_,
				createResultSetHandler(method));
		cmd.setSql(sql);
		cmd.setArgNames(getArgNames(method.getName()));
		sqlCommands_.put(method.getName(), cmd);
	}

	private ResultSetHandler createResultSetHandler(Method method) {
		if (method.getReturnType().isAssignableFrom(List.class)) {
			return new BeanListMetaDataResultSetHandler(beanClass_);
		} else if (method.getReturnType().isAssignableFrom(beanClass_)) {
			return new BeanMetaDataResultSetHandler(beanClass_);
		} else {
			return new ObjectResultSetHandler();
		}
	}

	private void setupUpdateMethodByManual(Method method, String sql) {
		UpdateDynamicCommand cmd = new UpdateDynamicCommand();
		cmd.setSql(sql);
		String[] argNames = getArgNames(method.getName());
		if (argNames.length == 0 && method.getParameterTypes().length == 1
				&& method.getParameterTypes()[0].isAssignableFrom(beanClass_)) {
			argNames = new String[] { StringUtil.decapitalize(ClassUtil
					.getShortClassName(beanClass_)) };
		}
		cmd.setArgNames(argNames);
		sqlCommands_.put(method.getName(), cmd);
	}

	private void setupInsertMethodByAuto(Method method) {
		checkAutoUpdateMethod(method);
		InsertAutoStaticCommand cmd = new InsertAutoStaticCommand(beanMetaData_);
		sqlCommands_.put(method.getName(), cmd);
	}

	private void setupUpdateMethodByAuto(Method method) {
		checkAutoUpdateMethod(method);
		UpdateAutoStaticCommand cmd = new UpdateAutoStaticCommand(beanMetaData_);
		sqlCommands_.put(method.getName(), cmd);
	}

	private void setupDeleteMethodByAuto(Method method) {
		checkAutoUpdateMethod(method);
		DeleteAutoStaticCommand cmd = new DeleteAutoStaticCommand(beanMetaData_);
		sqlCommands_.put(method.getName(), cmd);
	}

	private void setupSelectMethodByAuto(Method method) {
		SelectDynamicCommand cmd = new SelectDynamicCommand(beanMetaData_,
				createResultSetHandler(method));
		String[] argNames = getArgNames(method.getName());
		cmd.setArgNames(argNames);
		String sql = dbms_.getAutoSelectSql(getBeanMetaData());
		StringBuffer buf = new StringBuffer(sql);
		if (argNames.length != 0) {
			boolean began = false;
			if (!(sql.lastIndexOf("WHERE") > 0)) {
				buf.append("/*BEGIN*/ WHERE ");
				began = true;
			}
			for (int i = 0; i < argNames.length; ++i) {
				String columnName = beanMetaData_
						.convertFullColumnName(argNames[i]);
				buf.append("/*IF ");
				buf.append(argNames[i]);
				buf.append(" != null*/");
				buf.append(" ");
				if (!began || i != 0) {
					buf.append("AND ");
				}
				buf.append(columnName);
				buf.append(" = /*");
				buf.append(argNames[i]);
				buf.append("*/null");
				buf.append("/*END*/");
			}
			if (began) {
				buf.append("/*END*/");
			}
			
		}
		buf.append(getOrderBy(method.getName()));
		cmd.setSql(buf.toString());
		sqlCommands_.put(method.getName(), cmd);
	}

	private void checkAutoUpdateMethod(Method method) {
		if (method.getParameterTypes().length != 1
				|| !method.getParameterTypes()[0].isAssignableFrom(beanClass_)) {
			throw new IllegalSignatureRuntimeException("EDAO0006", method
					.toString());
		}
	}

	private boolean isSelect(Method method) {
		if (isInsert(method.getName())) {
			return false;
		}
		if (isUpdate(method.getName())) {
			return false;
		}
		if (isDelete(method.getName())) {
			return false;
		}
		return true;
	}

	private boolean isInsert(String methodName) {
		for (int i = 0; i < INSERT_NAMES.length; ++i) {
			if (methodName.startsWith(INSERT_NAMES[i])) {
				return true;
			}
		}
		return false;
	}

	private boolean isUpdate(String methodName) {
		for (int i = 0; i < UPDATE_NAMES.length; ++i) {
			if (methodName.startsWith(UPDATE_NAMES[i])) {
				return true;
			}
		}
		return false;
	}

	private boolean isDelete(String methodName) {
		for (int i = 0; i < DELETE_NAMES.length; ++i) {
			if (methodName.startsWith(DELETE_NAMES[i])) {
				return true;
			}
		}
		return false;
	}

	private String[] getArgNames(String methodName) {
		String argsKey = methodName + ARGS_KEY_SUFFIX;
		if (daoBeanDesc_.hasField(argsKey)) {
			Field argNamesField = daoBeanDesc_.getField(argsKey);
			String argNames = (String) FieldUtil.get(argNamesField, null);
			return StringUtil.split(argNames, " ,");
		} else {
			return new String[0];
		}
	}
	
	private String getOrderBy(String methodName) {
		String orderKey = methodName + ORDER_KEY_SUFFIX;
		if (daoBeanDesc_.hasField(orderKey)) {
			Field orderField = daoBeanDesc_.getField(orderKey);
			String order = (String) FieldUtil.get(orderField, null);
			String[] names = StringUtil.split(order, " ,");
			StringBuffer buf = new StringBuffer(" ORDER BY ");
			for (int i = 0; i < names.length; ++i) {
				String name = names[i];
				if (name.equalsIgnoreCase("ASC") || name.equalsIgnoreCase("DESC")) {
					buf.setLength(buf.length() - 2);
					buf.append(" ");
					buf.append(name);
					buf.append(", ");
				} else {
					String columnName = beanMetaData_
							.convertFullColumnName(name);
					buf.append(columnName);
					buf.append(", ");
				}
			}
			buf.setLength(buf.length() - 2);
			return buf.toString();
		} else {
			return "";
		}
	}

	/**
	 * @see org.seasar.dao.DaoMetaData#getBeanClass()
	 */
	public Class getBeanClass() {
		return daoClass_;
	}

	/**
	 * @see org.seasar.dao.DaoMetaData#getBeanMetaData()
	 */
	public BeanMetaData getBeanMetaData() {
		return beanMetaData_;
	}

	/**
	 * @see org.seasar.dao.DaoMetaData#getSqlCommand(java.lang.String)
	 */
	public SqlCommand getSqlCommand(String methodName)
			throws MethodNotFoundRuntimeException {

		SqlCommand cmd = (SqlCommand) sqlCommands_.get(methodName);
		if (cmd == null) {
			throw new MethodNotFoundRuntimeException(daoClass_, methodName,
					null);
		}
		return cmd;
	}

	/**
	 * @see org.seasar.dao.DaoMetaData#hasSqlCommand(java.lang.String)
	 */
	public boolean hasSqlCommand(String methodName) {
		return sqlCommands_.containsKey(methodName);
	}
}