package org.seasar.sql;

import java.sql.SQLException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import javax.sql.ConnectionEvent;
import javax.transaction.RollbackException;
import javax.transaction.Status;
import javax.transaction.Synchronization;
import javax.transaction.SystemException;
import javax.transaction.Transaction;

import org.seasar.log.Logger;
import org.seasar.timer.TimeoutManager;
import org.seasar.timer.TimeoutTarget;
import org.seasar.timer.TimeoutTask;
import org.seasar.transaction.TransactionManagerImpl;
import org.seasar.transaction.XAResourceManager;
import org.seasar.util.ELinkedList;
import org.seasar.util.SeasarRuntimeException;

public final class ConnectionPool implements XAConnectionEventListener, Synchronization {

    private final ConnectionPoolMetaData _connectionPoolMetaData;
    private final Map _activePool = new HashMap();
    private final Map _txPool = new HashMap();
    private final ELinkedList _freePool = new ELinkedList();

    public ConnectionPool(final ConnectionPoolMetaData connectionPoolMetaData) {
        if (connectionPoolMetaData == null) {
            throw new SeasarRuntimeException("ESSR0007", new Object[]{"connectionPoolMetafData"});
        }
        _connectionPoolMetaData = connectionPoolMetaData;
    }

    public final String getDataSourceName() {
        return _connectionPoolMetaData.getDataSourceName();
    }

    public final synchronized XAConnectionImpl requestXAConnection() throws SQLException {
        while (_activePool.size() >= _connectionPoolMetaData.getPoolSize()) {
            try {
                wait();
            } catch (InterruptedException ign) {
            }
        }
        XAConnectionImpl xaCon = requestFreeXAConnection();
        if (xaCon == null) {
            xaCon = _connectionPoolMetaData.getXAConnection();
            xaCon.addConnectionEventListener(this);
        }
        
        try {
            XAResourceManager.getInstance().addXAResource(xaCon.getXAResource());
            _activePool.put(xaCon, null);
            Transaction tx = TransactionManagerImpl.getInstance().getTransaction();
        	if (tx != null) {
            	tx.registerSynchronization(this);
        	}
        } catch (SystemException se) {
            throw new SQLException(se.toString());
        } catch (RollbackException re) {
            throw new SQLException(re.toString());
        }
        return xaCon;
    }

    public final synchronized void returnXAConnection(final XAConnectionImpl xaConnection) {
        XAResourceManager.getInstance().removeXAResource(xaConnection.getXAResource());
        _activePool.remove(xaConnection);
        Transaction tx = TransactionManagerImpl.getInstance().getTransaction();
        if (tx != null) {
            _txPool.put(tx, xaConnection);
        } else {
            returnFreePool(xaConnection);
        }
        notify();
    }

    public final synchronized void releaseXAConnection(final XAConnectionImpl xaConnection) {
        XAResourceManager.getInstance().removeXAResource(xaConnection.getXAResource());
        _activePool.remove(xaConnection);
        xaConnection.removeConnectionEventListener(this);
        closeXAConnection(xaConnection);
        notify();
    }

    public final int getPoolSize() {
        return _freePool.size();
    }

    public final int getActviePoolSize() {
        return _activePool.size();
    }

    public final void connectionClosed(final ConnectionEvent event) {
        returnXAConnection((XAConnectionImpl) event.getSource());
    }

    public final void connectionErrorOccurred(final ConnectionEvent event) {
        releaseXAConnection((XAConnectionImpl) event.getSource());
    }
    
    public final void xaConnectionClosed(final ConnectionEvent event) {
        releaseXAConnection((XAConnectionImpl) event.getSource());
    }
    
    public final void afterCompletion(final int status) {
        switch (status) {
            case Status.STATUS_COMMITTED:
            case Status.STATUS_ROLLEDBACK:
                returnFreePoolForTx();
                break;
        }
    }

    public final void beforeCompletion() { }

    public final synchronized void close() {
        for (ELinkedList.Entry e = _freePool.getFirstEntry(); e != null; e = e.getNext()) {
            FreeItem item = (FreeItem) e.getElement();
            closeXAConnection(item._xaConnection);
            item.destroy();
        }
        _freePool.clear();
        for (Iterator i = _txPool.values().iterator(); i.hasNext(); ) {
            XAConnectionImpl xaCon = (XAConnectionImpl) i.next();
            closeXAConnection(xaCon);
        }
        _txPool.clear();
        for (Iterator i = _activePool.keySet().iterator(); i.hasNext(); ) {
            XAConnectionImpl xaCon = (XAConnectionImpl) i.next();
            XAResourceManager.getInstance().removeXAResource(xaCon.getXAResource());
            closeXAConnection(xaCon);
        }
        _activePool.clear();
    }

    private synchronized XAConnectionImpl requestFreeXAConnection() throws SQLException {
        XAConnectionImpl xaCon = requestXAConnectionFromTxPool();
        if (xaCon == null) {
            xaCon = requestXAConnectionFromFreePool();
        }
        if (xaCon != null) {
        	xaCon.init();
        	return xaCon;
        }
        return null;
    }


    private synchronized XAConnectionImpl requestXAConnectionFromTxPool() throws SQLException {
        Transaction tx = TransactionManagerImpl.getInstance().getTransaction();
        if (tx != null) {
            return (XAConnectionImpl) _txPool.remove(tx);
        }
        return null;
    }

	private synchronized XAConnectionImpl requestXAConnectionFromFreePool() throws SQLException {
		if (_freePool.isEmpty()) {
			return null;
		}
		FreeItem item = (FreeItem) _freePool.removeLast();
		XAConnectionImpl xaCon = item._xaConnection;
        item.destroy();
        return xaCon;
    }
    
    private synchronized void returnFreePoolForTx() {
        try {
            XAConnectionImpl xaCon = requestXAConnectionFromTxPool();
            if (xaCon != null) {
                returnFreePool(xaCon);
            }
        } catch (SQLException ex) {
            throw SeasarRuntimeException.convertSeasarRuntimeException(ex);
        }
    }
    
    private synchronized void returnFreePool(final XAConnectionImpl xaConnection) {
        _freePool.addLast(new FreeItem(xaConnection));
        notifyAll();
    }
    
    private void closeXAConnection(final XAConnectionImpl xaConnection) {
    	try {
    		xaConnection.closePhysicalConnection();
    	} catch (SQLException ex) {
    		Logger.getLogger(getClass()).log(ex);
    	}
    }

    private class FreeItem implements TimeoutTarget {

        private XAConnectionImpl _xaConnection;
        private TimeoutTask _timeoutTask;

        FreeItem(XAConnectionImpl xaConnection) {
            _xaConnection = xaConnection;
            _timeoutTask = TimeoutManager.getInstance().addTimeoutTarget(
                    this, _connectionPoolMetaData.getTimeout(), false);
        }

        public void expired() {
            synchronized (_freePool) {
                _freePool.remove(this);
            }
            closeXAConnection(_xaConnection);
            destroy();
        }

        void destroy() {
            _xaConnection = null;
            _timeoutTask.cancel();
        }
    }
}
