package org.seasar.util;

import java.beans.BeanInfo;
import java.beans.IntrospectionException;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;

public class Reflector {

    private static Map _methodsMapMap = new SMap();
    private static Map _writeMethodMapMap = new SMap();
    private static Map _propertyDescMapCache = new SMap();
    private Class _targetClass;
    private Object _target;


    private Reflector(final Class targetClass, final Object target) {
        _targetClass = targetClass;
        _target = target;
    }

    public static Class getClass(final String className) {
        try {
            return Thread.currentThread().getContextClassLoader().loadClass(className);
        } catch (ClassNotFoundException ex) {
            throw new SeasarRuntimeException("ESSR0008", new Object[]{className, ex}, ex);
        }
    }


    public static Object newInstance(final String className) {
        return newInstance(getClass(className));
    }

    public static Object newInstance(final Class clazz) {
        try {
            return clazz.newInstance();
        } catch (Exception ex) {
            throw new SeasarRuntimeException("ESSR0008",
                    new Object[]{clazz.getName(), ex}, ex);
        }
    }

    public static Object newInstance(final String className, final Object[] arguments) {
        return newInstance(getClass(className), arguments);
    }


    public static Object newInstance(final Class clazz, final Object[] arguments) {
        Constructor constructor = getConstructor(clazz, arguments);
        try {
            return constructor.newInstance(arguments);
        } catch (Exception ex) {
            throw new SeasarRuntimeException("ESSR0008", new Object[]{clazz.getName(), ex}, ex);
        }
    }


    public static Method getMethod(final String className, final String methodName,
            final Class[] parameterTypes) {

        return getMethod(getClass(className), methodName, parameterTypes);
    }

    public static Method getMethod(final Class clazz, final String methodName,
            Class[] parameterTypes) {

        if (parameterTypes == null) {
            parameterTypes = ArrayUtil.EMPTY_CLASSES;
        }
        try {
            return clazz.getMethod(methodName, parameterTypes);
        } catch (NoSuchMethodException ignore) {
            Method[] methods = clazz.getDeclaredMethods();
            for (int i = 0; i < methods.length; i++) {
                if (methods[i].getName().equals(methodName) &&
                        ArrayUtil.equals(methods[i].getParameterTypes(), parameterTypes)) {

                    methods[i].setAccessible(true);
                    return methods[i];
                }
            }
            Class superClass = clazz.getSuperclass();
            if (superClass != null) {
                return getMethod(superClass, methodName, parameterTypes);
            } else {
                throw new SeasarRuntimeException("ESSR0009",
                        new Object[]{methodName, clazz.getName()});
            }
        }
    }
    
    public static Map getConstants(final String className) {
    	return getConstants(getClass(className));
    }
    
    public static Map getConstants(final Class clazz) {
    	Map ret = new SMap();
        Field[] fields = clazz.getFields();
        for (int i = 0; i < fields.length; i++) {
        	int mod = fields[i].getModifiers();
            if (Modifier.isStatic(mod) && Modifier.isFinal(mod)) {
                String name = fields[i].getName();
                Object value = getFieldValue(fields[i], null);
                ret.put(name, value);
            }
        }
        return ret;
    }


    public static Field getField(final Class clazz, final String fieldName) {
        try {
            Field[] fields = clazz.getDeclaredFields();
            for (int i = 0; i < fields.length; i++) {
                if (fields[i].getName().equals(fieldName)) {
                    fields[i].setAccessible(true);
                    return fields[i];
                }
            }
            Class superClass = clazz.getSuperclass();
            if (superClass != null) {
                return getField(superClass, fieldName);
            } else {
                return null;
            }
        } catch (Exception ex) {
            throw new SeasarRuntimeException("ESSR0009", new Object[]{fieldName, clazz.getName()}, ex);
        }
    }
    
    public static List getMethods(final Class clazz, final String methodName) {
        Map methodsMap = getMethodsMap(clazz);
        List ret = (List) methodsMap.get(methodName);
        if (ret == null) {
            throw new SeasarRuntimeException("ESSR0009", new Object[]{methodName, clazz.getName()});
        }
        return ret;
    }
    
    public static List getMethods(final Class clazz, final String methodName,
    		final int argSize) {
    	
    	List ret = new ArrayList();		
    	List methods = getMethods(clazz, methodName);
    	for (int i = 0; i < methods.size(); ++i) {
    		Method m = (Method) methods.get(i);
    		if (m.getParameterTypes().length == argSize) {
    			ret.add(m);
    		}
    	}
    	return ret;
    }


    public static Constructor getConstructor(final String className, final Object[] args) {
        return getConstructor(getClass(className), args);
    }

    public static Constructor getConstructor(final Class clazz, Object[] args) {
        if (args == null) {
            args = ArrayUtil.EMPTY_OBJECTS;
        }
        if (args.length == 0) {
            try {
                return clazz.getConstructor(null);
            } catch (NoSuchMethodException ex) {
                throw new SeasarRuntimeException("ESSR0008", new Object[]{clazz.getName(), ex}, ex);
            }
        }
        Constructor[] constructors = clazz.getDeclaredConstructors();
        outerLoop :
        for (int i = 0; i < constructors.length; i++) {
            Class[] parameterTypes = constructors[i].getParameterTypes();
            if (parameterTypes.length != args.length) {
                continue;
            }
            for (int j = 0; j < parameterTypes.length; j++) {
                if (isSameType(parameterTypes[j], args[j])) {
                    continue;
                } else {
                    continue outerLoop;
                }
            }
            constructors[i].setAccessible(true);
            return constructors[i];
        }
        throw new SeasarRuntimeException("ESSR0009", new Object[]{"Constructor", clazz.getName()});
    }

    public static boolean isSameType(final Class clazz, final Object object) {
        if (object == null) {
            if (clazz.isPrimitive()) {
                return false;
            } else {
                return true;
            }
        }
        if (clazz.isInstance(object)) {
            return true;
        }
        return clazz == getPrimitiveClass(object.getClass());
    }

    public static Class getPrimitiveClass(final Class clazz) {
        if (clazz == Integer.class) {
            return Integer.TYPE;
        } else if (clazz == Long.class) {
            return Long.TYPE;
        } else if (clazz == Double.class) {
            return Double.TYPE;
        } else if (clazz == Float.class) {
            return Float.TYPE;
        } else if (clazz == Boolean.class) {
            return Boolean.TYPE;
        } else if (clazz == Character.class) {
            return Character.TYPE;
        } else if (clazz == Byte.class) {
            return Byte.TYPE;
        } else {
            return null;
        }
    }

    public static Reflector create(final String className) {
        return create(className, ArrayUtil.EMPTY_OBJECTS);
    }

    public static Reflector create(final String className, final Object[] args) {
        if (className == null) {
            throw new SeasarRuntimeException("ESSR0007", new Object[]{"className"});
        }
        Class targetClass = getClass(className);
        Object target = newInstance(targetClass, args);
        return new Reflector(targetClass, target);
    }


    public static Reflector create(final String className, final String methodName) {
        return create(className, methodName, ArrayUtil.EMPTY_OBJECTS);
    }


    public static Reflector create(final String className, final String methodName,
            final Object[] args) {
        return create(getClass(className), methodName, args);
    }


    public static Reflector create(final Class clazz, final String methodName,
            final Object[] args) {

        if (clazz == null) {
            throw new SeasarRuntimeException("ESSR0007", new Object[]{"clazz"});
        }
        if (methodName == null) {
            throw new SeasarRuntimeException("ESSR0007", new Object[]{"methodName"});
        }
        Object target = null;
        Method method = getMethodByArgs(clazz, methodName, args);
        try {
            target = method.invoke(clazz, args);
        } catch (Exception ex) {
            throw new SeasarRuntimeException("ESSR0017", new Object[]{ex}, ex);
        }
        return new Reflector(clazz, target);
    }

	public static boolean isDeclaredException(final Method method, final Class exceptionClass) {
		Class[] exTypes = method.getExceptionTypes();
		for (int i = 0; i < exTypes.length; i++) {
			if (exTypes[i].isAssignableFrom(exceptionClass)) {
				return true;
			}
        }
        return false;
    }
    
    public static Object invoke(final Method method, final Object target,
            final Object[] args) throws SeasarException {

        try {
            return method.invoke(target, args);
        } catch (IllegalAccessException ex) {
            throw SeasarException.convertSeasarException(ex);
        } catch (IllegalArgumentException ex2) {
            throw SeasarException.convertSeasarException(ex2);
	    } catch (InvocationTargetException ex3) {
            Throwable t = ex3.getTargetException();
            if (t instanceof SeasarException) {
            	throw (SeasarException) t;
            } else if (t instanceof SeasarRuntimeException) {
        		throw (SeasarRuntimeException) t;
            } else if (t instanceof Error) {
        		throw (Error) t;
            } else if (t instanceof RuntimeException) {
        		throw (RuntimeException) t;
            } else {
            	throw SeasarException.convertSeasarException(t);
            }
        }
    }
    
    public static Object invokeNoException(final Method method, final Object target,
            final Object[] args) {

        try {
            return method.invoke(target, args);
        } catch (IllegalAccessException ex) {
            throw SeasarRuntimeException.convertSeasarRuntimeException(ex);
        } catch (IllegalArgumentException ex2) {
            throw SeasarRuntimeException.convertSeasarRuntimeException(ex2);
	    } catch (InvocationTargetException ex3) {
            Throwable t = ex3.getTargetException();
            throw SeasarRuntimeException.convertSeasarRuntimeException(t);
        }
    }

    public static Object invoke(final String targetClassName, final String methodName,
            final Object[] args) throws SeasarException {

        return invoke(getClass(targetClassName), methodName, args);
    }

    public static Object invoke(final Class targetClass, final String methodName,
            final Object[] args) throws SeasarException {

        Method m = getMethodByArgs(targetClass, methodName, args);
        try {
            return m.invoke(targetClass, args);
        } catch (Exception ex) {
            throw new SeasarException("ESSR0017", new Object[]{ex}, ex);
        }
    }

    public static Object invoke(final Object target, final String methodName,
            final Object[] args) throws SeasarException {

        Method m = getMethodByArgs(target.getClass(), methodName, args);
        try {
            return m.invoke(target, args);
        } catch (Exception ex) {
            throw new SeasarException("ESSR0017", new Object[]{ex}, ex);
        }
    }

    public static Object invokeNoException(final Object target, final String methodName,
            final Object[] args) {

        try {
            Method m = getMethodByArgs(target.getClass(), methodName, args);
            return m.invoke(target, args);
        } catch (Exception ex) {
            throw new SeasarRuntimeException("ESSR0017", new Object[]{ex}, ex);
        }
    }

    public static Object clone(final Object target) {
        if (target == null) {
            return null;
        }
        Class clazz = target.getClass();
        if (clazz.isArray()) {
            return ArrayUtil.clone((Object[]) target);
        } else {
            try {
                return Reflector.invoke(target, "clone", ArrayUtil.EMPTY_OBJECTS);
            } catch (SeasarException ex) {
                throw ex.convertSeasarRuntimeException();
            }
        }
    }
    
    public static String decapitalizePropertyName(String name) {
		if (StringUtil.isEmpty(name)) {
		    return name;
		}
		if (name.length() > 1 && Character.isUpperCase(name.charAt(1)) &&
				Character.isUpperCase(name.charAt(0))){
					
		    return name;
		}
		char chars[] = name.toCharArray();
		chars[0] = Character.toLowerCase(chars[0]);
		return new String(chars);
    }

    
    public static void setProperties(final Object target, final Properties properties) {
        if (properties == null) {
            return;
        }
        for (Iterator i = properties.keySet().iterator(); i.hasNext(); ) {
            String key = (String) i.next();
            String value = properties.getProperty(key);
            setProperty(target, key, value);
        }
    }

    public static void setProperties(final Object target, final Map properties) {
        if (properties == null) {
            return;
        }
        for (Iterator i = properties.keySet().iterator(); i.hasNext(); ) {
            String key = (String) i.next();
            String value = null;
            Object o = properties.get(key);
            if (o instanceof String) {
            	value = (String) o;
            }
            setProperty(target, key, value);
        }
    }

	public static Method getReadMethod(final Class clazz, final String propertyName) {
        String methodName = "get" + propertyName.substring(0, 1).toUpperCase() + propertyName.substring(1);
        Method m = getMethod(clazz, methodName, ArrayUtil.EMPTY_CLASSES);
        if (m == null) {
            throw new SeasarRuntimeException("ESSR0009", new Object[]{methodName, clazz.getName()});
        }
   		return m;
    }
    
    public static EMap getPropertyDescMap(final Class clazz) {
    	String key = clazz.getName().intern();
        EMap map = (EMap) _propertyDescMapCache.get(key);
        if (map != null) {
            return map;
        }
        synchronized (_propertyDescMapCache) {
        	map = (EMap) _propertyDescMapCache.get(key);
            if (map == null) {
                map = newPropertyDescMap(clazz);
                _propertyDescMapCache.put(key, map);
            }
        }
        return map;
    }
    
    public static EMap newPropertyDescMap(final Class clazz) {
    	EMap map = new EMap();
        Method[] methods = clazz.getMethods();
        for (int i = 0; i < methods.length; i++) {
        	Method m = methods[i];
            String methodName = m.getName();
            if (methodName.startsWith("get")) {
            	if (m.getParameterTypes().length != 0 || methodName.equals("getClass")) {
            		continue;
            	}
            	String propertyName = decapitalizePropertyName(methodName.substring(3)).intern();
            	Class propertyType = m.getReturnType();
            	PropertyDesc propDesc = (PropertyDesc) map.get(propertyName);
            	if (propDesc != null) {
            		if (!propDesc.getPropertyType().equals(propertyType)) {
            			throw new SeasarRuntimeException("ESSR0361", propertyName,
            				propDesc.getPropertyType(), propertyType);
            		}
            		propDesc.setReadMethod(m);
            	} else {
            		propDesc = new PropertyDesc(propertyName, propertyType, m, null);
            		map.put(propertyName, propDesc);
            	}
            } else if (methodName.startsWith("is")) {
            	if (m.getParameterTypes().length != 0 || !m.getReturnType().equals(Boolean.TYPE)) {
            		continue;
            	}
            	String propertyName = decapitalizePropertyName(methodName.substring(2)).intern();
            	Class propertyType = m.getReturnType();
            	PropertyDesc propDesc = (PropertyDesc) map.get(propertyName);
            	if (propDesc != null) {
            		if (!propDesc.getPropertyType().equals(propertyType)) {
            			throw new SeasarRuntimeException("ESSR0361", propertyName,
            				propDesc.getPropertyType(), propertyType);
            		}
            		propDesc.setReadMethod(m);
            	} else {
            		propDesc = new PropertyDesc(propertyName, propertyType, m, null);
            		map.put(propertyName, propDesc);
            	}
            } else if (methodName.startsWith("set")) {
            	if (m.getParameterTypes().length != 1 || methodName.equals("setClass")) {
            		continue;
            	}
            	String propertyName = decapitalizePropertyName(methodName.substring(3)).intern();
            	Class propertyType = m.getParameterTypes()[0];
            	PropertyDesc propDesc = (PropertyDesc) map.get(propertyName);
            	if (propDesc != null) {
            		if (!propDesc.getPropertyType().equals(propertyType)) {
            			throw new SeasarRuntimeException("ESSR0361", propertyName,
            				propDesc.getPropertyType(), propertyType);
            		}
            		propDesc.setWriteMethod(m);
            	} else {
            		propDesc = new PropertyDesc(propertyName, propertyType, null, m);
            		map.put(propertyName, propDesc);
            	}
            }
        }
        return map;
    }
    
    public static Object getProperty(final Object target, final String name) {
        Method m = getReadMethod(target.getClass(), name);
        try {
        	return invoke(m, target, ArrayUtil.EMPTY_OBJECTS);
        } catch (Exception ex) {
        	throw SeasarRuntimeException.convertSeasarRuntimeException(ex);
        }
    }
    
    public static Method getWriteMethod(final Class clazz, final String propertyName) {
        Map setMethodMap = getWriteMethodMap(clazz);
        String methodName = "set" + propertyName.substring(0, 1).toUpperCase() + propertyName.substring(1);
        Method m = (Method) setMethodMap.get(methodName);
        if (m == null) {
            throw new SeasarRuntimeException("ESSR0001", new Object[]{methodName});
        }
        return m;
    }
    
    public static void setProperty(final Object target, final String name, final String value) {
        Method m = getWriteMethod(target.getClass(), name);
        Class[] types = m.getParameterTypes();
        try {
            if (types[0].isPrimitive()) {
                if (types[0] == Integer.TYPE) {
                    m.invoke(target, new Object[]{Integer.valueOf(value)});
                } else if (types[0] == Boolean.TYPE) {
                    m.invoke(target, new Object[]{Boolean.valueOf(value)});
                } else if (types[0] == Long.TYPE) {
                    m.invoke(target, new Object[]{Long.valueOf(value)});
                } else if (types[0] == Short.TYPE) {
                    m.invoke(target, new Object[]{Short.valueOf(value)});
                } else if (types[0] == Double.TYPE) {
                    m.invoke(target, new Object[]{Double.valueOf(value)});
                } else if (types[0] == Float.TYPE) {
                    m.invoke(target, new Object[]{Float.valueOf(value)});
                } else if (types[0] == Character.TYPE) {
                    m.invoke(target, new Object[]{new Character(value.toCharArray()[0])});
                }
            } else {
                m.invoke(target, new Object[]{value});
            }
        } catch (Exception ex) {
            throw SeasarRuntimeException.convertSeasarRuntimeException(ex);
        }
    }
    
    public static void setProperty2(final Object target, final String name, final Object value) {
        Method m = getWriteMethod(target.getClass(), name);
        invokeNoException(m, target, new Object[]{value});
    }

    public static void setFieldValue(final Object target, final String fieldName,
            final Object value) throws SeasarException {

        Field field = getField(target.getClass(), fieldName);
        if (field != null) {
            try {
                field.set(target, value);
            } catch (Exception ex) {
                throw new SeasarException("ESSR0017", new Object[]{ex}, ex);
            }
        } else {
            throw new SeasarException("ESSR0009", new Object[]{fieldName, target.getClass().getName()});
        }
    }
    
    public static Object getFieldValue(final Field field, Object target) {
        try {
            return field.get(target);
        } catch (Exception ex) {
            throw SeasarRuntimeException.convertSeasarRuntimeException(ex);
        }
    }
    
    public static String toStringForBean(Object bean) {
    	try {
	    	StringBuffer buf = new StringBuffer(100);
			BeanInfo beanInfo = Introspector.getBeanInfo(bean.getClass());
			PropertyDescriptor[] propertyDescs = beanInfo.getPropertyDescriptors();
			buf.append("{");
			for (int i = 0; i < propertyDescs.length; ++i) {
				String name = propertyDescs[i].getName();
				Method m = propertyDescs[i].getReadMethod();
				if (m == null || name.equals("class")) {
					continue;
				}
				Object value = invokeNoException(m, bean, ArrayUtil.EMPTY_OBJECTS);
				buf.append(name).append("=").append(value).append(",");
			}
			buf.setLength(buf.length() - 1);
			buf.append("}");
			return buf.toString();
    	} catch (IntrospectionException ex) {
    		throw SeasarRuntimeException.convertSeasarRuntimeException(ex);
    	}
    }

    public Class getTargetClass() {
        return _targetClass;
    }

    public Object getTarget() {
        return _target;
    }

    public Object invoke(final String methodName) throws SeasarException {
        return invoke(methodName, ArrayUtil.EMPTY_OBJECTS);
    }

    public Object invoke(final String methodName, final Object[] args) throws SeasarException {
        Method m = getMethodByArgs(_targetClass, methodName, args);
        try {
            if (Modifier.isStatic(m.getModifiers())) {
                return m.invoke(_targetClass, args);
            } else {
                return m.invoke(_target, args);
            }
        } catch (Exception ex) {
            throw new SeasarException("ESSR0008", new Object[]{methodName, ex}, ex);
        }
    }


    public Object invokeNoException(final String methodName, final Object[] args) {
        try {
            return invoke(methodName, args);
        } catch (SeasarException ex) {
            throw new RuntimeException(ex.toString());
        }
    }

	private static Method getMethodByArgs(final Class clazz, final String methodName,
            Object[] args) {

        if (args == null) {
            args = ArrayUtil.EMPTY_OBJECTS;
        }
        List methods = getMethods(clazz, methodName);
        if (methods.size() == 1) {
        	return (Method) methods.get(0);
        }
        outerLoop :
        for (int i = 0; i < methods.size(); i++) {
            Method m = (Method) methods.get(i);
            Class[] parameterTypes = m.getParameterTypes();
            if (parameterTypes.length != args.length) {
                continue;
            }
            for (int j = 0; j < parameterTypes.length; j++) {
                if (isSameType(parameterTypes[j], args[j])) {
                    continue;
                } else {
                    continue outerLoop;
                }
            }
            return m;
        }
        throw new SeasarRuntimeException("ESSR0009", new Object[]{methodName, clazz.getName()});
    }

    private static Map newMethodsMap(final Class clazz) {
        Map methodsMap = new SMap();
        Method[] methods = clazz.getMethods();
        for (int i = 0; i < methods.length; i++) {
            List list = (List) methodsMap.get(methods[i].getName());
            if (list == null) {
                list = new ArrayList();
                methodsMap.put(methods[i].getName(), list);
            }
            list.add(methods[i]);
        }
        return methodsMap;
    }


    private static Map getWriteMethodMap(final Class clazz) {
        Map writeMethodMap = (Map) _writeMethodMapMap.get(clazz);
        if (writeMethodMap != null) {
            return writeMethodMap;
        }
        synchronized (_writeMethodMapMap) {
            if (writeMethodMap == null) {
                writeMethodMap = newWriteMethodMap(clazz);
                _writeMethodMapMap.put(clazz, writeMethodMap);
            }
        }
        return writeMethodMap;
    }

    private static Map getMethodsMap(final Class clazz) {
        Map methodsMap = (Map) _methodsMapMap.get(clazz);
        if (methodsMap != null) {
            return methodsMap;
        }
        synchronized (_methodsMapMap) {
        	methodsMap = (Map) _methodsMapMap.get(clazz);
            if (methodsMap == null) {
                methodsMap = newMethodsMap(clazz);
                _methodsMapMap.put(clazz, methodsMap);
            }
        }
        return methodsMap;
    }

    private static Map newWriteMethodMap(final Class clazz) {
        Method[] methods = clazz.getMethods();
        Map setMethodMap = new SMap();
        for (int i = 0; i < methods.length; i++) {
            String methodName = methods[i].getName();
            if (methodName.startsWith("set") && methods[i].getParameterTypes().length == 1) {
                setMethodMap.put(methodName, methods[i]);
            }
        }
        return setMethodMap;
    }
}