package org.seasar.eclipse;

import java.io.File;
import java.io.IOException;
import java.util.MissingResourceException;
import java.util.Properties;
import java.util.ResourceBundle;

import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IPluginDescriptor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Status;
import org.eclipse.jdt.internal.ui.util.ExceptionHandler;
import org.eclipse.ui.plugin.AbstractUIPlugin;

public final class SeasarPlugin extends AbstractUIPlugin {

	public static final String PLUGIN_ID = "org.seasar.eclipse" ; 
	public static final String SEASAR_HOME_KEY = "SEASAR_HOME";
	private static final String DEFAULT_SEASAR_HOME = "c:\\seasar";
	private static final String[] NEED_JAR_NAMES = new String[] {
		"seasar.jar", "sql.jar", "servlet.jar", "jta.jar", "log4j.jar", "jmxri.jar", "hsqldb.jar"
	};
	private static final String[] NEED_FILE_NAMES = new String[] {
		"message-config.properties", "nazuna-config.xml", "seasar-context.xml"
	};
	private static final String APPENDERS = "DEBUG, C, R";
	private static final String REPLACE_APPENDERS = "DEBUG, C";
	private static SeasarPlugin _plugin;
	private ResourceBundle _resourceBundle;

	public SeasarPlugin(IPluginDescriptor descriptor) {
		super(descriptor);
		_plugin = this;
		getPreferenceStore().setDefault(SEASAR_HOME_KEY, DEFAULT_SEASAR_HOME);
		try {
			_resourceBundle= ResourceBundle.getBundle("org.seasar.eclipse.resources");
		} catch (MissingResourceException x) {
			_resourceBundle = null;
		}
		getWorkspace().addResourceChangeListener(
			new SeasarProjectDeleteChangeListener(), IResourceChangeEvent.PRE_DELETE);

	}

	public static SeasarPlugin getDefault() {
		return _plugin;
	}

	public static IWorkspace getWorkspace() {
		return ResourcesPlugin.getWorkspace();
	}

	public static String getResourceString(String key) {
		ResourceBundle bundle= SeasarPlugin.getDefault().getResourceBundle();
		try {
			return bundle.getString(key);
		} catch (MissingResourceException e) {
			return key;
		}
	}

	public ResourceBundle getResourceBundle() {
		return _resourceBundle;
	}
	
	public String getSeasarHome() {
		return getPreferenceStore().getString(SEASAR_HOME_KEY);
	}
	
	public void setSeasarHome(String seasarHome) {
		getPreferenceStore().setValue(SEASAR_HOME_KEY, seasarHome);
	}
	
	public IPath getSeasarHomePath() {
		return new Path(getSeasarHome());
	}
	
	public String getDefaultSeasarHome() {
		return getPreferenceStore().getDefaultString(SEASAR_HOME_KEY);
	}
	
	public String getSeasarLib() {
		return getSeasarHome() + File.separator + "lib";
	}
	
	public IPath getSeasarLibPath() {
		return getSeasarHomePath().append("lib");
	}
	
	public File getSeasarLibDir() {
		return new File(getSeasarLib());
	}
	
	public String getSeasarClasses() {
		return getSeasarHome() + File.separator + "classes";
	}
	
	public File getSeasarClassesDir() {
		return new File(getSeasarClasses());
	}
	
	public String getSeasarSrc() {
		return getSeasarHome() + File.separator + "src";
	}

	public File getSeasarSrcDir() {
		return new File(getSeasarSrc());
	}
	
	public IPath getSeasarSrcPath() {
		return getSeasarHomePath().append("src");
	}
	
	public String getJettyXML() {
		return getSeasarClasses() + File.separator + "jetty.xml";
	}
	
	public void copySeasarLibJarsToDir(File toDir) throws IOException {
		File libDir = getSeasarLibDir();
		for (int i = 0; i < NEED_JAR_NAMES.length; ++i) {
			File fromFile = new File(libDir, NEED_JAR_NAMES[i]);
			File toFile = new File(toDir, NEED_JAR_NAMES[i]);
			if (fromFile.exists() && !toFile.exists()) {
				FileUtil.copyFile(fromFile, toFile);  
			}
		}
	}
	
	public void copySeasarClassesFilesToDir(File toDir) throws IOException {
		File classesDir = getSeasarClassesDir();
		for (int i = 0; i < NEED_FILE_NAMES.length; ++i) {
			File fromFile = new File(classesDir, NEED_FILE_NAMES[i]);
			File toFile = new File(toDir, NEED_FILE_NAMES[i]);
			if (fromFile.exists() && !toFile.exists()) {
				FileUtil.copyFile(fromFile, toFile);  
			}
		}
	}
	
	public void copyWebXMLToDir(File toDir) throws IOException {
		String webXML = "org" + File.separator + "seasar" +
			File.separator + "nazuna" + File.separator + "amf" +
			File.separator + "web.xml";
		File fromFile = new File(getSeasarSrcDir(), webXML);
		File toFile = new File(toDir, "web.xml");
		if (fromFile.exists() && !toFile.exists()) {
			FileUtil.copyFile(fromFile, toFile);  
		}
	}
	
	public void copyLog4jPropertiesToDir(File toDir) throws IOException {
		Properties log4jProperties = new Properties();
		File file = new File(getSeasarClassesDir(), "log4j.properties");
		String text = FileUtil.readText(file);
		StringBuffer buf = new StringBuffer(text.length());
		int pos = text.indexOf(APPENDERS);
		if (pos < 0) {
			return;
		}
		buf.append(text.substring(0, pos));
		buf.append(REPLACE_APPENDERS);
		buf.append(text.substring(pos + APPENDERS.length()));
		File toFile = new File(toDir, "log4j.properties");
		if (!toFile.exists()) {
			FileUtil.writeText(toFile, buf.toString());  
		}
	}
	
    public static IStatus getErrorStatus(String message) {
        return new Status(IStatus.ERROR, _plugin.getDescriptor().getUniqueIdentifier(), IStatus.ERROR, message, null);
    }

    public static IStatus getErrorStatus(Throwable t) {
        return getErrorStatus(t.toString());
    }

    public static CoreException getCoreException(String message) {
        return new CoreException(getErrorStatus(message));
    }

    public static CoreException getCoreException(Throwable t) {
        return new CoreException(getErrorStatus(t));
    }

    public static void handleException(Throwable t, String title) {
        ExceptionHandler.handle(getCoreException(t), title, t.toString());
    }

    public static void handleException(Throwable t) {
        handleException(t, getResourceString("plugin.name"));
    }

    public static void handleException(CoreException ex) {
        ExceptionHandler.handle(ex, getResourceString("plugin.name"), ex.getMessage());
    }
    
    public static void handleException(String title, String message) {
        ExceptionHandler.handle(getCoreException(message), title, message);
    }

    public static void handleException(Throwable t, String title, String message) {
        ExceptionHandler.handle(getCoreException(t), title, message);
    }

    public static void handleException(CoreException ex, String title, String message) {
        ExceptionHandler.handle(ex, title, message);
    }

    public static void handleException(String message) {
        handleException(getResourceString("plugin.name"), message);
    }

}
