package org.seasar.system;

import java.net.Socket;
import java.util.ArrayList;
import java.util.List;

import org.seasar.log.Logger;
import org.seasar.util.ElementHandler;
import org.seasar.util.Reflector;
import org.seasar.util.ResourceUtil;
import org.seasar.util.SeasarException;
import org.seasar.util.ThreadUtil;
import org.seasar.util.XMLHandler;
import org.seasar.util.XMLHandlerParser;
import org.seasar.util.XMLHandlerRule;
import org.xml.sax.Attributes;

public final class Seasar implements SeasarMBean {

	private static final String XML_FILE_NAME = "/seasar-config.xml";
    private static Seasar _instance = new Seasar();
    private static Logger _logger = Logger.getLogger(Seasar.class);
    private static List _services;
    private static XMLHandlerRule _xmlHandlerRule = new XMLHandlerRule();
    
    private boolean _started = false;
    private Thread _shutdownThread;
    private boolean _needExit = false;
    
    static {
    	configure();
    }
    
    private Seasar() {
    }
    
    public static void configure() {
    	if (ResourceUtil.isExist(XML_FILE_NAME)) {
            setupXMLHandlerRule();
            XMLHandlerParser.parse(XML_FILE_NAME, _xmlHandlerRule);
        }
    }

    public static void main(final String[] args) {
        if (args.length == 0) {
            usage();
            System.exit(1);
        } else if (args[0].equals("-start")) {
            startForMain();
        } else if (args[0].equals("-shutdown")) {
            shutdownForMain();
        } else if (args[0].equals("-restart")) {
        	if (args.length == 1) {
        		restartForMain();
        	} else {
				restartServiceForMain(args[1]);
        	}
		} else {
            usage();
            System.exit(1);
        }
    }

    public static Seasar getInstance() {
        return _instance;
    }

	public static String getHome() {
		return System.getProperty("seasar.home", "..");
	}

    public synchronized void start() throws SeasarException {
        startServices();
        JMXService.registerMBean(this, NAME);
        addShutdownHook();
        _started = true;
        _logger.log("ISSR0001", null);
    }

    public synchronized void stop() {
        stopServices();
        _started = false;
        _logger.log("ISSR0002", null);
        if (_shutdownThread != null) {
            Runtime.getRuntime().removeShutdownHook(_shutdownThread);
            _shutdownThread = null;
        }
        System.runFinalization();
        System.gc();
    }
    
    public boolean isStarted() {
    	return _started;
    }

    public void shutdown() {
        Runtime.getRuntime().removeShutdownHook(_shutdownThread);
        _needExit = true;
        _shutdownThread.start();
    }
    
	public void restartService(String className) {
		for (int i = 0; i < _services.size(); ++i) {
			Lifecycle service = (Lifecycle) _services.get(i);
			if (service.getClass().getName().equals(className)) {
				try {
					service.stop();
					service.start();
					Logger.getLogger(getClass()).log("ISSR0005", new Object[]{className});
				} catch (Throwable t) {
					Logger.getLogger(getClass()).log(t);
					_services.remove(i);
				}
				break;
			}
		}
	}

	private static boolean isRunning() {
        Socket s = null;
        try {
            s = new Socket("localhost", RMIAdaptorService.getInstance().getPort());
            s.close();
            return true;
        } catch (Exception ex) {
            return false;
        }
    }
    
    private static void usage() {
        _logger.error("Usage:java org.seasar.system.Seasar -start|-shutdown|-restart serviceClassName");
    }

    private static void setupXMLHandlerRule() {
    	_xmlHandlerRule.addElementHandler("/seasar/services",
            new ElementHandler() {
                public void start(XMLHandler xmlHandler, Attributes attributes) {
                    _services = new ArrayList();
                }
            });
        _xmlHandlerRule.addElementHandler("/seasar/services/service",
            new ElementHandler() {
                public void start(XMLHandler xmlHandler, Attributes attributes) {
                    String className = attributes.getValue("className");
                    Lifecycle service = (Lifecycle) Reflector.newInstance(className);
                    _services.add(service);
                    xmlHandler.push(service);
                }
                public void end(XMLHandler xmlHandler, String body) {
                    xmlHandler.pop();
                }
            });
    	_xmlHandlerRule.addElementHandler("/seasar/services/service/properties/property",
            new ElementHandler() {
                public void start(XMLHandler xmlHandler, Attributes attributes) {
                    String name = attributes.getValue("name");
                    String value = attributes.getValue("value");
                    Object service = xmlHandler.peek();
                    Reflector.setProperty(service, name, value);
                }
            });
    }

    private void startServices() throws SeasarException {
        for (int i = 0; i < _services.size(); i++) {
            try {
                ((Lifecycle) _services.get(i)).start();
            } catch (Exception ex) {
                for (int j = i - 1; j >= 0; j--) {
                    try {
                        ((Lifecycle) _services.get(j)).stop();
                    } catch (Exception ex2) {
                        _logger.log(ex2);
                    }
                }
                _services.clear();
                throw SeasarException.convertSeasarException(ex);
            }
        }
    }
    
    private void stopServices() {
        for (int i = _services.size() - 1; i >= 0; i--) {
            try {
                ((Lifecycle) _services.get(i)).stop();
            } catch (Throwable t) {
                _logger.log(t);
            }
        }
    }   

    private void addShutdownHook() throws SeasarException {
        _shutdownThread =
            new Thread() {
                public void run() {
                    if (_shutdownThread == null) {
                        return;
                    }
                    _shutdownThread = null;
                    Seasar.this.stop();
                    _logger.log("ISSR0003", null);
                    if (_needExit) {
                        System.exit(0);
                    }
                }
            };
        _shutdownThread.setDaemon(true);
        Runtime.getRuntime().addShutdownHook(_shutdownThread);
    }

    private static void startForMain() {
        try {
            if (isRunning()) {
                _logger.log("ESSR0301", null);
                System.exit(1);
            }
            _instance.start();
            synchronized (_instance) {
                _instance.wait();
            }
        } catch (Throwable t) {
            _logger.log(t);
            System.exit(1);
        }
    }

    private static void shutdownForMain() {
        try {
            if (!isRunning()) {
                _logger.log("ESSR0302", null);
                System.exit(1);
            }
            SeasarMBean seasar = (SeasarMBean) MBeanProxy.create(
            	SeasarMBean.class, NAME, "localhost", RMIAdaptorService.getInstance().getPort());
            seasar.shutdown();
            System.exit(0);
        } catch (Throwable t) {
            _logger.log(t);
            System.exit(1);
        }
    }
    
	private static void restartForMain() {
		try {
			if (isRunning()) {
				SeasarMBean seasar = (SeasarMBean) MBeanProxy.create(
					SeasarMBean.class, NAME, "localhost", RMIAdaptorService.getInstance().getPort());
				seasar.shutdown();
			}
			while (isRunning()) {
				ThreadUtil.sleep(100);
			}
			startForMain();
		} catch (Throwable t) {
			_logger.log(t);
			System.exit(1);
		}
	}
    
	private static void restartServiceForMain(String className) {
		try {
			if (!isRunning()) {
				_logger.log("ESSR0302", null);
				System.exit(1);
			}
			SeasarMBean seasar = (SeasarMBean) MBeanProxy.create(
				SeasarMBean.class, NAME, "localhost", RMIAdaptorService.getInstance().getPort());
			seasar.restartService(className);
			System.exit(0);
		} catch (Throwable t) {
			_logger.log(t);
			System.exit(1);
		}
	}
}