package org.seasar.nazuna;

import java.lang.reflect.Method;
import java.math.BigDecimal;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Timestamp;
import java.sql.Types;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.seasar.log.Logger;
import org.seasar.util.Conversion;
import org.seasar.util.Reflector;
import org.seasar.util.SeasarContext;
import org.seasar.util.SeasarException;
import org.seasar.util.StringUtil;

public final class Sqlet extends RuleBase {
	
	private static Logger _logger = Logger.getLogger(Sqlet.class);
	private static Method _executeQueryRealMethod = Reflector.getMethod(Sqlet.class,
		"executeQueryReal", new Class[]{String.class, List.class});
	private static Method _executeUpdateRealMethod = Reflector.getMethod(Sqlet.class,
		"executeUpdateReal", new Class[]{String.class, List.class});
	private SeasarContext _seasarContext;
	private ResultType _resultType;
	
	public Sqlet(String contextName, String resultPath) {
		_seasarContext = SeasarContext.getInstance(contextName);
		if (!StringUtil.isEmpty(resultPath)) {
			_resultType = ResultTypeFactory.getResultType(resultPath);
		}
	}
	
	public static void close(ResultSet rs, Statement stmt, Connection con)
			throws SeasarException {
				
        SQLException se = null;
        if (rs != null) {
            try {
                rs.close();
            } catch (SQLException ex) {
                se = ex;
                _logger.log(ex);
            }
        }
        if (stmt != null) {
            try {
                stmt.close();
            } catch (SQLException ex) {
                se = ex;
                _logger.log(ex);
            }
        }
        if (con != null) {
            try {
                con.close();
            } catch (SQLException ex) {
                se = ex;
                _logger.log(ex);
            }
        }
        if (se != null) {
            throw SeasarException.convertSeasarException(se);
        }
    }
    
    public static void setBindVariable(
    		PreparedStatement ps, int index, Object value)
            throws SeasarException {

        try {
            if (value != null) {
                if (value instanceof String) {
                    ps.setString(index, (String) value);
                } else if (value instanceof Integer) {
                    ps.setInt(index, ((Integer) value).intValue());
                } else if (value instanceof BigDecimal) {
                    ps.setBigDecimal(index, (BigDecimal) value);
                } else if (value instanceof Long) {
                    ps.setLong(index, ((Long) value).longValue());
                } else if (value instanceof Timestamp) {
                    ps.setTimestamp(index, (Timestamp) value);
                } else if (value instanceof java.util.Date) {
                    ps.setTimestamp(index, Conversion.toTimestamp(value, null));
                } else {
                    ps.setString(index, value.toString());
                }
            } else {
                ps.setNull(index, Types.VARCHAR);
            }
        } catch (SQLException ex) {
            throw SeasarException.convertSeasarException(ex);
        }
    }
    
    public static String getBindVariableText(Object bindVariable){
	    if (bindVariable instanceof String) {
	    	return "'" + bindVariable + "'";
	    } else if (bindVariable instanceof Number) {
	    	return bindVariable.toString();
	    } else if (bindVariable instanceof Timestamp) {
	    	SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd hh.mm.ss");
	    	return "'" + sdf.format((java.util.Date) bindVariable) + "'";
	    } else if (bindVariable instanceof java.util.Date) {
	    	SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
	    	return "'" + sdf.format((java.util.Date) bindVariable) + "'";
	    } else if (bindVariable == null) {
	    	return "null";
	    } else {
	    	return "'" + bindVariable.toString() + "'";
	    }	
  	}

	public static String getCompleteText(String text, List bindVariables){
	  	if (bindVariables == null || bindVariables.size() == 0) {
	  		return text;
	  	}
	    StringBuffer buf = new StringBuffer(200);
	    int pos = 0;
		int pos2 = 0;
		int index = 0;
		while (true) {
			pos = text.indexOf('?', pos2);
			if (pos > 0) {
				buf.append(text.substring(pos2, pos));
				buf.append(getBindVariableText(bindVariables.get(index++)));
				pos2 = pos + 1;
			} else {
				buf.append(text.substring(pos2));
				break;
			}
		}
		return buf.toString();
	}
	
	public static PreparedStatement getPreparedStatement(final Connection con,
            final String text, final List bindVariables) throws SeasarException {

        try {
        	if (_logger.isDebugEnabled()) {
        		_logger.debug(getCompleteText(text, bindVariables));
        	}
            PreparedStatement ps = con.prepareStatement(text);
            if (bindVariables != null) {
                for (int i = 0; i < bindVariables.size(); i++) {
                    setBindVariable(ps, i + 1, bindVariables.get(i));
                }
            }
            return ps;
        } catch (SQLException ex) {
            throw new SeasarException("ESSR0008", new Object[]{
            	getCompleteText(text, bindVariables), ex}, ex);
        }
    }
	
	public ResultType getResultType() {
		return _resultType;
	}
	
	public List executeQuery() throws SeasarException {
		return executeQuery(null);
	}
	
	public List executeQuery(Map parameters) throws SeasarException {
		RuleContext ruleContext = execute(parameters);
		String text = ruleContext.getText();
		List bindVariables = ruleContext.getBindVariables();
		return executeQuery(text, bindVariables);
	}
	
	public List executeQuery(String text, List bindVariables) throws SeasarException {
		return (List) getTransAttribute().invoke(
			_executeQueryRealMethod, this, new Object[]{text, bindVariables});
	}
	
	public List executeQueryReal(String text, List bindVariables)
			throws SeasarException {
				
		Connection con = _seasarContext.getConnection();
		PreparedStatement ps = null;
		ResultSet rs = null;
		try {
			ps = getPreparedStatement(con, text, bindVariables);
			rs = ps.executeQuery();
			return _resultType.fetch(rs);
		} catch (SQLException ex) {
			throw SeasarException.convertSeasarException(ex);
		} finally {
			close(rs, ps, con);
		}
	}
	
	public NzRecordSet executeRSQuery() throws SeasarException {
		return executeRSQuery(null);
	}
	
	public NzRecordSet executeRSQuery(Map parameters) throws SeasarException {
		RuleContext ruleContext = execute(parameters);
		String text = ruleContext.getText();
		List bindVariables = ruleContext.getBindVariables();
		List rows = executeQuery(text, bindVariables);
		return new NzRecordSet(_resultType.getPropertyNames(), rows);
	}
	
	public int executeUpdate() throws SeasarException {
		return executeUpdate(null);
	}
	
	public int executeUpdate(Map parameters) throws SeasarException {
		RuleContext ruleContext = execute(parameters);
		String text = ruleContext.getText();
		List bindVariables = ruleContext.getBindVariables();
		return executeUpdate(text, bindVariables);
	}
	
	public int executeUpdate(String text, List bindVariables) throws SeasarException {
		return ((Integer) getTransAttribute().invoke(
			_executeUpdateRealMethod, this, new Object[]{text, bindVariables})).intValue();
	}
	
	public int executeUpdateReal(String text, List bindVariables)
				throws SeasarException {
				
		Connection con = _seasarContext.getConnection();
		PreparedStatement ps = null;
		try {
			ps = getPreparedStatement(con, text, bindVariables);
			return ps.executeUpdate();
		} catch (SQLException ex) {
			throw SeasarException.convertSeasarException(ex);
		} finally {
			close(null, ps, con);
		}
	}
	
	public RuleContext execute() throws SeasarException {
		return execute(null);
	}
	
	public RuleContext execute(Map parameters) throws SeasarException {
		RuleContext ctx = new RuleContextImpl(this, parameters);
		for (int i = 0; i < _ruleStatements.length; ++i) {
			RuleStatement ruleStatement = _ruleStatements[i];
			ruleStatement.execute(ctx);
			if (ctx.isThrowed()) {
				ctx.throwSeasarExcepton();
			}
		}
		return ctx;
	}
	
	public List fetch(ResultSet rs) throws SeasarException {
		if (Map.class.isAssignableFrom(_resultType.getResultClass())) {
			return fetchForMap(rs);
		} else {
			return fetchForBean(rs);
		}
	}
	
	public List fetchForMap(ResultSet rs) {
		return null;
	}
	
	public List fetchForBean(ResultSet rs) throws SeasarException {
		List results = new ArrayList();
		try {
			Object bean = Reflector.newInstance(_resultType.getResultClass());
			while (rs.next()) {
				results.add(_resultType.fetchForBean(rs));
			}
		} catch (SQLException ex) {
			throw SeasarException.convertSeasarException(ex);
		}
		return results;
	}
}