package test.org.seasar.naming;

import junit.framework.*;
import java.util.*;
import javax.naming.*;
import org.seasar.naming.*;
import org.seasar.util.*;

public class NamingServerTest extends TestCase {

    private NamingServer _root;

    public NamingServerTest(String name) {
        super(name);
    }

    public void testSerialize() throws Exception {
        Serializer.serialize(_root);
    }

    public void testBind() throws Exception {
        try {
            _root.bind(NamingParser.getInstance().parse(""), "test");
            fail("1");
        } catch (InvalidNameException ex) {
            System.out.println(ex);
        }

        try {
            _root.bind(NamingParser.getInstance().parse("/"), "test");
            fail("2");
        } catch (InvalidNameException ex) {
            System.out.println(ex);
        }

        try {
            _root.bind(NamingParser.getInstance().parse("test/"), "test");
            fail("3");
        } catch (InvalidNameException ex) {
            System.out.println(ex);
        }

        String s = "hello";
        Name n = NamingParser.getInstance().parse("hello");
        _root.bind(n, s);
        assertEquals("4", s, _root.lookup(n));

        Name n2 = NamingParser.getInstance().parse("test/hello");
        _root.bind(n2, s);
        assertEquals("5", s, _root.lookup(n2));

        try {
            _root.bind(n2, s);
            fail("6");
        } catch (NameAlreadyBoundException ex) {
            System.out.println(ex);
        }
    }

    public void testRebind() throws Exception {
        try {
            _root.rebind(NamingParser.getInstance().parse(""), "test");
            fail("1");
        } catch (InvalidNameException ex) {
            System.out.println(ex);
        }

        try {
            _root.rebind(NamingParser.getInstance().parse("/"), "test");
            fail("2");
        } catch (InvalidNameException ex) {
            System.out.println(ex);
        }

        try {
            _root.rebind(NamingParser.getInstance().parse("test/"), "test");
            fail("3");
        } catch (InvalidNameException ex) {
            System.out.println(ex);
        }

        String s = "hello";
        Name n = NamingParser.getInstance().parse("hello");
        _root.rebind(n, s);
        assertEquals("4", s, _root.lookup(n));

        Name n2 = NamingParser.getInstance().parse("test/hello");
        _root.rebind(n2, s);
        assertEquals("5", s, _root.lookup(n2));

        _root.rebind(n2, s);
        assertEquals("6", s, _root.lookup(n2));
    }

    public void testUnbind() throws Exception {
        try {
            _root.unbind(NamingParser.getInstance().parse(""));
            fail("1");
        } catch (InvalidNameException ex) {
            System.out.println(ex);
        }

        try {
            _root.unbind(NamingParser.getInstance().parse("/"));
            fail("2");
        } catch (InvalidNameException ex) {
            System.out.println(ex);
        }

        String s = "hello";
        Name n = NamingParser.getInstance().parse("hello");
        _root.bind(n, s);
        _root.unbind(n);

        try {
            _root.unbind(n);
            fail("3");
        } catch (javax.naming.NameNotFoundException ex) {
            System.out.println(ex);
        }

        Name n2 = NamingParser.getInstance().parse("test/hello");
        _root.bind(n2, s);
        _root.unbind(n2);

        try {
            _root.unbind(n2);
            fail("4");
        } catch (javax.naming.NameNotFoundException ex) {
            System.out.println(ex);
        }
    }

    public void testLookup() throws Exception {
        String s = "hello";
        Name n = NamingParser.getInstance().parse("test/hello");
        _root.bind(n, s);
        assertEquals("1", s, _root.lookup(n));

        Name n2 = NamingParser.getInstance().parse("test/");
        assertEquals("2", NamingContext.class, _root.lookup(n2).getClass());
    }

    public void testGetNameMap() throws Exception {
        String s = "hello";
        Name n = NamingParser.getInstance().parse("test/hello");
        Name n2 = NamingParser.getInstance().parse("hello");
        _root.bind(n, s);
        _root.bind(n2, s);
        Map nm = _root.getNameMap(NamingParser.getInstance().parse(""));
        assertEquals("1", 2, nm.size());
        assertEquals("2", s, nm.get("hello"));
        assertEquals("3", NamingContext.class, nm.get("test").getClass());
    }

    public void testCreateSubcontext() throws Exception {
        Name n = NamingParser.getInstance().parse("test");
        _root.createSubcontext(n);
        assertEquals("1", NamingContext.class, _root.lookup(n).getClass());

        Name n2 = NamingParser.getInstance().parse("test2/test3");
        _root.createSubcontext(n2);
        assertEquals("2", NamingContext.class, _root.lookup(n2).getClass());
    }

    public void testDestroySubcontext() throws Exception {
        String s = "hello";
        Name n = NamingParser.getInstance().parse("test/hello");
        Name n2 = NamingParser.getInstance().parse("hello");
        Name n3 = NamingParser.getInstance().parse("");
        Name n4 = NamingParser.getInstance().parse("test");
        _root.bind(n, s);
        _root.bind(n2, s);
        _root.destroySubcontext(n3);
        Map nm = _root.getNameMap(n3);
        assertEquals("1", 0, nm.size());

        _root.bind(n, s);
        _root.bind(n2, s);
        _root.destroySubcontext(n4);
        Map nm2 = _root.getNameMap(n3);
        assertEquals("2", 1, nm2.size());
    }

    protected void setUp() throws Exception {
        _root = new NamingServer();
    }

    protected void tearDown() throws Exception {
        _root = null;
    }

    public static Test suite() {
        return new TestSuite(NamingServerTest.class);
    }

    public static void main(String[] args) {
        junit.textui.TestRunner.main(new String[]{NamingServerTest.class.getName()});
    }
}
