package org.seasar.expr;

import java.util.ArrayList;
import java.util.List;

import org.seasar.util.SeasarException;

public class ExprStatementParser {

	private transient ExprStatementTokenizer _tokenizer;
	private int _token;

	public ExprStatementParser(String s) throws SeasarException {
		_tokenizer = new ExprStatementTokenizer(s);
		nextToken();
	}

	public final BooleanExpression parseBooleanExpression()
		throws SeasarException {

		BooleanExpression boolExp = getOrExp();
		expect(ExprStatementTokenizer.EOF);
		return boolExp;
	}

	public final Expression parseExpression() throws SeasarException {
		Expression exp = getAddSubExp();
		expect(ExprStatementTokenizer.EOF);
		return exp;
	}

	private BooleanExpression getOrExp() throws SeasarException {
		BooleanExpression left = getAndExp();
		while (_token == ExprStatementTokenizer.OR) {
			nextToken();
			left = new OrExp(left, getAndExp());
		}
		return left;
	}

	private BooleanExpression getAndExp() throws SeasarException {
		BooleanExpression left = getNotExp();
		while (_token == ExprStatementTokenizer.AND) {
			nextToken();
			left = new AndExp(left, getNotExp());
		}
		return left;
	}

	private BooleanExpression getNotExp() throws SeasarException {
		switch (_token) {
			case ExprStatementTokenizer.NOT :
				nextToken();
				BooleanExpression e = getSimpleBoolExp();
				return new NotExp(e);
			default :
				return getSimpleBoolExp();
		}
	}

	private BooleanExpression getSimpleBoolExp() throws SeasarException {
		switch (_token) {
			case ExprStatementTokenizer.LEFT_PAREN :
				nextToken();
				BooleanExpression e = getOrExp();
				expect(ExprStatementTokenizer.RIGHT_PAREN);
				return e;
			case ExprStatementTokenizer.TRUE :
				nextToken();
				if (_token == ExprStatementTokenizer.IS) {
					nextToken();
					switch (_token) {
						case ExprStatementTokenizer.TRUE :
							nextToken();
							return BooleanExp.TRUE;
						case ExprStatementTokenizer.FALSE :
							nextToken();
							return BooleanExp.FALSE;
					}
				} else {
					return BooleanExp.TRUE;
				}
			case ExprStatementTokenizer.FALSE :
				nextToken();
				if (_token == ExprStatementTokenizer.IS) {
					nextToken();
					switch (_token) {
						case ExprStatementTokenizer.TRUE :
							nextToken();
							return BooleanExp.FALSE;
						case ExprStatementTokenizer.FALSE :
							nextToken();
							return BooleanExp.TRUE;
					}
				} else {
					return BooleanExp.FALSE;
				}
			default :
				Expression targetExp = getAddSubExp();
				switch (_token) {
					case ExprStatementTokenizer.EQUAL :
						nextToken();
						return new EqualExp(targetExp, getAddSubExp());
					case ExprStatementTokenizer.NOT_EQUAL :
						nextToken();
						return new NotEqualExp(targetExp, getAddSubExp());
					case ExprStatementTokenizer.GREATER_EQUAL :
						nextToken();
						return new GreaterEqualExp(targetExp, getAddSubExp());
					case ExprStatementTokenizer.GREATER_THAN :
						nextToken();
						return new GreaterThanExp(targetExp, getAddSubExp());
					case ExprStatementTokenizer.LESS_EQUAL :
						nextToken();
						return new LessEqualExp(targetExp, getAddSubExp());
					case ExprStatementTokenizer.LESS_THAN :
						nextToken();
						return new LessThanExp(targetExp, getAddSubExp());
					case ExprStatementTokenizer.IN :
						return getInExp(targetExp);
					case ExprStatementTokenizer.BETWEEN :
						return getBetweenExp(targetExp);
					case ExprStatementTokenizer.IS :
						nextToken();
						switch (_token) {
							case ExprStatementTokenizer.NULL :
								nextToken();
								return new IsNullExp(targetExp);
							case ExprStatementTokenizer.NOT :
								nextToken();
								expect(ExprStatementTokenizer.NULL);
								return new IsNotNullExp(targetExp);
							case ExprStatementTokenizer.TRUE :
								nextToken();
								return new IsTrueExp(targetExp);
							case ExprStatementTokenizer.FALSE :
								nextToken();
								return new IsFalseExp(targetExp);
						}
					case ExprStatementTokenizer.NOT :
						nextToken();
						switch (_token) {
							case ExprStatementTokenizer.LIKE :
								nextToken();
								return new NotExp(
									new LikeExp(targetExp, getAddSubExp()));
							case ExprStatementTokenizer.BETWEEN :
								return getNotBetweenExp(targetExp);
							case ExprStatementTokenizer.IN :
								return new NotExp(getInExp(targetExp));
						}
					case ExprStatementTokenizer.LIKE :
						nextToken();
						return new LikeExp(targetExp, getAddSubExp());
					case ExprStatementTokenizer.MATCH :
						nextToken();
						return new MatchExp(targetExp, getAddSubExp());
					case ExprStatementTokenizer.MATCH_IGNORE_CASE :
						nextToken();
						return new MatchExp(targetExp, getAddSubExp(), true);
					case ExprStatementTokenizer.NOT_MATCH :
						nextToken();
						return new NotExp(
							new MatchExp(targetExp, getAddSubExp()));
					case ExprStatementTokenizer.NOT_MATCH_IGNORE_CASE :
						nextToken();
						return new NotExp(
							new MatchExp(targetExp, getAddSubExp(), true));
				}
		}
		throw new SeasarException(
			"ESSR0032",
			new Object[] { ExprStatementTokenizer.getTokenName(_token)});
	}

	private BooleanExpression getInExp(Expression targetExp)
		throws SeasarException {
		nextToken();
		expect(ExprStatementTokenizer.LEFT_PAREN);
		List inExpList = new ArrayList();
		inExpList.add(getAddSubExp());
		while (_token == ExprStatementTokenizer.COMMA) {
			nextToken();
			inExpList.add(getAddSubExp());
		}
		expect(ExprStatementTokenizer.RIGHT_PAREN);
		return new InExp(targetExp, ExprUtil.toExpressionArray(inExpList));
	}

	private BooleanExpression getBetweenExp(Expression targetExp)
		throws SeasarException {
		nextToken();
		Expression fromExp = getAddSubExp();
		expect(ExprStatementTokenizer.AND);
		Expression toExp = getAddSubExp();
		return new BetweenExp(targetExp, fromExp, toExp);
	}

	private BooleanExpression getNotBetweenExp(Expression targetExp)
		throws SeasarException {
		nextToken();
		Expression fromExp = getAddSubExp();
		expect(ExprStatementTokenizer.AND);
		Expression toExp = getAddSubExp();
		return new NotBetweenExp(targetExp, fromExp, toExp);
	}

	private Expression getAddSubExp() throws SeasarException {
		Expression arg1Exp = getMultDivModExp();
		while (true) {
			switch (_token) {
				case ExprStatementTokenizer.ADD :
					nextToken();
					arg1Exp = new AddExp(arg1Exp, getMultDivModExp());
					break;
				case ExprStatementTokenizer.SUBTRACT :
					nextToken();
					arg1Exp = new SubtractExp(arg1Exp, getMultDivModExp());
					break;
				default :
					return arg1Exp;
			}
		}
	}

	private Expression getMultDivModExp() throws SeasarException {
		Expression arg1Exp = getConcatenateExp();
		while (true) {
			switch (_token) {
				case ExprStatementTokenizer.MULTIPLY :
					nextToken();
					arg1Exp = new MultiplyExp(arg1Exp, getConcatenateExp());
					break;
				case ExprStatementTokenizer.DIVIDE :
					nextToken();
					arg1Exp = new DivideExp(arg1Exp, getConcatenateExp());
					break;
				case ExprStatementTokenizer.MOD :
					nextToken();
					arg1Exp = new ModExp(arg1Exp, getConcatenateExp());
					break;
				default :
					return arg1Exp;
			}
		}
	}

	private Expression getConcatenateExp() throws SeasarException {
		Expression exp = getSimpleExp();
		if (_token != ExprStatementTokenizer.CONCATENATE) {
			return exp;
		}
		List exps = new ArrayList();
		exps.add(exp);
		while (_token == ExprStatementTokenizer.CONCATENATE) {
			nextToken();
			exps.add(getSimpleExp());
		}
		return new ConcatenateExp(ExprUtil.toExpressionArray(exps));
	}

	private Expression getSimpleExp() throws SeasarException {
		switch (_token) {
			case ExprStatementTokenizer.LEFT_PAREN :
				nextToken();
				Expression e = getAddSubExp();
				expect(ExprStatementTokenizer.RIGHT_PAREN);
				return e;
			case ExprStatementTokenizer.DOUBLE :
				return getDoubleExp();
			case ExprStatementTokenizer.LONG :
				return getLongExp();
			case ExprStatementTokenizer.INTEGER :
				return getIntegerExp();
			case ExprStatementTokenizer.QUOTED_STRING :
				return getStringExp();
			case ExprStatementTokenizer.WORD :
				return getWordExp();
			case ExprStatementTokenizer.TO_TIMESTAMP :
				return getToTimestampExp();
			case ExprStatementTokenizer.TO_STRING :
				return getToStringExp();
			case ExprStatementTokenizer.TO_BIGDECIMAL :
				return getToBigDecimalExp();
			case ExprStatementTokenizer.TO_DOUBLE :
				return getToDoubleExp();
			case ExprStatementTokenizer.TO_LONG :
				return getToLongExp();
			case ExprStatementTokenizer.TO_INTEGER :
				return getToIntegerExp();
			case ExprStatementTokenizer.TO_BOOLEAN :
				return getToBooleanExp();
			case ExprStatementTokenizer.NOW :
				return getNowExp();
			case ExprStatementTokenizer.NULL :
				return getNullExp();
			case ExprStatementTokenizer.SUBSTRING :
				return getSubstringExp();
			case ExprStatementTokenizer.POSITION :
				return getPositionExp();
			case ExprStatementTokenizer.CASE :
				return getCaseExp();
			case ExprStatementTokenizer.COALESCE :
				return getCoalesceExp();
			case ExprStatementTokenizer.NVL :
				return getNvlExp();
			case ExprStatementTokenizer.NULLIF :
				return getNullifExp();
			case ExprStatementTokenizer.TRIM :
				return getTrimExp();
			case ExprStatementTokenizer.NEW :
				return getNewExp();
			case ExprStatementTokenizer.TRUE :
				nextToken();
				return new BooleanExp(true);
			case ExprStatementTokenizer.FALSE :
				nextToken();
				return new BooleanExp(false);
			default :
				throw new SeasarException(
					"ESSR0032",
					new Object[] {
						 ExprStatementTokenizer.getTokenName(_token)});
		}
	}

	private Expression getDoubleExp() throws SeasarException {
		Expression e = new DoubleExp(_tokenizer.getDouble());
		nextToken();
		return e;
	}

	private Expression getLongExp() throws SeasarException {
		Expression e = new LongExp(_tokenizer.getLong());
		nextToken();
		return e;
	}

	private Expression getIntegerExp() throws SeasarException {
		Expression e = new IntegerExp(_tokenizer.getInteger());
		nextToken();
		return e;
	}

	private Expression getStringExp() throws SeasarException {
		String s = _tokenizer.getString();
		nextToken();
		if (_token == ExprStatementTokenizer.QUOTED_STRING) {
			s = s + "'" + _tokenizer.getString();
			nextToken();
		}
		return new StringExp(s);
	}

	private Expression getToTimestampExp() throws SeasarException {
		nextToken();
		expect(ExprStatementTokenizer.LEFT_PAREN);
		Expression p1 = getSimpleExp();
		Expression p2 = null;
		if (_token == ExprStatementTokenizer.COMMA) {
			nextToken();
			p2 = getStringExp();
		}
		String pattern = null;
		if (p2 != null) {
			pattern = (String) p2.evaluateValue(null);
		}
		expect(ExprStatementTokenizer.RIGHT_PAREN);
		return new ToTimestampExp(p1, pattern);
	}

	private Expression getToStringExp() throws SeasarException {
		nextToken();
		expect(ExprStatementTokenizer.LEFT_PAREN);
		Expression p1 = getAddSubExp();
		Expression p2 = null;
		if (_token == ExprStatementTokenizer.COMMA) {
			nextToken();
			p2 = getStringExp();
		}
		String pattern = null;
		if (p2 != null) {
			pattern = (String) p2.evaluateValue(null);
		}
		expect(ExprStatementTokenizer.RIGHT_PAREN);
		return new ToStringExp(p1, pattern);
	}

	private Expression getToBigDecimalExp() throws SeasarException {
		nextToken();
		expect(ExprStatementTokenizer.LEFT_PAREN);
		Expression p1 = getAddSubExp();
		Expression p2 = null;
		if (_token == ExprStatementTokenizer.COMMA) {
			nextToken();
			p2 = getStringExp();
		}
		String pattern = null;
		if (p2 != null) {
			pattern = (String) p2.evaluateValue(null);
		}
		expect(ExprStatementTokenizer.RIGHT_PAREN);
		return new ToBigDecimalExp(p1, pattern);
	}

	private Expression getToDoubleExp() throws SeasarException {
		nextToken();
		expect(ExprStatementTokenizer.LEFT_PAREN);
		Expression p1 = getAddSubExp();
		Expression p2 = null;
		if (_token == ExprStatementTokenizer.COMMA) {
			nextToken();
			p2 = getStringExp();
		}
		String pattern = null;
		if (p2 != null) {
			pattern = (String) p2.evaluateValue(null);
		}
		expect(ExprStatementTokenizer.RIGHT_PAREN);
		return new ToDoubleExp(p1, pattern);
	}

	private Expression getToLongExp() throws SeasarException {
		nextToken();
		expect(ExprStatementTokenizer.LEFT_PAREN);
		Expression p1 = getAddSubExp();
		Expression p2 = null;
		if (_token == ExprStatementTokenizer.COMMA) {
			nextToken();
			p2 = getStringExp();
		}
		String pattern = null;
		if (p2 != null) {
			pattern = (String) p2.evaluateValue(null);
		}
		expect(ExprStatementTokenizer.RIGHT_PAREN);
		return new ToLongExp(p1, pattern);
	}

	private Expression getToIntegerExp() throws SeasarException {
		nextToken();
		expect(ExprStatementTokenizer.LEFT_PAREN);
		Expression p1 = getAddSubExp();
		Expression p2 = null;
		if (_token == ExprStatementTokenizer.COMMA) {
			nextToken();
			p2 = getStringExp();
		}
		String pattern = null;
		if (p2 != null) {
			pattern = (String) p2.evaluateValue(null);
		}
		expect(ExprStatementTokenizer.RIGHT_PAREN);
		return new ToIntegerExp(p1, pattern);
	}

	private Expression getToBooleanExp() throws SeasarException {
		nextToken();
		expect(ExprStatementTokenizer.LEFT_PAREN);
		BooleanExpression argExp = getOrExp();
		expect(ExprStatementTokenizer.RIGHT_PAREN);
		return new ToBooleanExp(argExp);
	}

	private Expression getNowExp() throws SeasarException {
		nextToken();
		expect(ExprStatementTokenizer.LEFT_PAREN);
		expect(ExprStatementTokenizer.RIGHT_PAREN);
		return NowExp.NOW;
	}

	private Expression getNullExp() throws SeasarException {
		nextToken();
		return NullExp.NULL;
	}

	private Expression getSubstringExp() throws SeasarException {
		nextToken();
		expect(ExprStatementTokenizer.LEFT_PAREN);
		List argExps = new ArrayList();
		Expression arg1Exp = getAddSubExp();
		argExps.add(arg1Exp);
		expect(ExprStatementTokenizer.FROM);
		Expression fromExp = getAddSubExp();
		argExps.add(fromExp);
		Expression forExp = null;
		if (_token == ExprStatementTokenizer.FOR) {
			nextToken();
			forExp = getAddSubExp();
			argExps.add(forExp);
		}
		expect(ExprStatementTokenizer.RIGHT_PAREN);
		return new MethodExp(
			"org.seasar.util.StringUtil",
			"substr",
			ExprUtil.toExpressionArray(argExps));
	}

	private Expression getPositionExp() throws SeasarException {
		nextToken();
		expect(ExprStatementTokenizer.LEFT_PAREN);
		Expression argExp = getAddSubExp();
		expect(ExprStatementTokenizer.IN);
		Expression strExp = getAddSubExp();
		expect(ExprStatementTokenizer.RIGHT_PAREN);
		List argExps = new ArrayList();
		argExps.add(strExp);
		argExps.add(argExp);
		return new MethodExp(
			"org.seasar.util.StringUtil",
			"strpos",
			ExprUtil.toExpressionArray(argExps));
	}

	private Expression getCaseExp() throws SeasarException {
		nextToken();
		List boolExpList = new ArrayList();
		List expList = new ArrayList();
		if (_token == ExprStatementTokenizer.WHEN) {
			while (_token == ExprStatementTokenizer.WHEN) {
				nextToken();
				boolExpList.add(getOrExp());
				expect(ExprStatementTokenizer.THEN);
				expList.add(getAddSubExp());
			}
		} else {
			Expression targetExp = getAddSubExp();
			while (_token == ExprStatementTokenizer.WHEN) {
				nextToken();
				boolExpList.add(new EqualExp(targetExp, getAddSubExp()));
				expect(ExprStatementTokenizer.THEN);
				expList.add(getAddSubExp());
			}
		}
		if (_token == ExprStatementTokenizer.ELSE) {
			nextToken();
			boolExpList.add(new BooleanExp(true));
			expList.add(getAddSubExp());
		}
		expect(ExprStatementTokenizer.END);
		BooleanExpression[] boolExps =
			ExprUtil.toBooleanExpressionArray(boolExpList);
		Expression[] exps = ExprUtil.toExpressionArray(expList);
		return new CaseExp(boolExps, exps);
	}

	private Expression getCoalesceExp() throws SeasarException {
		nextToken();
		expect(ExprStatementTokenizer.LEFT_PAREN);
		List expList = new ArrayList();
		expList.add(getAddSubExp());
		while (_token == ExprStatementTokenizer.COMMA) {
			nextToken();
			expList.add(getAddSubExp());
		}
		expect(ExprStatementTokenizer.RIGHT_PAREN);
		return new CoalesceExp(ExprUtil.toExpressionArray(expList));
	}

	private Expression getNvlExp() throws SeasarException {
		nextToken();
		expect(ExprStatementTokenizer.LEFT_PAREN);
		Expression arg1Exp = getAddSubExp();
		expect(ExprStatementTokenizer.COMMA);
		Expression arg2Exp = getAddSubExp();
		expect(ExprStatementTokenizer.RIGHT_PAREN);
		return new NvlExp(arg1Exp, arg2Exp);
	}

	private Expression getNullifExp() throws SeasarException {
		nextToken();
		expect(ExprStatementTokenizer.LEFT_PAREN);
		Expression arg1Exp = getAddSubExp();
		expect(ExprStatementTokenizer.COMMA);
		Expression arg2Exp = getAddSubExp();
		expect(ExprStatementTokenizer.RIGHT_PAREN);
		return new NullifExp(arg1Exp, arg2Exp);
	}

	private Expression getTrimExp() throws SeasarException {
		nextToken();
		expect(ExprStatementTokenizer.LEFT_PAREN);
		String methodName = "btrim";
		switch (_token) {
			case ExprStatementTokenizer.LEADING :
				methodName = "ltrim";
				nextToken();
				break;
			case ExprStatementTokenizer.TRAILING :
				methodName = "rtrim";
				nextToken();
				break;
			case ExprStatementTokenizer.BOTH :
				nextToken();
				break;
		}
		Expression trimStrExp = new StringExp(" ");
		if (_token != ExprStatementTokenizer.FROM) {
			trimStrExp = getAddSubExp();
		}
		expect(ExprStatementTokenizer.FROM);
		Expression strExp = getAddSubExp();
		expect(ExprStatementTokenizer.RIGHT_PAREN);
		List argExps = new ArrayList();
		argExps.add(strExp);
		argExps.add(trimStrExp);
		return new MethodExp(
			"org.seasar.util.StringUtil",
			methodName,
			ExprUtil.toExpressionArray(argExps));
	}

	private Expression getWordExp() throws SeasarException {
		String s = _tokenizer.getString();
		nextToken();
		if (_token != ExprStatementTokenizer.LEFT_PAREN) {
			return getVariableExp(s);
		} else {
			return getMethodExp(s);
		}
	}

	private Expression getVariableExp(String name) throws SeasarException {
		if (_token == ExprStatementTokenizer.LEFT_BRACKET) {
			nextToken();
			Expression indexExp = getAddSubExp();
			expect(ExprStatementTokenizer.RIGHT_BRACKET);
			if (_token == ExprStatementTokenizer.ASSIGN) {
				nextToken();
				return new SetArrayVariableExp(name, indexExp, getAddSubExp());
			} else {
				return new GetArrayVariableExp(name, indexExp);
			}
		} else {
			if (_token == ExprStatementTokenizer.ASSIGN) {
				nextToken();
				return new SetVariableExp(name, getAddSubExp());
			} else {
				return new GetVariableExp(name);
			}
		}
	}

	private Expression getMethodExp(String s) throws SeasarException {
		nextToken();
		String targetName = null;
		String methodName = null;
		AliasConfig aliasConfig = AliasConfigManager.resolveAlias(s);
		if (aliasConfig != null) {
			targetName = aliasConfig.getClassName();
			methodName = aliasConfig.getMethodName();
		} else {
			int pos = s.lastIndexOf('.');
			if (pos < 0) {
				throw new SeasarException("ESSR0001", s);
			}
			targetName = s.substring(0, pos);
			methodName = s.substring(pos + 1);
		}
		List argExps = new ArrayList();
		if (_token != ExprStatementTokenizer.RIGHT_PAREN) {
			argExps.add(getAddSubExp());
			while (_token == ExprStatementTokenizer.COMMA) {
				nextToken();
				argExps.add(getAddSubExp());
			}
		}
		expect(ExprStatementTokenizer.RIGHT_PAREN);
		return new MethodExp(
			targetName,
			methodName,
			ExprUtil.toExpressionArray(argExps));
	}

	private Expression getNewExp() throws SeasarException {
		nextToken();
		String className = _tokenizer.getString();
		nextToken();
		if (_token == ExprStatementTokenizer.LEFT_PAREN) {
			nextToken();
			List argExpList = new ArrayList();
			if (_token != ExprStatementTokenizer.RIGHT_PAREN) {
				argExpList.add(getAddSubExp());
				while (_token == ExprStatementTokenizer.COMMA) {
					nextToken();
					argExpList.add(getAddSubExp());
				}
			}
			expect(ExprStatementTokenizer.RIGHT_PAREN);
			return new NewExp(
				ExprUtil.getClass(className),
				ExprUtil.toExpressionArray(argExpList));
		} else {
			expect(ExprStatementTokenizer.LEFT_BRACKET);
			int arraySize = 0;
			if (_token == ExprStatementTokenizer.INTEGER) {
				arraySize = _tokenizer.getInteger().intValue();
				nextToken();
			}
			expect(ExprStatementTokenizer.RIGHT_BRACKET);
			List argExpList = new ArrayList();
			if (_token == ExprStatementTokenizer.LEFT_BRACE) {
				nextToken();
				argExpList.add(getAddSubExp());
				while (_token == ExprStatementTokenizer.COMMA) {
					nextToken();
					argExpList.add(getAddSubExp());
				}
				expect(ExprStatementTokenizer.RIGHT_BRACE);
				if (arraySize < argExpList.size()) {
					arraySize = argExpList.size();
				}
			}
			return new NewArrayExp(
				ExprUtil.getClass(className),
				arraySize,
				ExprUtil.toExpressionArray(argExpList));
		}
	}

	private void nextToken() throws SeasarException {
		_token = _tokenizer.nextToken();
	}

	public void expect(int t) throws SeasarException {
		ExprStatementTokenizer.assertToken(t, _token);
		nextToken();
	}
}
