package org.seasar.expr;

import java.util.Map;

import org.seasar.util.SMap;
import org.seasar.util.SeasarException;
import org.seasar.util.SeasarRuntimeException;

public final class ExprStatementTokenizer {

	public final static int INVALID_CHAR = -1;
	public final static int OR = 1;
	public final static int AND = 2;
	public final static int NOT = 3;
	public final static int WORD = 4;
	public final static int QUOTED_STRING = 5;
	public final static int INTEGER = 6;
	public final static int LONG = 7;
	public final static int DOUBLE = 8;
	public final static int LEFT_PAREN = 9;
	public final static int RIGHT_PAREN = 10;
	public final static int ADD = 11;
	public final static int SUBTRACT = 12;
	public final static int MULTIPLY = 13;
	public final static int DIVIDE = 14;
	public final static int MOD = 15;
	public final static int EQUAL = 16;
	public final static int NOT_EQUAL = 17;
	public final static int GREATER_THAN = 18;
	public final static int GREATER_EQUAL = 19;
	public final static int LESS_THAN = 20;
	public final static int LESS_EQUAL = 21;
	public final static int COMMA = 22;
	public final static int NEW = 23;
	public final static int FROM = 24;
	public final static int IN = 25;
	public final static int BETWEEN = 26;
	public final static int TO_TIMESTAMP = 27;
	public final static int TO_STRING = 28;
	public final static int TO_BIGDECIMAL = 29;
	public final static int TO_LONG = 30;
	public final static int TO_INTEGER = 31;
	public final static int FOR = 32;
	public final static int ASSIGN = 33;
	public final static int NOW = 34;
	public final static int TRUE = 35;
	public final static int FALSE = 36;
	public final static int CASE = 37;
	public final static int IS = 38;
	public final static int NULL = 39;
	public final static int LIKE = 40;
	public final static int COALESCE = 41;
	public final static int CONCATENATE = 42;
	public final static int NULLIF = 43;
	public final static int NVL = 44;
	public final static int TRIM = 45;
	public final static int TO_BOOLEAN = 46;
	public final static int MATCH = 47;
	public final static int MATCH_IGNORE_CASE = 48;
	public final static int LEFT_BRACKET = 49;
	public final static int RIGHT_BRACKET = 50;
	public final static int LEFT_BRACE = 51;
	public final static int RIGHT_BRACE = 52;
	public final static int LEADING = 53;
	public final static int TRAILING = 54;
	public final static int BOTH = 55;
	public final static int POSITION = 56;
	public final static int SUBSTRING = 57;
	public final static int WHEN = 58;
	public final static int THEN = 59;
	public final static int ELSE = 60;
	public final static int END = 61;
	public final static int NOT_MATCH = 62;
	public final static int NOT_MATCH_IGNORE_CASE = 63;
	public final static int TO_DOUBLE = 64;
	public final static int ASC = 65;
	public final static int DESC = 66;
	public final static int COUNT = 67;
	public final static int SUM = 68;
	public final static int MAX = 69;
	public final static int MIN = 70;
	public final static int AVG = 71;
	public final static int AS = 72;
	public final static int EOF = 99;

	private static Map _wordMap = new SMap();
	private transient ExprCoreTokenizer _tokenizer;

	static {
		setupWord();
	}

	public ExprStatementTokenizer(String s) {
		_tokenizer = new ExprCoreTokenizer(s);
	}

	public static void assertToken(int expected, int real) {
		if (expected != real) {
			throw new SeasarRuntimeException(
				"ESSR0002",
				new Object[] {
					"token",
					getTokenName(expected),
					getTokenName(real)});
		}
	}

	public int nextToken() throws SeasarException {
		try {
			switch (_tokenizer.nextToken()) {
				case ExprCoreTokenizer.TT_EOF :
					return EOF;
				case ExprCoreTokenizer.TT_WORD :
					Integer type =
						(Integer) _wordMap.get(_tokenizer.sval.toLowerCase());
					if (type != null) {
						return type.intValue();
					} else {
						return WORD;
					}
				case ExprCoreTokenizer.TT_INTEGER :
					return INTEGER;
				case ExprCoreTokenizer.TT_LONG :
					return LONG;
				case ExprCoreTokenizer.TT_DOUBLE :
					return DOUBLE;
				case '\'' :
					return QUOTED_STRING;
				case '(' :
					return LEFT_PAREN;
				case ')' :
					return RIGHT_PAREN;
				case '+' :
					return ADD;
				case '-' :
					return SUBTRACT;
				case '/' :
					return DIVIDE;
				case '%' :
					return MOD;
				case ',' :
					return COMMA;
				case '[' :
					return LEFT_BRACKET;
				case ']' :
					return RIGHT_BRACKET;
				case '{' :
					return LEFT_BRACE;
				case '}' :
					return RIGHT_BRACE;
				default :
					throw new SeasarException(
						"ESSR0313",
						new Object[] { String.valueOf(_tokenizer.ttype)});
			}
		} catch (Exception ex) {
			throw SeasarException.convertSeasarException(ex);
		}
	}

	public int nextToken(int t) throws SeasarException {
		int next = nextToken();
		assertToken(t, next);
		return next;
	}

	public String getString() {
		return _tokenizer.sval;
	}

	public Integer getInteger() {
		return _tokenizer.ival;
	}

	public Long getLong() {
		return _tokenizer.lval;
	}

	public Double getDouble() {
		return _tokenizer.dval;
	}

	public String toString() {
		return _tokenizer.toString();
	}

	public static String getTokenName(int t) {
		switch (t) {
			case ADD :
				return "ADD";
			case SUBTRACT :
				return "SUBTRACT";
			case MULTIPLY :
				return "MULTIPLY";
			case DIVIDE :
				return "DIVIDE";
			case MOD :
				return "MOD";
			case LEFT_PAREN :
				return "LEFT_PAREN";
			case RIGHT_PAREN :
				return "RIGHT_PAREN";
			case AND :
				return "AND";
			case OR :
				return "OR";
			case NOT :
				return "NOT";
			case QUOTED_STRING :
				return "QUOTED_STRING";
			case EOF :
				return "EOF";
			case WORD :
				return "WORD";
			case INTEGER :
				return "INTEGER";
			case LONG :
				return "LONG";
			case DOUBLE :
				return "DOUBLE";
			case TO_TIMESTAMP :
				return "TO_TIMESTAMP";
			case TO_STRING :
				return "TO_STRING";
			case TO_BIGDECIMAL :
				return "TO_BIGDECIMAL";
			case TO_DOUBLE :
				return "TO_DOUBLE";
			case TO_LONG :
				return "TO_LONG";
			case TO_INTEGER :
				return "TO_INTEGER";
			case TO_BOOLEAN :
				return "TO_BOOLEAN";
			case NOW :
				return "NOW";
			case EQUAL :
				return "EQUAL";
			case NOT_EQUAL :
				return "NOT_EQUAL";
			case GREATER_THAN :
				return "GREATER_THAN";
			case GREATER_EQUAL :
				return "GREATER_EQUAL";
			case LESS_THAN :
				return "LESS_THAN";
			case LESS_EQUAL :
				return "LESS_EQUAL";
			case COMMA :
				return "COMMA";
			case SUBSTRING :
				return "SUBSTRING";
			case IN :
				return "IN";
			case BETWEEN :
				return "BETWEEN";
			case TRUE :
				return "TRUE";
			case FALSE :
				return "FALSE";
			case CASE :
				return "CASE";
			case IS :
				return "IS";
			case NULL :
				return "NULL";
			case LIKE :
				return "LIKE";
			case COALESCE :
				return "COALESCE";
			case POSITION :
				return "POSITION";
			case FROM :
				return "FROM";
			case FOR :
				return "FOR";
			case CONCATENATE :
				return "CONCATENATE";
			case NVL :
				return "NVL";
			case NULLIF :
				return "NULLIF";
			case TRIM :
				return "TRIM";
			case MATCH :
				return "MATCH";
			case MATCH_IGNORE_CASE :
				return "MATCH_IGNORE_CASE";
			case NOT_MATCH :
				return "NOT_MATCH";
			case NOT_MATCH_IGNORE_CASE :
				return "NOT_MATCH_IGNORE_CASE";
			case LEADING :
				return "LEADING";
			case TRAILING :
				return "TRAILING";
			case BOTH :
				return "BOTH";
			case WHEN :
				return "WHEN";
			case THEN :
				return "THEN";
			case ELSE :
				return "ELSE";
			case END :
				return "END";
			case NEW :
				return "NEW";
			case ASSIGN :
				return "ASSIGN";
			case ASC :
				return "ASC";
			case DESC :
				return "DESC";
			case COUNT :
				return "COUNT";
			case SUM :
				return "SUM";
			case MAX :
				return "MAX";
			case MIN :
				return "MIN";
			case AVG :
				return "AVG";
			case AS :
				return "AS";
			default :
				throw new SeasarRuntimeException(
					"ESSR0027",
					new Object[] { String.valueOf(t)});
		}
	}

	private static void setupWord() {
		_wordMap.put("or", new Integer(OR));
		_wordMap.put("and", new Integer(AND));
		_wordMap.put("not", new Integer(NOT));
		_wordMap.put("=", new Integer(EQUAL));
		_wordMap.put("eq", new Integer(EQUAL));
		_wordMap.put("!=", new Integer(NOT_EQUAL));
		_wordMap.put("<>", new Integer(NOT_EQUAL));
		_wordMap.put("ne", new Integer(NOT_EQUAL));
		_wordMap.put(">", new Integer(GREATER_THAN));
		_wordMap.put("gt", new Integer(GREATER_THAN));
		_wordMap.put(">=", new Integer(GREATER_EQUAL));
		_wordMap.put("ge", new Integer(GREATER_EQUAL));
		_wordMap.put("<", new Integer(LESS_THAN));
		_wordMap.put("lt", new Integer(LESS_THAN));
		_wordMap.put("<=", new Integer(LESS_EQUAL));
		_wordMap.put("le", new Integer(LESS_EQUAL));
		_wordMap.put(":=", new Integer(ASSIGN));
		_wordMap.put("||", new Integer(CONCATENATE));
		_wordMap.put("to_timestamp", new Integer(TO_TIMESTAMP));
		_wordMap.put("to_string", new Integer(TO_STRING));
		_wordMap.put("to_bigdecimal", new Integer(TO_BIGDECIMAL));
		_wordMap.put("to_double", new Integer(TO_DOUBLE));
		_wordMap.put("to_long", new Integer(TO_LONG));
		_wordMap.put("to_integer", new Integer(TO_INTEGER));
		_wordMap.put("to_boolean", new Integer(TO_BOOLEAN));
		_wordMap.put("in", new Integer(IN));
		_wordMap.put("between", new Integer(BETWEEN));
		_wordMap.put("true", new Integer(TRUE));
		_wordMap.put("false", new Integer(FALSE));
		_wordMap.put("case", new Integer(CASE));
		_wordMap.put("is", new Integer(IS));
		_wordMap.put("null", new Integer(NULL));
		_wordMap.put("like", new Integer(LIKE));
		_wordMap.put("coalesce", new Integer(COALESCE));
		_wordMap.put("from", new Integer(FROM));
		_wordMap.put("for", new Integer(FOR));
		_wordMap.put("now", new Integer(NOW));
		_wordMap.put("nvl", new Integer(NVL));
		_wordMap.put("nullif", new Integer(NULLIF));
		_wordMap.put("trim", new Integer(TRIM));
		_wordMap.put("leading", new Integer(LEADING));
		_wordMap.put("trailing", new Integer(TRAILING));
		_wordMap.put("both", new Integer(BOTH));
		_wordMap.put("position", new Integer(POSITION));
		_wordMap.put("substring", new Integer(SUBSTRING));
		_wordMap.put("when", new Integer(WHEN));
		_wordMap.put("then", new Integer(THEN));
		_wordMap.put("else", new Integer(ELSE));
		_wordMap.put("end", new Integer(END));
		_wordMap.put("new", new Integer(NEW));
		_wordMap.put("*", new Integer(MULTIPLY));
		_wordMap.put("~", new Integer(MATCH));
		_wordMap.put("~*", new Integer(MATCH_IGNORE_CASE));
		_wordMap.put("!~", new Integer(NOT_MATCH));
		_wordMap.put("!~*", new Integer(NOT_MATCH_IGNORE_CASE));
		_wordMap.put("asc", new Integer(ASC));
		_wordMap.put("desc", new Integer(DESC));
		_wordMap.put("count", new Integer(COUNT));
		_wordMap.put("sum", new Integer(SUM));
		_wordMap.put("max", new Integer(MAX));
		_wordMap.put("min", new Integer(MIN));
		_wordMap.put("avg", new Integer(AVG));
		_wordMap.put("as", new Integer(AS));
	}
}