package org.seasar.expr;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.List;

import org.seasar.util.Assertion;
import org.seasar.util.Reflector;
import org.seasar.util.SeasarException;
import org.seasar.util.StringUtil;

public final class MethodExp implements Expression {

	private final String _targetName;
	private Class _targetClass;
	private String[] _names;
	private final String _methodName;
	private final Expression[] _exps;

	public MethodExp(
		final String targetName,
		final String methodName,
		final Expression[] exps) {

		Assertion.assertNotNull("targetName", targetName);
		Assertion.assertNotNull("methodName", methodName);
		Assertion.assertNotNull("exps", exps);

		_targetName = targetName;
		if (ExprUtil.isClassName(targetName)) {
			_targetClass = Reflector.getClass(targetName);
		} else {
			_names = StringUtil.split(targetName, ".");
		}
		_methodName = methodName;
		_exps = exps;
	}

	public Object evaluateValue(final ExprContext ruleContext)
		throws SeasarException {

		Object[] args = ExprUtil.evaluateValues(_exps, ruleContext);
		if (_targetClass != null) {
			return invoke(_targetClass, null, args);
		} else {
			Object target = ruleContext.getValue(_names[0]);
			for (int i = 1; i < _names.length; ++i) {
				target = ExprUtil.getProperty(target, _names[i]);
			}
			return invoke(target.getClass(), target, args);
		}
	}

	private Object invoke(Class clazz, Object target, Object[] args)
		throws SeasarException {

		List methods = Reflector.getMethods(clazz, _methodName);
		return invoke(methods, target, args);
	}

	private Object invoke(List methods, Object target, Object[] args)
		throws SeasarException {

		outerLoop : for (int i = 0; i < methods.size(); ++i) {
			Method method = (Method) methods.get(i);
			Class[] paramTypes = method.getParameterTypes();
			if (paramTypes.length != args.length) {
				continue;
			}
			for (int j = 0; j < args.length; ++j) {
				if (args[j] == null
					|| ExprUtil.isTargetClass(
						paramTypes[j],
						args[j].getClass())) {
					continue;
				}
				continue outerLoop;
			}
			return invoke(method, target, args);
		}
		throw new SeasarException("ESSR0001", new Object[] { _methodName });
	}

	private Object invoke(Method method, Object target, Object[] args)
		throws SeasarException {

		try {
			return method.invoke(target, args);
		} catch (InvocationTargetException ex) {
			throw SeasarException.convertSeasarException(ex.getCause());
		} catch (IllegalAccessException ex) {
			throw SeasarException.convertSeasarException(ex);
		}
	}
}
