package test.org.seasar.expr;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;

import org.seasar.expr.ExprCoreTokenizer;

public class ExprCoreTokenizerTest extends TestCase {

    public ExprCoreTokenizerTest(String name) {
        super(name);
    }

    public void testEOF() throws Exception {
    	ExprCoreTokenizer tokenizer = new ExprCoreTokenizer("");
    	assertEquals("1", ExprCoreTokenizer.TT_EOF, tokenizer.nextToken());
    	assertEquals("2", ExprCoreTokenizer.TT_EOF, tokenizer.nextToken());
    }
    
    public void testWhitespace() throws Exception {
    	ExprCoreTokenizer tokenizer = new ExprCoreTokenizer("\t       \n");
    	assertEquals("1", ExprCoreTokenizer.TT_EOF, tokenizer.nextToken());
    }
    
    public void testHyphen() throws Exception {
    	ExprCoreTokenizer tokenizer = new ExprCoreTokenizer("       - ");
    	assertEquals("1", '-', tokenizer.nextToken());
    	assertEquals("2", ExprCoreTokenizer.TT_EOF, tokenizer.nextToken());
    }
    
    public void testInteger() throws Exception {
    	ExprCoreTokenizer tokenizer = new ExprCoreTokenizer("       123");
    	assertEquals("1", ExprCoreTokenizer.TT_INTEGER, tokenizer.nextToken());
    	assertEquals("2", new Integer(123), tokenizer.ival);
    	assertEquals("3", ExprCoreTokenizer.TT_EOF, tokenizer.nextToken());
    	
    	ExprCoreTokenizer tokenizer2 = new ExprCoreTokenizer("    -1234567890");
    	assertEquals("4", ExprCoreTokenizer.TT_INTEGER, tokenizer2.nextToken());
    	assertEquals("5", new Integer(-1234567890), tokenizer2.ival);
    	
    	ExprCoreTokenizer tokenizer3 = new ExprCoreTokenizer(String.valueOf(Integer.MAX_VALUE));
    	assertEquals("6", ExprCoreTokenizer.TT_INTEGER, tokenizer3.nextToken());
    	assertEquals("7", new Integer(Integer.MAX_VALUE), tokenizer3.ival);
    	
    	ExprCoreTokenizer tokenizer4 = new ExprCoreTokenizer(String.valueOf(Integer.MIN_VALUE));
    	assertEquals("8", ExprCoreTokenizer.TT_INTEGER, tokenizer4.nextToken());
    	assertEquals("9", new Integer(Integer.MIN_VALUE), tokenizer4.ival);
    }
    
    public void testLong() throws Exception {
    	ExprCoreTokenizer tokenizer = new ExprCoreTokenizer("       12345678901");
    	assertEquals("1", ExprCoreTokenizer.TT_LONG, tokenizer.nextToken());
    	assertEquals("2", new Long("12345678901"), tokenizer.lval);
    	assertEquals("3", ExprCoreTokenizer.TT_EOF, tokenizer.nextToken());
    }
    
    public void testDouble() throws Exception {
    	ExprCoreTokenizer tokenizer = new ExprCoreTokenizer("       123.45 ");
    	assertEquals("1", ExprCoreTokenizer.TT_DOUBLE, tokenizer.nextToken());
    	assertEquals("2", new Double("123.45"), tokenizer.dval);
    	assertEquals("3", ExprCoreTokenizer.TT_EOF, tokenizer.nextToken());
    }
    
    public void testOrdinary() throws Exception {
    	ExprCoreTokenizer tokenizer = new ExprCoreTokenizer("(1 + 2)");
    	assertEquals("1", '(', tokenizer.nextToken());
    	assertEquals("2", ExprCoreTokenizer.TT_INTEGER, tokenizer.nextToken());
    	assertEquals("3", new Integer(1), tokenizer.ival);
    	assertEquals("4", '+', tokenizer.nextToken());
    	assertEquals("5", ExprCoreTokenizer.TT_INTEGER, tokenizer.nextToken());
    	assertEquals("6", new Integer(2), tokenizer.ival);
    	assertEquals("7", ')', tokenizer.nextToken());
    	assertEquals("8", ExprCoreTokenizer.TT_EOF, tokenizer.nextToken());
    }
    
    public void testWord() throws Exception {
    	ExprCoreTokenizer tokenizer = new ExprCoreTokenizer("123aaa45");
    	assertEquals("1", ExprCoreTokenizer.TT_INTEGER, tokenizer.nextToken());
    	assertEquals("2", new Integer(123), tokenizer.ival);
    	assertEquals("3", ExprCoreTokenizer.TT_WORD, tokenizer.nextToken());
    	assertEquals("4", "aaa45", tokenizer.sval);
    	assertEquals("5", ExprCoreTokenizer.TT_EOF, tokenizer.nextToken());
    }
    
    public void testQuote() throws Exception {
    	ExprCoreTokenizer tokenizer = new ExprCoreTokenizer("'\n123''aaa'aa");
    	assertEquals("1", '\'', tokenizer.nextToken());
    	assertEquals("2", "\n123'aaa", tokenizer.sval);
    	assertEquals("3", ExprCoreTokenizer.TT_WORD, tokenizer.nextToken());
    	assertEquals("4", "aa", tokenizer.sval);
    	assertEquals("5", ExprCoreTokenizer.TT_EOF, tokenizer.nextToken());
    }
    
    public void testQuote2() throws Exception {
    	ExprCoreTokenizer tokenizer = new ExprCoreTokenizer("out.println('a\b')");
    	tokenizer.nextToken();
    	tokenizer.nextToken();
    	assertEquals("1", '\'', tokenizer.nextToken());
    	assertEquals("2", "a\b", tokenizer.sval);
    	tokenizer.nextToken();
    	assertEquals("3", ExprCoreTokenizer.TT_EOF, tokenizer.nextToken());
    }
    
    public void testAtMark() throws Exception {
    	ExprCoreTokenizer tokenizer = new ExprCoreTokenizer("@aaa");
    	assertEquals("1", ExprCoreTokenizer.TT_WORD, tokenizer.nextToken());
    	assertEquals("2", "@aaa", tokenizer.sval);
    	assertEquals("3", ExprCoreTokenizer.TT_EOF, tokenizer.nextToken());
    }

    protected void setUp() throws Exception {
    }

    protected void tearDown() throws Exception {
    }

    public static Test suite ( ) {
        return new TestSuite(ExprCoreTokenizerTest.class);
    }

    public static void main (String[] args) {
        junit.textui.TestRunner.main(new String[]{ExprCoreTokenizerTest.class.getName()});
    }
}