package test.org.seasar.expr;

import java.math.BigDecimal;
import java.util.HashMap;
import java.util.Map;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;

import org.seasar.expr.BooleanExpression;
import org.seasar.expr.ExprContext;
import org.seasar.expr.ExprContextImpl;
import org.seasar.expr.ExprStatementParser;
import org.seasar.expr.Expression;
import org.seasar.util.Conversion;

public class ExprStatementParserTest extends TestCase {

	public ExprStatementParserTest(String name) {
		super(name);
	}

	public void testOrExp() throws Exception {
		BooleanExpression boolExp =
			new ExprStatementParser("true or false").parseBooleanExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testOrExp2() throws Exception {
		BooleanExpression boolExp =
			new ExprStatementParser("(true or false) and false")
				.parseBooleanExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testOrExp3() throws Exception {
		BooleanExpression boolExp =
			new ExprStatementParser("true or false and false")
				.parseBooleanExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testAndExp() throws Exception {
		BooleanExpression boolExp =
			new ExprStatementParser("true and false").parseBooleanExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testAndExp2() throws Exception {
		BooleanExpression boolExp =
			new ExprStatementParser("not false and false")
				.parseBooleanExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testAndExp3() throws Exception {
		BooleanExpression boolExp =
			new ExprStatementParser("not (false and false)")
				.parseBooleanExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testNotExp() throws Exception {
		BooleanExpression boolExp =
			new ExprStatementParser("not false").parseBooleanExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testNotExp2() throws Exception {
		BooleanExpression boolExp =
			new ExprStatementParser("true").parseBooleanExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testEqualExp() throws Exception {
		BooleanExpression boolExp =
			new ExprStatementParser("1 = 1").parseBooleanExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testEqualExp2() throws Exception {
		BooleanExpression boolExp =
			new ExprStatementParser("1 = 2").parseBooleanExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testEqualExp3() throws Exception {
		BooleanExpression boolExp =
			new ExprStatementParser("1 eq 1").parseBooleanExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testNotEqualExp() throws Exception {
		BooleanExpression boolExp =
			new ExprStatementParser("1 != 1").parseBooleanExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testNotEqualExp2() throws Exception {
		BooleanExpression boolExp =
			new ExprStatementParser("1 != 2").parseBooleanExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testNotEqualExp3() throws Exception {
		BooleanExpression boolExp =
			new ExprStatementParser("1 ne 2").parseBooleanExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testGreaterEqualExp() throws Exception {
		BooleanExpression boolExp =
			new ExprStatementParser("1 >= 1").parseBooleanExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testGreaterEqualExp2() throws Exception {
		BooleanExpression boolExp =
			new ExprStatementParser("1 >= 2").parseBooleanExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testGreaterEqualExp3() throws Exception {
		BooleanExpression boolExp =
			new ExprStatementParser("2 >= 1").parseBooleanExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testGreaterEqualExp4() throws Exception {
		BooleanExpression boolExp =
			new ExprStatementParser("2 ge 1").parseBooleanExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testGreaterThanExp() throws Exception {
		BooleanExpression boolExp =
			new ExprStatementParser("1 > 1").parseBooleanExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testGreaterThanExp2() throws Exception {
		BooleanExpression boolExp =
			new ExprStatementParser("1 > 2").parseBooleanExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testGreaterThanExp3() throws Exception {
		BooleanExpression boolExp =
			new ExprStatementParser("2 > 1").parseBooleanExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testGreaterThanExp4() throws Exception {
		BooleanExpression boolExp =
			new ExprStatementParser("2 gt 1").parseBooleanExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testLessEqualExp() throws Exception {
		BooleanExpression boolExp =
			new ExprStatementParser("1 <= 1").parseBooleanExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testLessEqualExp2() throws Exception {
		BooleanExpression boolExp =
			new ExprStatementParser("1 <= 2").parseBooleanExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testLessEqualExp3() throws Exception {
		BooleanExpression boolExp =
			new ExprStatementParser("2 <= 1").parseBooleanExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testLessEqualExp4() throws Exception {
		BooleanExpression boolExp =
			new ExprStatementParser("2 le 1").parseBooleanExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testLessThanExp() throws Exception {
		BooleanExpression boolExp =
			new ExprStatementParser("1 < 1").parseBooleanExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testLessThanExp2() throws Exception {
		BooleanExpression boolExp =
			new ExprStatementParser("1 < 2").parseBooleanExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testLessThanExp3() throws Exception {
		BooleanExpression boolExp =
			new ExprStatementParser("2 < 1").parseBooleanExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testLessThanExp4() throws Exception {
		BooleanExpression boolExp =
			new ExprStatementParser("2 lt 1").parseBooleanExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testInExp() throws Exception {
		BooleanExpression boolExp =
			new ExprStatementParser("2 in (1,2,3,null)")
				.parseBooleanExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testInExp2() throws Exception {
		BooleanExpression boolExp =
			new ExprStatementParser("4 in (1,2,3,null)")
				.parseBooleanExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testNotInExp() throws Exception {
		BooleanExpression boolExp =
			new ExprStatementParser("2 not in (1,2,3,null)")
				.parseBooleanExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testNotInExp2() throws Exception {
		BooleanExpression boolExp =
			new ExprStatementParser("4 not in (1,2,3,null)")
				.parseBooleanExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testBetweenExp() throws Exception {
		BooleanExpression boolExp =
			new ExprStatementParser("2 between 1 and 3")
				.parseBooleanExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testBetweenExp2() throws Exception {
		BooleanExpression boolExp =
			new ExprStatementParser("4 between 1 and 3")
				.parseBooleanExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testNotBetweenExp() throws Exception {
		BooleanExpression boolExp =
			new ExprStatementParser("2 not between 1 and 3")
				.parseBooleanExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testNotBetweenExp2() throws Exception {
		BooleanExpression boolExp =
			new ExprStatementParser("4 not between 1 and 3")
				.parseBooleanExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testIsNullExp() throws Exception {
		BooleanExpression boolExp =
			new ExprStatementParser("null is null").parseBooleanExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testIsNullExp2() throws Exception {
		BooleanExpression boolExp =
			new ExprStatementParser("1 is null").parseBooleanExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testIsNotNullExp() throws Exception {
		BooleanExpression boolExp =
			new ExprStatementParser("null is not null")
				.parseBooleanExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testIsNotNullExp2() throws Exception {
		BooleanExpression boolExp =
			new ExprStatementParser("1 is not null").parseBooleanExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testLikeExp() throws Exception {
		BooleanExpression boolExp =
			new ExprStatementParser("'SCOTT' LIKE '_CO__'")
				.parseBooleanExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testNotLikeExp() throws Exception {
		BooleanExpression boolExp =
			new ExprStatementParser("'SCOTT' NOT LIKE '_CO__'")
				.parseBooleanExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testNotLikeExp2() throws Exception {
		BooleanExpression boolExp =
			new ExprStatementParser("'SCOTT' NOT LIKE ''")
				.parseBooleanExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testMatchExp() throws Exception {
		BooleanExpression boolExp =
			new ExprStatementParser("'abcAde' ~ '[a-zA-Z]*'")
				.parseBooleanExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testMatchExp2() throws Exception {
		BooleanExpression boolExp =
			new ExprStatementParser("'aBcAde' ~* '[a-z]*'")
				.parseBooleanExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testNotMatchExp() throws Exception {
		BooleanExpression boolExp =
			new ExprStatementParser("'abcAde' !~ '[a-zA-Z]*'")
				.parseBooleanExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testNotMatchExp2() throws Exception {
		BooleanExpression boolExp =
			new ExprStatementParser("'aBcAde' !~* '[a-z]*'")
				.parseBooleanExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testAddSubExp() throws Exception {
		Expression exp = new ExprStatementParser("1 + 2").parseExpression();
		assertEquals("1", new Integer(3), exp.evaluateValue(null));
	}

	public void testAddSubExp2() throws Exception {
		Expression exp = new ExprStatementParser("1 - 2").parseExpression();
		assertEquals("1", new Integer(-1), exp.evaluateValue(null));
	}

	public void testAddSubExp3() throws Exception {
		Expression exp = new ExprStatementParser("1 + 2 * 3").parseExpression();
		assertEquals("1", new Integer(7), exp.evaluateValue(null));
	}

	public void testAddSubExp4() throws Exception {
		Expression exp =
			new ExprStatementParser("(1 + 2) * 3").parseExpression();
		assertEquals("1", new Integer(9), exp.evaluateValue(null));
	}

	public void testMultDivModExp() throws Exception {
		Expression exp = new ExprStatementParser("2 * 3").parseExpression();
		assertEquals("1", new Integer(6), exp.evaluateValue(null));
	}

	public void testMultDivModExp2() throws Exception {
		Expression exp = new ExprStatementParser("5 / 2").parseExpression();
		assertEquals("1", new Integer(2), exp.evaluateValue(null));
	}

	public void testMultDivModExp3() throws Exception {
		Expression exp = new ExprStatementParser("5 % 2").parseExpression();
		assertEquals("1", new Integer(1), exp.evaluateValue(null));
	}

	public void testMultDivModExp4() throws Exception {
		Expression exp =
			new ExprStatementParser("12 / 3 * 4").parseExpression();
		assertEquals("1", new Integer(16), exp.evaluateValue(null));
	}

	public void testMultDivModExp5() throws Exception {
		Expression exp =
			new ExprStatementParser("12 / (3 * 4)").parseExpression();
		assertEquals("1", new Integer(1), exp.evaluateValue(null));
	}

	public void testMultDivModExp6() throws Exception {
		Expression exp = new ExprStatementParser("1.2 * 1.2").parseExpression();
		assertEquals("1", new Double(1.44), exp.evaluateValue(null));
	}

	public void testIntegerExp() throws Exception {
		Expression exp = new ExprStatementParser("1").parseExpression();
		assertEquals("1", new Integer(1), exp.evaluateValue(null));
	}

	public void testLongExp() throws Exception {
		Expression exp =
			new ExprStatementParser("10000000000").parseExpression();
		assertEquals("1", new Long(10000000000L), exp.evaluateValue(null));
	}

	public void testDoubleExp() throws Exception {
		Expression exp = new ExprStatementParser("1.1").parseExpression();
		assertEquals("1", new Double("1.1"), exp.evaluateValue(null));
	}

	public void testStringExp() throws Exception {
		Expression exp = new ExprStatementParser("'123'").parseExpression();
		assertEquals("1", "123", exp.evaluateValue(null));
	}

	public void testGetVariableExp() throws Exception {
		ExprContext ctx = new ExprContextImpl();
		Expression exp = new ExprStatementParser("aaa + 2").parseExpression();
		ctx.setValue("aaa", new Integer(1));
		assertEquals("1", new Integer(3), exp.evaluateValue(ctx));
	}

	public void testSetVariableExp() throws Exception {
		ExprContext ctx = new ExprContextImpl();
		Expression exp = new ExprStatementParser("aaa := 2").parseExpression();
		assertEquals("1", new Integer(2), exp.evaluateValue(ctx));
		assertEquals("2", new Integer(2), ctx.getValue("aaa"));
	}

	public void testSetVariableExp2() throws Exception {
		ExprContext ctx = new ExprContextImpl();
		Expression exp =
			new ExprStatementParser("map.aaa := 2").parseExpression();
		Map map = new HashMap();
		ctx.setValue("map", map);
		assertEquals("1", new Integer(2), exp.evaluateValue(ctx));
		assertEquals("2", new Integer(2), map.get("aaa"));
	}

	public void testToTImestampExp() throws Exception {
		Expression exp =
			new ExprStatementParser("to_timestamp('20021119', 'yyyyMMdd')")
				.parseExpression();
		assertEquals(
			"1",
			Conversion.toTimestamp("20021119", "yyyyMMdd"),
			exp.evaluateValue(null));
	}

	public void testToStringExp() throws Exception {
		Expression exp =
			new ExprStatementParser("to_string(1234, '#,###')")
				.parseExpression();
		assertEquals("1", "1,234", exp.evaluateValue(null));
	}

	public void testToStringExp2() throws Exception {
		ExprContext ctx = new ExprContextImpl();
		Expression exp =
			new ExprStatementParser("to_string(aaa, 'yyyy-MM-dd')")
				.parseExpression();
		ctx.setValue("aaa", Conversion.toTimestamp("20021119", "yyyyMMdd"));
		assertEquals("1", "2002-11-19", exp.evaluateValue(ctx));
	}

	public void testToBigDecimalExp() throws Exception {
		Expression exp =
			new ExprStatementParser("to_bigdecimal('1234')").parseExpression();
		assertEquals("1", new BigDecimal(1234), exp.evaluateValue(null));
	}

	public void testToDoubleExp() throws Exception {
		Expression exp =
			new ExprStatementParser("to_double(1234)").parseExpression();
		assertEquals("1", new Double(1234), exp.evaluateValue(null));
	}

	public void testToLongExp() throws Exception {
		Expression exp =
			new ExprStatementParser("to_long('1234')").parseExpression();
		assertEquals("1", new Long(1234), exp.evaluateValue(null));
	}

	public void testToIntegerExp() throws Exception {
		Expression exp =
			new ExprStatementParser("to_integer('1234')").parseExpression();
		assertEquals("1", new Integer(1234), exp.evaluateValue(null));
	}

	public void testNowExp() throws Exception {
		Expression exp = new ExprStatementParser("now()").parseExpression();
		assertEquals(
			"1",
			true,
			exp.evaluateValue(null) instanceof java.sql.Timestamp);
	}

	public void testNullExp() throws Exception {
		Expression exp = new ExprStatementParser("null").parseExpression();
		assertEquals("1", null, exp.evaluateValue(null));
	}

	public void testCaseExp() throws Exception {
		Expression exp =
			new ExprStatementParser("case when false then 'aaa' when true then 'bbb' end")
				.parseExpression();
		assertEquals("1", "bbb", exp.evaluateValue(null));
	}

	public void testCaseExp2() throws Exception {
		Expression exp =
			new ExprStatementParser("case when false then 'aaa' when false then 'bbb' else 'ccc' end")
				.parseExpression();
		assertEquals("1", "ccc", exp.evaluateValue(null));
	}

	public void testCaseExp3() throws Exception {
		ExprContext ctx = new ExprContextImpl();
		Expression exp =
			new ExprStatementParser("case aaa when '1' then 'aaa' when '2' then 'bbb' else 'ccc' end")
				.parseExpression();
		ctx.setValue("aaa", "2");
		assertEquals("1", "bbb", exp.evaluateValue(ctx));
	}

	public void testCoalesceExp() throws Exception {
		Expression exp =
			new ExprStatementParser("coalesce(null, '111')").parseExpression();
		assertEquals("1", "111", exp.evaluateValue(null));
	}

	public void testConcatenateExp() throws Exception {
		Expression exp =
			new ExprStatementParser("'aaa' || '111'").parseExpression();
		assertEquals("1", "aaa111", exp.evaluateValue(null));
	}

	public void testNvlExp() throws Exception {
		Expression exp =
			new ExprStatementParser("nvl(null, '111')").parseExpression();
		assertEquals("1", "111", exp.evaluateValue(null));
	}

	public void testNvlExp2() throws Exception {
		Expression exp =
			new ExprStatementParser("nvl('222', '111')").parseExpression();
		assertEquals("1", "222", exp.evaluateValue(null));
	}

	public void testNullifExp() throws Exception {
		Expression exp =
			new ExprStatementParser("nullif('aaa', 'aaa')").parseExpression();
		assertEquals("1", null, exp.evaluateValue(null));
	}

	public void testNullifExp2() throws Exception {
		Expression exp =
			new ExprStatementParser("nullif('aaa', null)").parseExpression();
		assertEquals("1", "aaa", exp.evaluateValue(null));
	}

	public void testNullifExp3() throws Exception {
		Expression exp =
			new ExprStatementParser("nullif(null, 'aaa')").parseExpression();
		assertEquals("1", null, exp.evaluateValue(null));
	}

	public void testAbs() throws Exception {
		Expression exp =
			new ExprStatementParser("java.lang.Math.abs(-1)").parseExpression();
		assertEquals("1", new Integer(1), exp.evaluateValue(null));
	}

	public void testTrim() throws Exception {
		Expression exp =
			new ExprStatementParser("trim(both from ' abc ')")
				.parseExpression();
		assertEquals("1", "abc", exp.evaluateValue(null));
	}

	public void testTrim2() throws Exception {
		Expression exp =
			new ExprStatementParser("trim(leading from ' abc ')")
				.parseExpression();
		assertEquals("1", "abc ", exp.evaluateValue(null));
	}

	public void testTrim3() throws Exception {
		Expression exp =
			new ExprStatementParser("trim(trailing from ' abc ')")
				.parseExpression();
		assertEquals("1", " abc", exp.evaluateValue(null));
	}

	public void testTrim4() throws Exception {
		Expression exp =
			new ExprStatementParser("trim(both 'xy' from 'xyxtrimyyx')")
				.parseExpression();
		assertEquals("1", "trim", exp.evaluateValue(null));
	}

	public void testTrim5() throws Exception {
		Expression exp =
			new ExprStatementParser("trim('xy' from 'xyxtrimyyx')")
				.parseExpression();
		assertEquals("1", "trim", exp.evaluateValue(null));
	}

	public void testTrim6() throws Exception {
		Expression exp =
			new ExprStatementParser("trim(from '   trim   ')")
				.parseExpression();
		assertEquals("1", "trim", exp.evaluateValue(null));
	}

	public void testPosition() throws Exception {
		Expression exp =
			new ExprStatementParser("position('bc' in 'abcde')")
				.parseExpression();
		assertEquals("1", new Integer(2), exp.evaluateValue(null));
	}

	public void testSubstring() throws Exception {
		Expression exp =
			new ExprStatementParser("substring('abcde' from 2 for 3)")
				.parseExpression();
		assertEquals("1", "bcd", exp.evaluateValue(null));
	}

	public void testSubstring2() throws Exception {
		Expression exp =
			new ExprStatementParser("substring('abcde' from 2)")
				.parseExpression();
		assertEquals("1", "bcde", exp.evaluateValue(null));
	}

	public void testNew() throws Exception {
		Expression exp =
			new ExprStatementParser("new java.math.BigDecimal('1')")
				.parseExpression();
		assertEquals("1", new BigDecimal("1"), exp.evaluateValue(null));
	}

	public void testNewArray() throws Exception {
		Expression exp =
			new ExprStatementParser("new java.lang.String[2]")
				.parseExpression();
		String[] ret = (String[]) exp.evaluateValue(null);
		assertEquals("1", 2, ret.length);
	}

	public void testNewArray2() throws Exception {
		Expression exp =
			new ExprStatementParser("new java.lang.String[]{'aaa', 'bbb'}")
				.parseExpression();
		String[] ret = (String[]) exp.evaluateValue(null);
		assertEquals("1", 2, ret.length);
		assertEquals("2", "aaa", ret[0]);
		assertEquals("3", "bbb", ret[1]);
	}

	public void testGetArrayVariableExp() throws Exception {
		ExprContext ctx = new ExprContextImpl();
		Expression exp = new ExprStatementParser("aaa[1]").parseExpression();
		ctx.setValue("aaa", new String[] { "111", "222", "333" });
		assertEquals("1", "222", exp.evaluateValue(ctx));
	}

	public void testSetArrayVariableExp() throws Exception {
		ExprContext ctx = new ExprContextImpl();
		Expression exp =
			new ExprStatementParser("aaa[0] := '111'").parseExpression();
		String[] array = new String[1];
		ctx.setValue("aaa", array);
		assertEquals("1", "111", exp.evaluateValue(ctx));
		assertEquals("2", "111", array[0]);
	}

	public void testToBooleanExp() throws Exception {
		Expression exp =
			new ExprStatementParser("to_boolean(true)").parseExpression();
		assertEquals("1", Boolean.TRUE, exp.evaluateValue(null));
	}

	public void testIsTrueExp() throws Exception {
		BooleanExpression exp =
			new ExprStatementParser("true is true").parseBooleanExpression();
		assertEquals("1", true, exp.evaluate(null));
	}

	public void testIsTrueExp2() throws Exception {
		BooleanExpression exp =
			new ExprStatementParser("to_boolean(true) is true")
				.parseBooleanExpression();
		assertEquals("1", true, exp.evaluate(null));
	}

	public void testIsFalseExp() throws Exception {
		BooleanExpression exp =
			new ExprStatementParser("false is false").parseBooleanExpression();
		assertEquals("1", true, exp.evaluate(null));
	}

	public void testIsFalseExp2() throws Exception {
		BooleanExpression exp =
			new ExprStatementParser("to_boolean(false) is false")
				.parseBooleanExpression();
		assertEquals("1", true, exp.evaluate(null));
	}

	protected void setUp() throws Exception {
	}

	protected void tearDown() throws Exception {
	}

	public static Test suite() {
		return new TestSuite(ExprStatementParserTest.class);
	}

	public static void main(String[] args) {
		junit.textui.TestRunner.main(
			new String[] { ExprStatementParserTest.class.getName()});
	}
}