package test.org.seasar.nazuna;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;

import org.seasar.expr.Expression;
import org.seasar.nazuna.RuleStatementParser;

public class RuleStatementParserTest extends TestCase {

	public RuleStatementParserTest(String name) {
		super(name);
	}

	public void testAbs() throws Exception {
		Expression exp =
			new RuleStatementParser("abs(-1) * 2").parseExpression();
		assertEquals("1", new Integer(2), exp.evaluateValue(null));
	}

	public void testAscii() throws Exception {
		Expression exp =
			new RuleStatementParser("ascii('x')").parseExpression();
		assertEquals("1", new Integer(120), exp.evaluateValue(null));
	}

	public void testBtrim() throws Exception {
		Expression exp =
			new RuleStatementParser("btrim(' x ')").parseExpression();
		assertEquals("1", "x", exp.evaluateValue(null));
	}

	public void testBtrim2() throws Exception {
		Expression exp =
			new RuleStatementParser("btrim('aba', 'a')").parseExpression();
		assertEquals("1", "b", exp.evaluateValue(null));
	}

	public void testBtrim3() throws Exception {
		Expression exp =
			new RuleStatementParser("btrim('xyxtrimyyx', 'xy')")
				.parseExpression();
		assertEquals("1", "trim", exp.evaluateValue(null));
	}

	public void testBtrim4() throws Exception {
		Expression exp =
			new RuleStatementParser("btrim(' abc ')").parseExpression();
		assertEquals("1", "abc", exp.evaluateValue(null));
	}

	public void testLtrim() throws Exception {
		Expression exp =
			new RuleStatementParser("ltrim('zzzytrim', 'xyz')")
				.parseExpression();
		assertEquals("1", "trim", exp.evaluateValue(null));
	}

	public void testLtrim2() throws Exception {
		Expression exp =
			new RuleStatementParser("ltrim(' abc')").parseExpression();
		assertEquals("1", "abc", exp.evaluateValue(null));
	}

	public void testRtrim() throws Exception {
		Expression exp =
			new RuleStatementParser("rtrim('trimxxxx', 'x')").parseExpression();
		assertEquals("1", "trim", exp.evaluateValue(null));
	}

	public void testRtrim2() throws Exception {
		Expression exp =
			new RuleStatementParser("rtrim('abc ')").parseExpression();
		assertEquals("1", "abc", exp.evaluateValue(null));
	}

	public void testChr() throws Exception {
		Expression exp = new RuleStatementParser("chr(65)").parseExpression();
		assertEquals("1", "A", exp.evaluateValue(null));
	}

	public void testLength() throws Exception {
		Expression exp =
			new RuleStatementParser("length('111')").parseExpression();
		assertEquals("1", new Integer(3), exp.evaluateValue(null));
	}

	public void testLower() throws Exception {
		Expression exp =
			new RuleStatementParser("lower('ABC')").parseExpression();
		assertEquals("1", "abc", exp.evaluateValue(null));
	}

	public void testLpad() throws Exception {
		Expression exp =
			new RuleStatementParser("lpad('123', 5)").parseExpression();
		assertEquals("1", "  123", exp.evaluateValue(null));
	}

	public void testLpad2() throws Exception {
		Expression exp =
			new RuleStatementParser("lpad('123', 6, 'xy')").parseExpression();
		assertEquals("1", "xyx123", exp.evaluateValue(null));
	}

	public void testRepeat() throws Exception {
		Expression exp =
			new RuleStatementParser("repeat('Pg', 4)").parseExpression();
		assertEquals("1", "PgPgPgPg", exp.evaluateValue(null));
	}

	public void testReplace() throws Exception {
		Expression exp =
			new RuleStatementParser("replace('123452', '2', 'abc')")
				.parseExpression();
		assertEquals("1", "1abc345abc", exp.evaluateValue(null));
	}

	public void testRpad() throws Exception {
		Expression exp =
			new RuleStatementParser("rpad('123', 5)").parseExpression();
		assertEquals("1", "123  ", exp.evaluateValue(null));
	}

	public void testRpad2() throws Exception {
		Expression exp =
			new RuleStatementParser("rpad('123', 6, 'xy')").parseExpression();
		assertEquals("1", "123xyx", exp.evaluateValue(null));
	}

	public void testStrpos() throws Exception {
		Expression exp =
			new RuleStatementParser("strpos('abcde', 'bc')").parseExpression();
		assertEquals("1", new Integer(2), exp.evaluateValue(null));
	}

	public void testSubstr() throws Exception {
		Expression exp =
			new RuleStatementParser("substr('abcde', 2, 3)").parseExpression();
		assertEquals("1", "bcd", exp.evaluateValue(null));
	}

	public void testSubstr2() throws Exception {
		Expression exp =
			new RuleStatementParser("substr('abcde', 2)").parseExpression();
		assertEquals("1", "bcde", exp.evaluateValue(null));
	}

	public void testUpper() throws Exception {
		Expression exp =
			new RuleStatementParser("upper('abc')").parseExpression();
		assertEquals("1", "ABC", exp.evaluateValue(null));
	}

	public void testTranslate() throws Exception {
		Expression exp =
			new RuleStatementParser("translate('12345', '14', 'ad')")
				.parseExpression();
		assertEquals("1", "a23d5", exp.evaluateValue(null));
	}

	public void testCeil() throws Exception {
		Expression exp =
			new RuleStatementParser("ceil(12.6)").parseExpression();
		assertEquals("1", new Double("13"), exp.evaluateValue(null));
	}

	public void testCeil2() throws Exception {
		Expression exp =
			new RuleStatementParser("ceil(-12.6)").parseExpression();
		assertEquals("1", new Double("-12"), exp.evaluateValue(null));
	}

	public void testDegrees() throws Exception {
		Expression exp =
			new RuleStatementParser("degrees(0.5)").parseExpression();
		Double d = (Double) exp.evaluateValue(null);
		assertEquals("1", new Double("28.64788975654116"), d);
	}

	public void testExp() throws Exception {
		Expression exp = new RuleStatementParser("exp(1.0)").parseExpression();
		Double d = (Double) exp.evaluateValue(null);
		assertEquals("1", new Double("2.7182818284590455"), d);
	}

	public void testFloor() throws Exception {
		Expression exp =
			new RuleStatementParser("floor(-42.8)").parseExpression();
		assertEquals("1", new Double("-43"), exp.evaluateValue(null));
	}

	public void testFloor2() throws Exception {
		Expression exp =
			new RuleStatementParser("floor(42.8)").parseExpression();
		assertEquals("1", new Double("42"), exp.evaluateValue(null));
	}

	public void testLn() throws Exception {
		Expression exp = new RuleStatementParser("ln(2.0)").parseExpression();
		Double d = (Double) exp.evaluateValue(null);
		assertEquals("1", new Double("0.6931471805599453"), d);
	}

	public void testPow() throws Exception {
		Expression exp =
			new RuleStatementParser("pow(9.0, 3.0)").parseExpression();
		assertEquals("1", new Double("729"), exp.evaluateValue(null));
	}

	public void testRadians() throws Exception {
		Expression exp =
			new RuleStatementParser("radians(45.0)").parseExpression();
		Double d = (Double) exp.evaluateValue(null);
		assertEquals("1", new Double("0.7853981633974483"), d);
	}

	public void testRandom() throws Exception {
		Expression exp = new RuleStatementParser("random()").parseExpression();
		Object ret = exp.evaluateValue(null);
		System.out.println(ret);
		assertNotNull("1", ret);
	}

	public void testRound() throws Exception {
		Expression exp =
			new RuleStatementParser("round(42.4)").parseExpression();
		assertEquals("1", new Long("42"), exp.evaluateValue(null));
	}

	public void testExecute() throws Exception {
		String s = "execute('test.org.seasar.nazuna.Hello4', 'Nazuna')";
		Expression exp = new RuleStatementParser(s).parseExpression();
		assertEquals("1", "Hello Nazuna", exp.evaluateValue(null));
	}

	public void testExecute2() throws Exception {
		String s = "execute('test.org.seasar.nazuna.Hello4', null)";
		Expression exp = new RuleStatementParser(s).parseExpression();
		assertEquals("1", "Hello null", exp.evaluateValue(null));
	}
	
	public void testExecute3() throws Exception {
		Expression exp = new RuleStatementParser("execute('/test/org/seasar/nazuna/Return.xml')").parseExpression();
		assertEquals("1", "Hello", exp.evaluateValue(null));
	}

	public void testExecute4() throws Exception {
		Expression exp = new RuleStatementParser("execute('/test/org/seasar/nazuna/Hello2.xml', 'Nazuna')").parseExpression();
		assertEquals("1", null, exp.evaluateValue(null));
	}

	public void testExecute5() throws Exception {
		Expression exp = new RuleStatementParser("execute('/test/org/seasar/nazuna/Execute.xml', 'Nazuna')").parseExpression();
		assertEquals("1", "Hello Nazuna", exp.evaluateValue(null));
	}

	public void testExecute6() throws Exception {
		Expression exp = new RuleStatementParser("execute('/test/org/seasar/nazuna/Execute.xml')").parseExpression();
		assertEquals("1", "Hello Nazuna", exp.evaluateValue(null));
	}

	public void testExecuteRulet() throws Exception {
		String s = "executeRulet('test.org.seasar.nazuna.AddRulet', 1, 2)";
		Expression exp = new RuleStatementParser(s).parseExpression();
		assertEquals("1", new Integer(3), exp.evaluateValue(null));
	}

	public void testSign() throws Exception {
		Expression exp = new RuleStatementParser("sign(2)").parseExpression();
		assertEquals("1", new Integer(1), exp.evaluateValue(null));

		Expression exp2 = new RuleStatementParser("sign(-2)").parseExpression();
		assertEquals("2", new Integer(-1), exp2.evaluateValue(null));

		Expression exp3 = new RuleStatementParser("sign(0)").parseExpression();
		assertEquals("3", new Integer(0), exp3.evaluateValue(null));
	}

	public void testSign2() throws Exception {
		Expression exp = new RuleStatementParser("sign(2.0)").parseExpression();
		assertEquals("1", new Integer(1), exp.evaluateValue(null));

		Expression exp2 =
			new RuleStatementParser("sign(-2.0)").parseExpression();
		assertEquals("2", new Integer(-1), exp2.evaluateValue(null));

		Expression exp3 =
			new RuleStatementParser("sign(0.0)").parseExpression();
		assertEquals("3", new Integer(0), exp3.evaluateValue(null));
	}

	public void testSqrt() throws Exception {
		Expression exp = new RuleStatementParser("sqrt(2.0)").parseExpression();
		Double d = (Double) exp.evaluateValue(null);
		assertEquals("1", new Double("1.4142135623730951"), d);
	}

	public void testTrunc() throws Exception {
		Expression exp =
			new RuleStatementParser("trunc(42.8)").parseExpression();
		assertEquals("1", new Double("42"), exp.evaluateValue(null));
	}

	public void testTrunc2() throws Exception {
		Expression exp =
			new RuleStatementParser("trunc(-42.8)").parseExpression();
		assertEquals("1", new Double("-42"), exp.evaluateValue(null));
	}

	public void testAcos() throws Exception {
		Expression exp = new RuleStatementParser("acos(1.0)").parseExpression();
		assertNotNull("1", exp.evaluateValue(null));
	}

	public void testAsin() throws Exception {
		Expression exp = new RuleStatementParser("asin(1.0)").parseExpression();
		assertNotNull("1", exp.evaluateValue(null));
	}

	public void testAtan() throws Exception {
		Expression exp = new RuleStatementParser("atan(1.0)").parseExpression();
		assertNotNull("1", exp.evaluateValue(null));
	}

	public void testAtan2() throws Exception {
		Expression exp =
			new RuleStatementParser("atan2(1.0, 1.0)").parseExpression();
		assertNotNull("1", exp.evaluateValue(null));
	}

	public void testCos() throws Exception {
		Expression exp = new RuleStatementParser("cos(1.0)").parseExpression();
		assertNotNull("1", exp.evaluateValue(null));
	}

	public void testSin() throws Exception {
		Expression exp = new RuleStatementParser("sin(1.0)").parseExpression();
		assertNotNull("1", exp.evaluateValue(null));
	}

	protected void setUp() throws Exception {
	}

	protected void tearDown() throws Exception {
	}

	public static Test suite() {
		return new TestSuite(RuleStatementParserTest.class);
	}

	public static void main(String[] args) {
		junit.textui.TestRunner.main(
			new String[] { RuleStatementParserTest.class.getName()});
	}
}