// Copyright 2002 Macromedia, Inc. and its licensors.  All Rights Reserved.
//
// Use and/or redistribution of this file, in whole or in part, is subject
// to the Macromedia Sample Files License, which was distributed with this file.
//
#include "ValidatorClasses/creditcardservice.as"
/*
 *
 * Validator Base Class
 *
 */
ValidatorClass = function () {
	this.maxLength = -1;
	this.validationMessage = "";
	this.validatedString = undefined;
}

ValidatorClass.prototype = new Object();

ValidatorClass.prototype.getMaxLength = function() {
	return this.maxLength;
}

ValidatorClass.prototype.setMaxLength = function(length) {
	this.maxLength = length;
}

ValidatorClass.prototype.getFailedValidationMessage = function() {
	return this.validationMessage;
}

ValidatorClass.prototype.setFailedValidationMessage = function(message) {
	this.validationMessage = message;
}

ValidatorClass.prototype.getValidatedString = function() {
	return this.validatedString;
}

ValidatorClass.prototype.setValidatedString = function(str) {
	this.validatedString = str;
}

// The validate function returns true or false and the caller
// can then call getFailedValidationMessage() or getValidString() as
// needed in case of error or success, respectively.
ValidatorClass.prototype.validate = function() {
	if ( this.validatedString == undefined ) {
		// No string to validate
		this.setFailedValidationMessage("No string to validate");
		return false;
	}
	else if ( this.validatedString.length > this.maxLength ) {
		this.setFailedValidationMessage("Maximum input length exceeded");
		return false;
	}
	return true;
}

/******
	'LIBRARY' style functions for the Validator class and its derivatives
******/

// stripNonDigits returns no more than 16 digits from the supplied string
ValidatorClass.prototype.stripNonDigits = function (str) {
	return stripNonDigits(str);
}



// stripSpace() removes all occurrences of white space from a string.
// stripSpace() returns a NEW String object. The supplied String is unmodified.
ValidatorClass.prototype.stripSpace = function(s) {
	return stripSpace(s);
}


// trim() removes leading and trailing spaces from a string.
// If the supplied string is non-zero length, a NEW string is
// returned and the original string is unmodified.
ValidatorClass.prototype.trim = function(s) {
	return trim(s);
}


// isWhiteSpace returns true if the supplied single character string is
// considered white space. Space is \u0020, tab is \u0009.
ValidatorClass.prototype.isWhiteSpace = function(cS) {
	return isWhiteSpace(cS);
}


// isDigit returns true if the supplied single character string is
// a digit from 0 (\u0030) thru 9 (\u0039).
ValidatorClass.prototype.isDigit = function(d) {
	return isDigit(d);
}


// isDash returns true if the supplied single character string is
// the dash ('-') character (\u002d)
ValidatorClass.prototype.isDash = function(d) {
	return isDash(d);
}



/*
 *
 * Phone Validator Class
 * EXTENDS Validator
 */
ValidatePhone = function(length, message) {
	this.setMaxLength(length);
	this.setFailedValidationMessage(message);
}

ValidatePhone.prototype = new ValidatorClass();


//
// validatePhone checks for one of several 10 digit phone number styles.
// 		Accepted styles:	all digits, i.e. 5551230000
//							(555)123-0000
//							555.123.0000
//							555_123_0000
//							anything with 10 digits, i.e. prefix(555)exchange123-extension0000
// Returns true if string meets validation criteria, otherwise returns false.
// This objects validatedString is undefined if false is returned here.
//
ValidatePhone.prototype.validate = function(str) {

	// Remove all non digit characters
	str = this.stripNonDigits(str);

	// If max length exceeded, return false.
	// If ten digits are not left, return false.
	if ( (str.length > this.getMaxLength()) || (str.length <> 10) ) {
		return false;
	}

	this.validatedString = str;
	return true;
}
	

/*
 *
 * Email Validator Class
 * EXTENDS Validator
 */
ValidateEmail = function(length, message) {
	this.setMaxLength(length);
	this.setFailedValidationMessage(message);
}

ValidateEmail.prototype = new ValidatorClass();

//
// Email validation verifies the supplied string is sufficient length
// to be valid (min 5 chars) and no longer than the object's maximum
// length. It is also verified that an '@' and a '.' exist and that the
// first '@' is before the last '.'. There is obviously more that could
// be done here.
//
ValidateEmail.prototype.validate = function(str) {

	str = this.trim(str);

	if ( ( str.length > this.getMaxLength() ) ||
		 ( str.length < 5 ) ||
		 ( (atIndex = str.indexOf("@")) == -1 ) ||
		 ( (dotIndex = str.lastIndexOf(".")) == -1 ) ||
		 ( dotIndex < atIndex ))
	{
		return false;
	}
	this.validatedString = str;
	return true;
}


/*
 *
 * ZipCode Validator Class
 * EXTENDS Validator
 */
ValidateZipCode = function(length, message) {
	this.setMaxLength(length);
	this.setFailedValidationMessage(message);
}

ValidateZipCode.prototype = new ValidatorClass();

//
// ZipCode validator checks for one of two zip code styles
// 		Accepted styles:	<5 digit zip>
//							<5 digit zip>-<4 digit ext>
// Note: this validation only allows digits and the dash character.
//			
ValidateZipCode.prototype.validate = function(str) {

	str = this.stripSpace(str);
	
	if ( (str.length > this.getMaxLength()) || (str.length < 5) ) {
		return false;
	}
	
	// At this point, the length is 5 or more and all
	// white space is removed. Validate that exactly  5 or 9 digits
	// exist with the string (matching the zip and zip+4 formats).
	this.validatedString = "";
	for ( i = 0; i < str.length; i++ ) {
		if ( this.isDigit(str.charAt(i)) ) {
			this.validatedString = 
				this.validatedString.concat( str.charAt(i) );
		}
		else if ( (!this.isDash(str.charAt(i))) ||
				  ( this.isDash(str.charAt(i)) &&  (i != 5) ) ) {
			return false;
		}
	}
	return ( (this.validatedString.length == 5) || (this.validatedString.length == 9) );
}

//Author: Brad Sahr (bsahr)
