package org.seasar.extension.dataset.reader;

import java.io.IOException;
import java.io.InputStream;
import java.math.BigDecimal;

import org.apache.poi.hssf.usermodel.HSSFCell;
import org.apache.poi.hssf.usermodel.HSSFCellStyle;
import org.apache.poi.hssf.usermodel.HSSFDataFormat;
import org.apache.poi.hssf.usermodel.HSSFDateUtil;
import org.apache.poi.hssf.usermodel.HSSFRow;
import org.apache.poi.hssf.usermodel.HSSFSheet;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.seasar.extension.dataset.DataReader;
import org.seasar.extension.dataset.DataRow;
import org.seasar.extension.dataset.DataSet;
import org.seasar.extension.dataset.DataSetConstants;
import org.seasar.extension.dataset.DataTable;
import org.seasar.extension.dataset.impl.DataColumnImpl;
import org.seasar.extension.dataset.impl.DataRowImpl;
import org.seasar.extension.dataset.impl.DataSetImpl;
import org.seasar.extension.dataset.impl.DataTableImpl;
import org.seasar.framework.exception.IORuntimeException;
import org.seasar.framework.util.Base64Util;
import org.seasar.framework.util.TimestampConversionUtil;

/**
 * @author higa
 *
 */
public class XlsReader implements DataReader, DataSetConstants {

	private DataSet dataSet_;
	private HSSFWorkbook workbook_;
	private HSSFDataFormat dataFormat_;

	public XlsReader(InputStream in) {
		try {
			workbook_ = new HSSFWorkbook(in);
		} catch (IOException ex) {
			throw new IORuntimeException(ex);
		}
		dataFormat_ = workbook_.createDataFormat();
		dataSet_ = new DataSetImpl();
		for (int i = 0; i < workbook_.getNumberOfSheets(); ++i) {
			dataSet_.addTable(
				createTable(
					workbook_.getSheetName(i),
					workbook_.getSheetAt(i)));
		}
	}

	/**
	 * @see org.seasar.extension.dataset.DataReader#read()
	 */
	public DataSet read() {
		return dataSet_;
	}

	private DataTable createTable(String sheetName, HSSFSheet sheet) {
		DataTable table = new DataTableImpl(sheetName);
		int rowCount = sheet.getLastRowNum();
		if (rowCount > 0) {
			setupColumns(table, sheet.getRow(0));
			setupRows(table, sheet);
		}
		return table;
	}

	private void setupColumns(DataTable table, HSSFRow row) {
		for (int i = 0;; ++i) {
			HSSFCell cell = row.getCell((short) i);
			if (cell == null) {
				break;
			}
			table.addColumn(
				new DataColumnImpl(cell.getStringCellValue().trim()));
		}
	}

	private void setupRows(DataTable table, HSSFSheet sheet) {
		for (int i = 1;; ++i) {
			HSSFRow row = sheet.getRow((short) i);
			if (row == null || i > 10) {
				break;
			}
			setupRow(table, row);
		}
	}

	private void setupRow(DataTable table, HSSFRow row) {
		DataRow dataRow = new DataRowImpl(table);
		for (int i = 0; i < table.getColumnSize(); ++i) {
			HSSFCell cell = row.getCell((short) i);
			dataRow.setValue(i, getValue(cell));
		}
		table.addRow(dataRow);
	}

	public boolean isCellBase64Formatted(HSSFCell cell) {
		HSSFCellStyle cs = cell.getCellStyle();
		short dfNum = cs.getDataFormat();
		return BASE64_FORMAT.equals(dataFormat_.getFormat(dfNum));
	}

	public Object getValue(HSSFCell cell) {
		if (cell == null) {
			return null;
		}
		switch (cell.getCellType()) {
			case HSSFCell.CELL_TYPE_NUMERIC :
				if (HSSFDateUtil.isCellDateFormatted(cell)) {
					return TimestampConversionUtil.toTimestamp(
						cell.getDateCellValue());
				}
				return new BigDecimal(cell.getNumericCellValue());
			case HSSFCell.CELL_TYPE_STRING :
				String s = cell.getStringCellValue().trim();
				if (isCellBase64Formatted(cell)) {
					return Base64Util.decode(s);
				}
				return s;
			default :
				return null;
		}
	}
}
