package test.org.seasar.framework.sel.parser;

import java.math.BigDecimal;
import java.util.HashMap;
import java.util.Map;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;

import org.seasar.framework.util.TimestampConversionUtil;
import org.seasar.framework.sel.BoolExpression;
import org.seasar.framework.sel.Expression;
import org.seasar.framework.sel.SelContext;
import org.seasar.framework.sel.context.SelContextImpl;
import org.seasar.framework.sel.parser.SelParser;

public class SelParserTest extends TestCase {

	public static final String HOGE = "hoge";

	public SelParserTest(String name) {
		super(name);
	}

	public void testOrExp() throws Exception {
		BoolExpression boolExp =
			new SelParser("true or false").parseBoolExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testOrExp2() throws Exception {
		BoolExpression boolExp =
			new SelParser("(true or false) and false").parseBoolExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testOrExp3() throws Exception {
		BoolExpression boolExp =
			new SelParser("true or false and false").parseBoolExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testAndExp() throws Exception {
		BoolExpression boolExp =
			new SelParser("true and false").parseBoolExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testAndExp2() throws Exception {
		BoolExpression boolExp =
			new SelParser("not false and false").parseBoolExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testAndExp3() throws Exception {
		BoolExpression boolExp =
			new SelParser("not (false and false)").parseBoolExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testNotExp() throws Exception {
		BoolExpression boolExp =
			new SelParser("not false").parseBoolExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testNotExp2() throws Exception {
		BoolExpression boolExp = new SelParser("true").parseBoolExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testEqualExp() throws Exception {
		BoolExpression boolExp = new SelParser("1 = 1").parseBoolExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testEqualExp2() throws Exception {
		BoolExpression boolExp = new SelParser("1 = 2").parseBoolExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testEqualExp3() throws Exception {
		BoolExpression boolExp = new SelParser("1 eq 1").parseBoolExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testNotEqualExp() throws Exception {
		BoolExpression boolExp = new SelParser("1 != 1").parseBoolExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testNotEqualExp2() throws Exception {
		BoolExpression boolExp = new SelParser("1 != 2").parseBoolExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testNotEqualExp3() throws Exception {
		BoolExpression boolExp = new SelParser("1 ne 2").parseBoolExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testGreaterEqualExp() throws Exception {
		BoolExpression boolExp = new SelParser("1 >= 1").parseBoolExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testGreaterEqualExp2() throws Exception {
		BoolExpression boolExp = new SelParser("1 >= 2").parseBoolExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testGreaterEqualExp3() throws Exception {
		BoolExpression boolExp = new SelParser("2 >= 1").parseBoolExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testGreaterEqualExp4() throws Exception {
		BoolExpression boolExp = new SelParser("2 ge 1").parseBoolExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testGreaterThanExp() throws Exception {
		BoolExpression boolExp = new SelParser("1 > 1").parseBoolExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testGreaterThanExp2() throws Exception {
		BoolExpression boolExp = new SelParser("1 > 2").parseBoolExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testGreaterThanExp3() throws Exception {
		BoolExpression boolExp = new SelParser("2 > 1").parseBoolExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testGreaterThanExp4() throws Exception {
		BoolExpression boolExp = new SelParser("2 gt 1").parseBoolExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testLessEqualExp() throws Exception {
		BoolExpression boolExp = new SelParser("1 <= 1").parseBoolExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testLessEqualExp2() throws Exception {
		BoolExpression boolExp = new SelParser("1 <= 2").parseBoolExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testLessEqualExp3() throws Exception {
		BoolExpression boolExp = new SelParser("2 <= 1").parseBoolExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testLessEqualExp4() throws Exception {
		BoolExpression boolExp = new SelParser("2 le 1").parseBoolExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testLessThanExp() throws Exception {
		BoolExpression boolExp = new SelParser("1 < 1").parseBoolExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testLessThanExp2() throws Exception {
		BoolExpression boolExp = new SelParser("1 < 2").parseBoolExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testLessThanExp3() throws Exception {
		BoolExpression boolExp = new SelParser("2 < 1").parseBoolExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testLessThanExp4() throws Exception {
		BoolExpression boolExp = new SelParser("2 lt 1").parseBoolExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testInExp() throws Exception {
		BoolExpression boolExp =
			new SelParser("2 in (1,2,3,null)").parseBoolExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testInExp2() throws Exception {
		BoolExpression boolExp =
			new SelParser("4 in (1,2,3,null)").parseBoolExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testNotInExp() throws Exception {
		BoolExpression boolExp =
			new SelParser("2 not in (1,2,3,null)").parseBoolExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testNotInExp2() throws Exception {
		BoolExpression boolExp =
			new SelParser("4 not in (1,2,3,null)").parseBoolExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testBetweenExp() throws Exception {
		BoolExpression boolExp =
			new SelParser("2 between 1 and 3").parseBoolExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testBetweenExp2() throws Exception {
		BoolExpression boolExp =
			new SelParser("4 between 1 and 3").parseBoolExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testNotBetweenExp() throws Exception {
		BoolExpression boolExp =
			new SelParser("2 not between 1 and 3").parseBoolExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testNotBetweenExp2() throws Exception {
		BoolExpression boolExp =
			new SelParser("4 not between 1 and 3").parseBoolExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testIsNullExp() throws Exception {
		BoolExpression boolExp =
			new SelParser("null is null").parseBoolExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testIsNullExp2() throws Exception {
		BoolExpression boolExp =
			new SelParser("1 is null").parseBoolExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testIsNotNullExp() throws Exception {
		BoolExpression boolExp =
			new SelParser("null is not null").parseBoolExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testIsNotNullExp2() throws Exception {
		BoolExpression boolExp =
			new SelParser("1 is not null").parseBoolExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testLikeExp() throws Exception {
		BoolExpression boolExp =
			new SelParser("'SCOTT' LIKE '_CO__'").parseBoolExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testNotLikeExp() throws Exception {
		BoolExpression boolExp =
			new SelParser("'SCOTT' NOT LIKE '_CO__'").parseBoolExpression();
		assertEquals("1", false, boolExp.evaluate(null));
	}

	public void testNotLikeExp2() throws Exception {
		BoolExpression boolExp =
			new SelParser("'SCOTT' NOT LIKE ''").parseBoolExpression();
		assertEquals("1", true, boolExp.evaluate(null));
	}

	public void testAddSubExp() throws Exception {
		Expression exp = new SelParser("1 + 2").parseExpression();
		assertEquals("1", new Integer(3), exp.evaluateValue(null));
	}

	public void testAddSubExp2() throws Exception {
		Expression exp = new SelParser("1 - 2").parseExpression();
		assertEquals("1", new Integer(-1), exp.evaluateValue(null));
	}

	public void testAddSubExp3() throws Exception {
		Expression exp = new SelParser("1 + 2 * 3").parseExpression();
		assertEquals("1", new Integer(7), exp.evaluateValue(null));
	}

	public void testAddSubExp4() throws Exception {
		Expression exp = new SelParser("(1 + 2) * 3").parseExpression();
		assertEquals("1", new Integer(9), exp.evaluateValue(null));
	}

	public void testMultDivModExp() throws Exception {
		Expression exp = new SelParser("2 * 3").parseExpression();
		assertEquals("1", new Integer(6), exp.evaluateValue(null));
	}

	public void testMultDivModExp2() throws Exception {
		Expression exp = new SelParser("5 / 2").parseExpression();
		assertEquals("1", new Integer(2), exp.evaluateValue(null));
	}

	public void testMultDivModExp3() throws Exception {
		Expression exp = new SelParser("5 % 2").parseExpression();
		assertEquals("1", new Integer(1), exp.evaluateValue(null));
	}

	public void testMultDivModExp4() throws Exception {
		Expression exp = new SelParser("12 / 3 * 4").parseExpression();
		assertEquals("1", new Integer(16), exp.evaluateValue(null));
	}

	public void testMultDivModExp5() throws Exception {
		Expression exp = new SelParser("12 / (3 * 4)").parseExpression();
		assertEquals("1", new Integer(1), exp.evaluateValue(null));
	}

	public void testMultDivModExp6() throws Exception {
		Expression exp = new SelParser("1.2 * 1.2").parseExpression();
		assertEquals("1", new BigDecimal("1.44"), exp.evaluateValue(null));
	}

	public void testIntegerExp() throws Exception {
		Expression exp = new SelParser("1").parseExpression();
		assertEquals("1", new Integer(1), exp.evaluateValue(null));
	}

	public void testLongExp() throws Exception {
		Expression exp = new SelParser("10000000000").parseExpression();
		assertEquals("1", new Long(10000000000L), exp.evaluateValue(null));
	}

	public void testBigDecimalExp() throws Exception {
		Expression exp = new SelParser("1.1").parseExpression();
		assertEquals("1", new BigDecimal("1.1"), exp.evaluateValue(null));
	}

	public void testStringExp() throws Exception {
		Expression exp = new SelParser("'123'").parseExpression();
		assertEquals("1", "123", exp.evaluateValue(null));
	}

	public void testStringExp2() throws Exception {
		Expression exp = new SelParser("'a''b'").parseExpression();
		assertEquals("1", "a'b", exp.evaluateValue(null));
	}

	public void testToTimestampExp() throws Exception {
		Expression exp =
			new SelParser("to_timestamp('20021119', 'yyyyMMdd')")
				.parseExpression();
		assertEquals(
			"1",
			TimestampConversionUtil.toTimestamp("20021119", "yyyyMMdd"),
			exp.evaluateValue(null));
	}

	public void testToStringExp() throws Exception {
		Expression exp =
			new SelParser("to_string(1234, '#,###')").parseExpression();
		assertEquals("1", "1,234", exp.evaluateValue(null));
	}

	public void testNowExp() throws Exception {
		Expression exp = new SelParser("now()").parseExpression();
		assertEquals(
			"1",
			true,
			exp.evaluateValue(null) instanceof java.sql.Timestamp);
	}

	public void testNullExp() throws Exception {
		Expression exp = new SelParser("null").parseExpression();
		assertEquals("1", null, exp.evaluateValue(null));
	}

	public void testConcatenateExp() throws Exception {
		Expression exp = new SelParser("'aaa' || '111'").parseExpression();
		assertEquals("1", "aaa111", exp.evaluateValue(null));
	}

	public void testGetVariableExp() throws Exception {
		SelContext ctx = new SelContextImpl();
		Expression exp = new SelParser("aaa + 2").parseExpression();
		ctx.setVariable("aaa", new Integer(1));
		assertEquals("1", new Integer(3), exp.evaluateValue(ctx));
	}
	
	public void testGetVariableExp2() throws Exception {
		SelContext ctx = new SelContextImpl();
		Expression exp = new SelParser("aaa:bbb + 2").parseExpression();
		ctx.setVariable("aaa:bbb", new Integer(1));
		assertEquals("1", new Integer(3), exp.evaluateValue(ctx));
	}

	public void testGetArrayVariableExp() throws Exception {
		SelContext ctx = new SelContextImpl();
		Expression exp = new SelParser("aaa[1]").parseExpression();
		ctx.setVariable("aaa", new String[] { "111", "222", "333" });
		assertEquals("1", "222", exp.evaluateValue(ctx));
	}

	public void testNew() throws Exception {
		Expression exp =
			new SelParser("new java.math.BigDecimal('1')").parseExpression();
		assertEquals("1", new BigDecimal("1"), exp.evaluateValue(null));
	}

	public void testNewArray() throws Exception {
		Expression exp =
			new SelParser("new java.lang.String[2]").parseExpression();
		String[] ret = (String[]) exp.evaluateValue(null);
		assertEquals("1", 2, ret.length);
	}

	public void testNewArray2() throws Exception {
		Expression exp =
			new SelParser("new java.lang.String[]{'aaa', 'bbb'}")
				.parseExpression();
		String[] ret = (String[]) exp.evaluateValue(null);
		assertEquals("1", 2, ret.length);
		assertEquals("2", "aaa", ret[0]);
		assertEquals("3", "bbb", ret[1]);
	}

	public void testIsTrueExp() throws Exception {
		BoolExpression exp =
			new SelParser("true is true").parseBoolExpression();
		assertEquals("1", true, exp.evaluate(null));
	}

	public void testIsFalseExp() throws Exception {
		BoolExpression exp =
			new SelParser("false is false").parseBoolExpression();
		assertEquals("1", true, exp.evaluate(null));
	}

	public void testStaticMethodExp() throws Exception {
		Expression exp =
			new SelParser("java.lang.Integer.toString(2)").parseExpression();
		assertEquals("1", "2", exp.evaluateValue(null));
	}

	public void testWordExp() throws Exception {
		Expression exp =
			new SelParser("java.lang.Integer.toString(2).toString()")
				.parseExpression();
		assertEquals("1", "2", exp.evaluateValue(null));
	}

	public void testWordExp2() throws Exception {
		Expression exp =
			new SelParser("java.lang.Integer.toString(2).toCharArray()[0]")
				.parseExpression();
		assertEquals("1", new Character('2'), exp.evaluateValue(null));
	}

	public void testWordExp3() throws Exception {
		Expression exp = new SelParser("aaa[0].bbb[0]").parseExpression();
		Map map = new HashMap();
		map.put("bbb", new String[] { "hoge" });
		Object aaa = new Object[] { map };
		SelContext ctx = new SelContextImpl();
		ctx.setVariable("aaa", aaa);
		assertEquals("1", "hoge", exp.evaluateValue(ctx));
	}

	public void testStaticField() throws Exception {
		Expression exp =
			new SelParser("java.lang.Integer.MIN_VALUE").parseExpression();
		assertEquals(
			"1",
			new Integer(Integer.MIN_VALUE),
			exp.evaluateValue(null));
	}
	
	public void testStaticField2() throws Exception {
		Expression exp =
			new SelParser("test.org.seasar.framework.sel.parser.SelParserTest.HOGE.toUpperCase()").parseExpression();
		assertEquals("1", "HOGE", exp.evaluateValue(null));
	}

	protected void setUp() throws Exception {
	}

	protected void tearDown() throws Exception {
	}

	public static Test suite() {
		return new TestSuite(SelParserTest.class);
	}

	public static void main(String[] args) {
		junit.textui.TestRunner.main(
			new String[] { SelParserTest.class.getName()});
	}
}