/*
AppKitEx.m

Author: Makoto Kinoshita

Copyright 2004 The Shiira Project. All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted 
provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this list of conditions 
  and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice, this list of 
  conditions and the following disclaimer in the documentation and/or other materials provided 
  with the distribution.

THIS SOFTWARE IS PROVIDED BY THE SHIIRA PROJECT ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, 
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE SHIIRA PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
POSSIBILITY OF SUCH DAMAGE.
*/

#import "AppKitEx.h"

@implementation NSAttributedString (Truncate)

- (NSAttributedString*)truncateForWidth:(int)inWidth
{
    static NSString*    _horizontalEllipsis = nil;
    if (!_horizontalEllipsis) {
        unichar uni[1];
        uni[0] = 0x2026;
        _horizontalEllipsis = [[NSString stringWithCharacters:uni length:1] retain];
    }
    
    NSAttributedString *result = self;
    if ([self size].width > inWidth) {
        NSMutableAttributedString *newString;
        newString = [[NSMutableAttributedString alloc] initWithAttributedString:self];
        [newString autorelease];
        
        int curLength = [self length] - 1; // start by chopping off at least one
        
        while ([newString size].width > inWidth) {
            NSRange range;
            range = NSMakeRange(curLength - 1, 2); // replace 2 characters with '…'
            [newString replaceCharactersInRange:range withString:_horizontalEllipsis];
            curLength--;
        }
        result = newString;
    }
    return result;
}

@end

@implementation NSAttributedString (Attachement)

- (void)_removeAttachmentFrom:(NSMutableAttributedString*)attrStr
{
    // Get range of attacment
    id      attr;
    NSRange range;
    attr = [attrStr attribute:NSAttachmentAttributeName 
            atIndex:0 
            effectiveRange:&range];
    if (range.length <= 0) {
        return;
    }
    
    // Get range again
    if (!attr) {
        attr = [attrStr attribute:NSAttachmentAttributeName 
                atIndex:range.location 
                effectiveRange:&range];
        if (range.length <= 0 || !attr) {
            return;
        }
    }
    
    // Delete attachment
    [attrStr deleteCharactersInRange:range];
}

- (NSString*)stringWithoutAttachment
{
    NSMutableAttributedString*  attrStr;
    attrStr = [[NSMutableAttributedString alloc] initWithAttributedString:self];
    
    while ([attrStr containsAttachments]) {
        [self _removeAttachmentFrom:attrStr];
    }
    
    return [attrStr string];
}

@end

#pragma mark -

@implementation NSTableView (ViewHierarchy)

- (id)scrollView
{
    id  view;
    view = self;
    while (view && ![view isKindOfClass:[NSScrollView class]]) {
        view = [view superview];
    }
    return view;
}

@end

#pragma mark -

@implementation NSTextViewEx

+ (void)load
{
    [self poseAsClass:[NSTextView class]];
}

- (void)keyDown:(NSEvent*)event
{
    // For Java text area
    if (![NSStringFromClass([event class]) isEqualToString:@"SyntheticEvent"]) {
        // Get modifier flags and key code
        unsigned    modifierFlags;
        unsigned    optFlag;
        short       keyCode;
        modifierFlags = [event modifierFlags];
        optFlag = modifierFlags & NSAlternateKeyMask;
        keyCode = [event keyCode];
        
        // For opt + enter
        if (optFlag && keyCode == 0x24 /* enter */) {
            event = [NSEvent keyEventWithType:[event type] 
                    location:[event locationInWindow] 
                    modifierFlags:0 
                    timestamp:[event timestamp] 
                    windowNumber:[event windowNumber] 
                    context:[event context] 
                    characters:[event charactersIgnoringModifiers] 
                    charactersIgnoringModifiers:[event charactersIgnoringModifiers] 
                    isARepeat:[event isARepeat] 
                    keyCode:[event keyCode]];
        }
    }
    
    [super keyDown:event];
}

@end

#pragma mark -

@implementation NSView (Image)

- (NSImage*)viewImage
{
    return [self viewImageWithRect:[self frame]];
}

- (NSImage*)viewImageWithRect:(NSRect)rect bimap:(NSBitmapImageRep*)bitmapImageRep
{
    // Check argument
    if (NSIsEmptyRect(rect)) {
        return nil;
    }
    
    // Create image rep
    NSRect  offscreenRect = NSMakeRect(0, 0, 800, 800);
    
    [NSGraphicsContext saveGraphicsState];
    [NSGraphicsContext setCurrentContext:[NSGraphicsContext graphicsContextWithBitmapImageRep:bitmapImageRep]];
    
    [self drawRect:offscreenRect];
    [NSGraphicsContext restoreGraphicsState];
    
    // Create iamge
    NSImage*    image;
    image = [[NSImage alloc] initWithSize:offscreenRect.size];
    [image setFlipped:[self isFlipped]];
    [image autorelease];
    [image addRepresentation:bitmapImageRep];
    [bitmapImageRep release];
    
    return image;
}

#if 1
- (NSImage*)viewImageWithRect:(NSRect)rect
{
    // Check argument
    if (NSIsEmptyRect(rect)) {
        return nil;
    }
    
    // Create image rep
    NSBitmapImageRep*   imageRep = nil;
    
    NSRect  offscreenRect = NSMakeRect(0, 0, 800, 800);
    imageRep = [[NSBitmapImageRep alloc] initWithBitmapDataPlanes:nil pixelsWide:offscreenRect.size.width pixelsHigh:offscreenRect.size.height bitsPerSample:8 samplesPerPixel:4 hasAlpha:YES isPlanar:NO colorSpaceName:NSCalibratedRGBColorSpace bitmapFormat:0 bytesPerRow:(4 * offscreenRect.size.width) bitsPerPixel:32];
    
    [NSGraphicsContext saveGraphicsState];
    [NSGraphicsContext setCurrentContext:[NSGraphicsContext graphicsContextWithBitmapImageRep:imageRep]];
    
    [self drawRect:offscreenRect];
    [NSGraphicsContext restoreGraphicsState];
    
    // Create iamge
    NSImage*    image;
    image = [[NSImage alloc] initWithSize:offscreenRect.size];
    [image setFlipped:[self isFlipped]];
    [image autorelease];
    [image addRepresentation:imageRep];
    [imageRep release];
    
    return image;
}
#else
- (NSImage*)viewImageWithRect:(NSRect)rect
{
    // Check argument
    if (NSIsEmptyRect(rect)) {
        return nil;
    }
    
    // Create image rep
    NSBitmapImageRep*   imageRep = nil;
    
#if 1
    if ([self window]) {
        [self lockFocus];
        imageRep = [[NSBitmapImageRep alloc] initWithFocusedViewRect:NSMakeRect(0, 0, [self frame].size.width, [self frame].size.height)];
        
        // Create iamge
        NSImage*    image;
        image = [[NSImage alloc] initWithSize:[self frame].size];
        [image autorelease];
        [image addRepresentation:imageRep];
        [self unlockFocus];
        [imageRep release];
        
        return image;
    }
#else
    // For 10.4
    if ([self respondsToSelector:@selector(bitmapImageRepForCachingDisplayInRect:)]) {
        imageRep = [self bitmapImageRepForCachingDisplayInRect:[self frame]];
        [self cacheDisplayInRect:[self frame] toBitmapImageRep:imageRep];
    }
    else {
        if ([self window]) {
            [self lockFocus];
            imageRep = [[NSBitmapImageRep alloc] initWithFocusedViewRect:rect];
            [imageRep autorelease];
            [self unlockFocus];
        }
    }
#endif
    
#if 0
    if (!imageRep) {
        return nil;
    }
    
    // Create iamge
    NSImage*    image;
    image = [[NSImage alloc] initWithSize:[self frame].size];
    [image autorelease];
    [image addRepresentation:imageRep];
#if 0
    [image lockFocus];
    [imageRep drawAtPoint:NSMakePoint(-1 * rect.origin.x, -1 * rect.origin.y)];
    [image unlockFocus];
#endif
    
    return image;
#endif
return nil;
}
#endif

- (NSImageView*)viewImageView
{
    // Get image
    NSImage*    image;
    image = [self viewImage];
    if (!image) {
        return nil;
    }
    
    // Get frame
    NSRect  frame;
    frame = [self frame];
    
    // Create image view
    NSImageView*    imageView;
    imageView = [[NSImageView alloc] initWithFrame:NSMakeRect(0, 0, frame.size.width, frame.size.height)];
    [imageView autorelease];
    [imageView setImage:image];
    
    return imageView;
}

@end

#pragma mark -

@implementation NSWindowEx : NSWindow

#if 0
+ (void)load
{
    [self poseAsClass:[NSWindow class]];
}

- (void)closeWindowAction:(id)sender
{
    // Pass to delegate
    id  delegate;
    delegate = [self delegate];
    if (delegate && [delegate respondsToSelector:_cmd]) {
        [delegate performSelector:_cmd withObject:sender];
        return;
    }
    
    // Close window
    [self performClose:self];
}

- (void)closeTabAction:(id)sender
{
    // Pass to delegate
    id  delegate;
    delegate = [self delegate];
    if (delegate && [delegate respondsToSelector:_cmd]) {
        [delegate performSelector:_cmd withObject:sender];
        return;
    }
    
    // Do nothing in NSWindow
}

- (BOOL)validateMenuItem:(id<NSMenuItem>)menuItem
{
    // Get tag
    int tag;
    tag = [menuItem tag];
    
    // User defaults
    NSUserDefaults* defaults;
    defaults = [NSUserDefaults standardUserDefaults];
    
    switch (tag) {
    // File menu
    case SRCloseWindowTag: {
        // Prioritize SRMainWindowController
        id  windowController;
        windowController = [self windowController];
        if (windowController && [windowController isKindOfClass:[SRMainWindowController class]]) {
            return [windowController validateMenuItem:menuItem];
        }
        
        [menuItem setKeyEquivalent:@"w"];
        [menuItem setKeyEquivalentModifierMask:NSCommandKeyMask];
        return YES;
    }
    case SRCloseTabTag: {
        // Prioritize SRMainWindowController
        id  windowController;
        windowController = [self windowController];
        if (windowController && [windowController isKindOfClass:[SRMainWindowController class]]) {
            return [windowController validateMenuItem:menuItem];
        }
        
        [menuItem setKeyEquivalent:@""];
        return NO;
    }
    }
    
    return YES;
}
#endif

@end
