/* vim: set encoding=utf8:
 *
 * buffer.c
 *
 * This file is part of Shiki.
 *
 * Copyright(C)2006 WAKATSUKI toshihiro
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * $Id: buffer.c,v 1.17 2007/02/03 11:23:38 aloha Exp $
 */
#include"shiki.h"

/* GtkTextBuffer から，対応する ShikiBuffer 構造体を逆引き */
static gint compBuffer(gconstpointer a, gconstpointer b) {
  return ((ShikiBuffer *)a)->text_buffer == b ? 0 : b - a;
}

static GList *get_ShikiBufferListElement_By_GtkTextBuffer(GtkTextBuffer *b) {
  return g_list_find_custom(Shiki_EDITOR_BUFFER_LIST, b, compBuffer);
}

/* バッファがまだ保存されていないのに本当に終了するのかを尋ねる */
static gboolean delete_event_handler(GtkWidget *widget, GdkEvent *event, GtkTextBuffer *buffer){
  /* delete-event が発行された際，FALSE を返したらバッファは破棄される */
  return Shiki_need_buffer_save_p(buffer) && !Shiki_yes_or_no_p("バッファは変更されています．変更内容を破棄しますか ?");
}

/* バッファにテキストが挿入された */
static void insert_text_handler(GtkTextBuffer *buffer, GtkTextIter *iter, gchar *str, gint len) {
  /* Undo のための情報を記録 */
  ShikiUndoInfo *undoInfo = g_malloc(sizeof(ShikiUndoInfo));
  g_return_if_fail(undoInfo != NULL);
  undoInfo->action = SHIKI_UNDO_INSERT;
  undoInfo->str    = g_strdup(str);
  undoInfo->strlen = len;
  undoInfo->start  = gtk_text_iter_get_offset(iter);
  undoInfo->end    = undoInfo->start + undoInfo->strlen;

  if(Shiki_CURRENT_UNDO_INFO_LIST) {
    GList *p = Shiki_CURRENT_UNDO_INFO_LIST->prev;
    Shiki_CURRENT_UNDO_INFO_LIST = g_list_prepend(Shiki_CURRENT_UNDO_INFO_LIST, undoInfo);
    Shiki_CURRENT_UNDO_INFO_LIST->prev = p;
  } else
    Shiki_CURRENT_UNDO_INFO_LIST = g_list_prepend(Shiki_CURRENT_UNDO_INFO_LIST, undoInfo);
}

/* バッファからテキストが消去された */
static void delete_range_handler(GtkTextBuffer *buffer, GtkTextIter *start, GtkTextIter *end) {
  /* Undo のための情報を記録 */
  ShikiUndoInfo *undoInfo = g_malloc(sizeof(ShikiUndoInfo));
  g_return_if_fail(undoInfo != NULL);
  undoInfo->action = SHIKI_UNDO_DELETE;
  undoInfo->str    = gtk_text_buffer_get_text(buffer, start, end, FALSE);
  undoInfo->start  = gtk_text_iter_get_offset(start);
  undoInfo->end    = gtk_text_iter_get_offset(end);
  undoInfo->strlen = end - start;

  if(Shiki_CURRENT_UNDO_INFO_LIST) {
    GList *p = Shiki_CURRENT_UNDO_INFO_LIST->prev;
    Shiki_CURRENT_UNDO_INFO_LIST = g_list_prepend(Shiki_CURRENT_UNDO_INFO_LIST, undoInfo);
    Shiki_CURRENT_UNDO_INFO_LIST->prev = p;
  } else
    Shiki_CURRENT_UNDO_INFO_LIST = g_list_prepend(Shiki_CURRENT_UNDO_INFO_LIST, undoInfo);
}

void Shiki_undo() {
  GtkTextIter start, end;
  ShikiUndoInfo *undoInfo;
  GList *p;
  if(!Shiki_CURRENT_UNDO_INFO_LIST) {
    Shiki_msgbox("これ以上 Undo できません");
    return;
  }
  undoInfo = Shiki_CURRENT_UNDO_INFO_LIST->data;
  if(undoInfo->action == SHIKI_UNDO_UNDO) {
    GList *l = g_list_nth(Shiki_CURRENT_UNDO_INFO_LIST, 2);
    if(l)
      Shiki_CURRENT_UNDO_INFO_LIST = l;
    else {
      Shiki_msgbox("これ以上 Undo できません");
      return;
    }
  } else if(undoInfo->action == SHIKI_UNDO_REDO) {
    GList *l = g_list_next(Shiki_CURRENT_UNDO_INFO_LIST);
    if(l)
      Shiki_CURRENT_UNDO_INFO_LIST = l;
    else {
      Shiki_msgbox("これ以上 Undo できません");
      return;
    }
  }

  undoInfo = Shiki_CURRENT_UNDO_INFO_LIST->data;

  gtk_text_buffer_get_iter_at_offset(Shiki_CURRENT_TEXT_BUFFER, &start, undoInfo->start);
  gtk_text_buffer_get_iter_at_offset(Shiki_CURRENT_TEXT_BUFFER, &end, undoInfo->end);
 
  if(!Shiki_CURRENT_UNDO_INFO_LIST->next) {
    p = Shiki_CURRENT_UNDO_INFO_LIST;
    Shiki_CURRENT_UNDO_INFO_LIST = g_list_next(Shiki_CURRENT_UNDO_INFO_LIST);
  } else {
    Shiki_CURRENT_UNDO_INFO_LIST = g_list_next(Shiki_CURRENT_UNDO_INFO_LIST);
    p = Shiki_CURRENT_UNDO_INFO_LIST->prev;
  }

  if(undoInfo->action == SHIKI_UNDO_INSERT)
    gtk_text_buffer_delete(Shiki_CURRENT_TEXT_BUFFER, &start, &end);
  else if(undoInfo->action == SHIKI_UNDO_DELETE)
    gtk_text_buffer_insert(Shiki_CURRENT_TEXT_BUFFER, &start, undoInfo->str, -1);           

  Shiki_CURRENT_UNDO_INFO_LIST->prev = p;

  undoInfo = g_malloc(sizeof(ShikiUndoInfo));
  g_return_if_fail(undoInfo != NULL);
  undoInfo->action = SHIKI_UNDO_UNDO;

  p = Shiki_CURRENT_UNDO_INFO_LIST->prev;
  Shiki_CURRENT_UNDO_INFO_LIST = g_list_prepend(Shiki_CURRENT_UNDO_INFO_LIST, undoInfo);
  Shiki_CURRENT_UNDO_INFO_LIST->prev = p;

  { /* Debug print */
    GList *l = Shiki_CURRENT_UNDO_INFO_LIST;
    while(l->prev) l = l->prev;
    g_print("Undo : NULL -> ");
    for(; l != NULL; l = l->next) {
      if(l == Shiki_CURRENT_UNDO_INFO_LIST) 
        g_print(" | ");
      switch(((ShikiUndoInfo *)l->data)->action) {
        case SHIKI_UNDO_UNDO :
          g_print("[U] -> ");
          break;
        case SHIKI_UNDO_REDO :
          g_print("[R] -> ");
          break;
        case SHIKI_UNDO_INSERT :
          g_print("[+] -> ");
          break;
        case SHIKI_UNDO_DELETE :
          g_print("[-] -> ");
          break;
      }
    }
    g_print("NIL\n");
  }  
}

void Shiki_redo() {
  GtkTextIter start, end;
  ShikiUndoInfo *undoInfo;
  GList *p;
  if(!Shiki_CURRENT_UNDO_INFO_LIST) {
    Shiki_msgbox("これ以上 Redo できません");
    return;
  }
  undoInfo = Shiki_CURRENT_UNDO_INFO_LIST->data;
  if(undoInfo->action == SHIKI_UNDO_UNDO)
    undoInfo = g_list_nth_data(Shiki_CURRENT_UNDO_INFO_LIST, 1);
  else if(undoInfo->action == SHIKI_UNDO_REDO) {
    GList *l = g_list_nth_prev(Shiki_CURRENT_UNDO_INFO_LIST, 3);
    if(l) {
      Shiki_CURRENT_UNDO_INFO_LIST = l;
      undoInfo = g_list_nth_data(Shiki_CURRENT_UNDO_INFO_LIST, 1);
    } else {
      Shiki_msgbox("これ以上 Redo できません");
      return;
    }
  }

  gtk_text_buffer_get_iter_at_offset(Shiki_CURRENT_TEXT_BUFFER, &start, undoInfo->start);
  gtk_text_buffer_get_iter_at_offset(Shiki_CURRENT_TEXT_BUFFER, &end, undoInfo->end);

  p = Shiki_CURRENT_UNDO_INFO_LIST->prev;

  if(undoInfo->action == SHIKI_UNDO_INSERT)
    gtk_text_buffer_delete(Shiki_CURRENT_TEXT_BUFFER, &start, &end);
  else if(undoInfo->action == SHIKI_UNDO_DELETE)
    gtk_text_buffer_insert(Shiki_CURRENT_TEXT_BUFFER, &start, undoInfo->str, -1);           

  undoInfo = g_malloc(sizeof(ShikiUndoInfo));
  g_return_if_fail(undoInfo != NULL);
  undoInfo->action = SHIKI_UNDO_REDO;

  Shiki_CURRENT_UNDO_INFO_LIST = g_list_prepend(Shiki_CURRENT_UNDO_INFO_LIST, undoInfo);

  Shiki_CURRENT_UNDO_INFO_LIST->prev = p;

  { /* Debug ptirt */
    GList *l = Shiki_CURRENT_UNDO_INFO_LIST;
    while(l->prev) l = l->prev;
    g_print("Redo : NULL -> ");
    for(; l != NULL; l = l->next) {
      if(l == Shiki_CURRENT_UNDO_INFO_LIST) 
        g_print(" | ");
      switch(((ShikiUndoInfo *)l->data)->action) {
        case SHIKI_UNDO_UNDO :
          g_print("[U] -> ");
          break;
        case SHIKI_UNDO_REDO :
          g_print("[R] -> ");
          break;
        case SHIKI_UNDO_INSERT :
          g_print("[+] -> ");
          break;
        case SHIKI_UNDO_DELETE :
          g_print("[-] -> ");
          break;
      }
    }
    g_print("NIL\n");
  }  
}

/* バッファの状態に合わせてモードラインを変更 */
void Shiki_update_modeline() {
  gchar *mlf;
  ScmEvalPacket packet;
  gint result = Scm_EvalCString("(if *mode-line-format* (*mode-line-format*) \"\")", Shiki_CURRENT_BUFFER_ENV, &packet);

  if(result == -1) {
    ScmObj os = Scm_MakeOutputStringPort(TRUE);
    Scm_Write(packet.exception, os, SCM_WRITE_DISPLAY);
    mlf = Scm_GetString(SCM_STRING(Scm_GetOutputString(SCM_PORT(os))));
  } else
    mlf = Scm_GetString(SCM_STRING(packet.results[0]));

  gtk_label_set_text(GTK_LABEL(Shiki_EDITOR_MODELINE_LABEL), mlf);
}

static void cursor_moved_handler() {
  Shiki_update_modeline();
}

/* ノートブックにタブとページ (バッファ) を追加 */
GtkTextBuffer *Shiki_new_buffer_create(gchar *filename) {
  /*-------------------- 新しいタブを作る ----------------------------------*/
  /* ShikiBuffer には，タブに関連する情報が全て保持されている */
  ShikiBuffer *tabinfo  = g_malloc(sizeof(ShikiBuffer));
  tabinfo->locale        = "Gtk Default (utf8)";
  tabinfo->undoInfoList  = NULL;
  tabinfo->filename      = filename;
  tabinfo->name      = g_path_get_basename(filename);
  tabinfo->tabpage_label = g_strndup(tabinfo->name, 10);
  tabinfo->env           = Scm_MakeModule(NULL, FALSE);

  /* 環境が GC されるのを防ぐ */
  Scm_Define(SCM_CURRENT_MODULE(), SCM_SYMBOL(SCM_INTERN(tabinfo->name)), tabinfo->env);  

  /* xyzzy lisp 関数を登録 */
  Scm_Init_xyzzylisp(SCM_MODULE(tabinfo->env));

  /* スクロールウィンドウ (タブの中身の大外) を作る */
  tabinfo->tabpage = GTK_SCROLLED_WINDOW(gtk_scrolled_window_new(NULL, NULL));
  gtk_scrolled_window_set_policy (tabinfo->tabpage, GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

  /* 枠の中に格納するテキストビューワと，テキストバッファを作る */
  tabinfo->text_view = GTK_TEXT_VIEW(gtk_text_view_new());
  gtk_text_view_set_wrap_mode(tabinfo->text_view, GTK_WRAP_WORD);
  tabinfo->text_buffer = gtk_text_view_get_buffer(tabinfo->text_view);
  gtk_widget_set_size_request(GTK_WIDGET(tabinfo->text_view), 680, 700);

  gtk_container_add(GTK_CONTAINER(tabinfo->tabpage), GTK_WIDGET(tabinfo->text_view));
  g_signal_connect(tabinfo->text_buffer, "mark_set", G_CALLBACK(cursor_moved_handler), tabinfo->text_view);
  g_signal_connect(tabinfo->text_buffer, "insert-text", G_CALLBACK(insert_text_handler), NULL);
  g_signal_connect(tabinfo->text_buffer, "delete-range", G_CALLBACK(delete_range_handler), NULL);

  /* タブを削除する際，デリートハンドラを削除しておかないと警告が出るから */
  tabinfo->delete_handler_id = g_signal_connect(Shiki_EDITOR_WINDOW, "delete_event", G_CALLBACK(delete_event_handler), tabinfo->text_buffer);

  /* 様々な初期化処理 */

  /* 括弧の強調表示のためのタグを作る */
  gtk_text_buffer_create_tag(tabinfo->text_buffer, "parent_emphasis_background", "background", "green", NULL);

  /* 検索/置換の際にマッチした部分 */
  gtk_text_buffer_create_tag(tabinfo->text_buffer, "match_highlighting", "background", "pink", NULL);

  /* キーワードハイライティング用 */
  gtk_text_buffer_create_tag(tabinfo->text_buffer, "keyword_highlighting", "foreground", "blue", NULL);
  /* 関数 */
  gtk_text_buffer_create_tag(tabinfo->text_buffer, "function_highlighting", "foreground", "red", NULL);
  /* コメント */
  gtk_text_buffer_create_tag (tabinfo->text_buffer, "comment_highlighting", "foreground", "purple", NULL);
  /* 文字列 */
  gtk_text_buffer_create_tag (tabinfo->text_buffer, "string_highlighting", "foreground", "orange", NULL);
  /* タブをノートブックに登録する */
  gtk_notebook_append_page(Shiki_EDITOR_NOTEBOOK, GTK_WIDGET(tabinfo->tabpage), gtk_label_new(tabinfo->tabpage_label));
  /* 対応するタブ情報を大域テーブルに保存しておく */
  Shiki_EDITOR_BUFFER_LIST = g_list_append(Shiki_EDITOR_BUFFER_LIST, tabinfo);

  gtk_widget_show_all(GTK_WIDGET(Shiki_EDITOR_NOTEBOOK));  
  /* 開いたページに移動する */
  gtk_notebook_set_current_page(Shiki_EDITOR_NOTEBOOK, g_list_length(Shiki_EDITOR_BUFFER_LIST) - 1);
  //Shiki_CURRENT_TAB_INFO = tabinfo;

  if(-1 == Scm_EvalCString("(set! *mode-line-format* (lambda () (format #f \"--~A- ~A (Gauche Interaction) [GtkDefault (utf8)]     L~S:~S             \" (if (buffer-modified-p) \"--\" \"**\") (buffer-name (selected-buffer)) (current-line-number) (current-column))))", tabinfo->env, NULL))
    fprintf(stderr, "Fatal error : mode-line-format initialize in Shiki_new_buffer_create()");
  return tabinfo->text_buffer;
}

void Shiki_create_file_buffer(const gchar *filename) {
  gchar *text, *utf8filename = g_locale_to_utf8(filename, -1, NULL, NULL, NULL);
  GtkTextIter p;
  ScmEvalPacket packet;
  gint result;

  /* g_file_get_contents(filename, &contents, &len, NULL); */

  /* 新しいバッファを開く */
  Shiki_new_buffer_create(g_strdup(filename));
  gtk_window_set_title (GTK_WINDOW (Shiki_EDITOR_WINDOW), filename); 

  Scm_Define(SCM_MODULE(Shiki_CURRENT_BUFFER_ENV), SCM_SYMBOL(SCM_INTERN("*filename*")), SCM_MAKE_STR_COPYING(utf8filename));
  g_free(utf8filename);

  if(-1 == Scm_EvalCString("(use gauche.charconv)", Shiki_CURRENT_BUFFER_ENV, NULL))
    fprintf(stderr, "Fatal error : can't use gauche.charconv module in Shiki_create_file_buffer()");

  /* ファイルから文字列を取り出し，変換後にバッファに格納 */
  result = Scm_EvalCString("(port->string (open-input-conversion-port (open-input-file *filename*) \"*jp\" :owner? #t))", Shiki_CURRENT_BUFFER_ENV, &packet);
  if(result != -1) {
    text = Scm_GetString(SCM_STRING(packet.results[0]));
    gtk_text_buffer_set_text(Shiki_CURRENT_TEXT_BUFFER, text, -1);
  } else {
    /* open-input-conversion-port が失敗したならば，とりあえずシステム
     * ロケールに変換してみる．駄目だったら諦めて駄目元で UTF8
     */
    gchar *contents;
    gsize br, bw, len;
    GError *err = NULL;

    if(g_file_get_contents(filename, &contents, &len, NULL)) {
      if(!(text = g_locale_to_utf8(contents, -1, &br, &bw, &err)))
        gtk_text_buffer_set_text(Shiki_CURRENT_TEXT_BUFFER, text, -1); 
      else
        gtk_text_buffer_set_text(Shiki_CURRENT_TEXT_BUFFER, contents, -1); 
      g_free(contents);
    }
  }

  /* バッファ未変更に */
  gtk_text_buffer_set_modified(Shiki_CURRENT_TEXT_BUFFER, FALSE);
  /* カーソル位置を先頭に */
  gtk_text_buffer_get_start_iter(Shiki_CURRENT_TEXT_BUFFER, &p);
  gtk_text_buffer_place_cursor(Shiki_CURRENT_TEXT_BUFFER, &p);
  Shiki_update_modeline(Shiki_CURRENT_TEXT_BUFFER);
  gtk_widget_show_all(GTK_WIDGET(Shiki_EDITOR_NOTEBOOK));
}

void Shiki_open_file_dialog() {
  const gchar *filename = Shiki_file_name_dialog("ファイルを開く");

  if(!filename) return;
  Shiki_create_file_buffer(filename);
}

void Shiki_delete_buffer(GtkTextBuffer *buffer) {
  /* バッファリストから，リストの要素，バッファ，バッファの番号を逆引き */
  /* 効率が悪いが，Scheme の世界になるべく Gtk のオブジェクトを持ち込まないため */
  GList *bufListElem = get_ShikiBufferListElement_By_GtkTextBuffer(buffer);
  ShikiBuffer *tabInfo  = bufListElem->data;
  gint bufNum = g_list_position(Shiki_EDITOR_BUFFER_LIST, bufListElem);

  /* タブが 1 つしか残っていなかったら消させない */
  if(g_list_length(Shiki_EDITOR_BUFFER_LIST) == 1)
    return;

  /* Scheme 世界の束縛を絶つ */
  Scm_Define(SCM_CURRENT_MODULE(), SCM_SYMBOL(SCM_INTERN(tabInfo->name)), SCM_FALSE);

  /* デリートハンドラをエディタトップレベルのウィジットから取り除く */
  g_signal_handler_disconnect(Shiki_EDITOR_WINDOW, tabInfo->delete_handler_id);
  Shiki_EDITOR_BUFFER_LIST = g_list_delete_link(Shiki_EDITOR_BUFFER_LIST, bufListElem);  
  gtk_widget_destroy(GTK_WIDGET(tabInfo->tabpage));
  g_free(tabInfo->tabpage_label);
  g_free(tabInfo->name);
  g_free(tabInfo->filename);
  g_free(tabInfo);
  gtk_notebook_remove_page(Shiki_EDITOR_NOTEBOOK, bufNum);
  /* 強制再描画 */
  gtk_widget_queue_draw(GTK_WIDGET(Shiki_EDITOR_NOTEBOOK));
}

GtkTextBuffer *Shiki_find_buffer(const gchar *name) {
  GList *l;
  for(l = Shiki_EDITOR_BUFFER_LIST; l != NULL; l = l->next)
    if(strcmp(((ShikiBuffer *)l->data)->name, name) == 0)
      return ((ShikiBuffer *)l->data)->text_buffer;
  return NULL;
}

  gchar *Shiki_buffer_substring(gint start, gint end) {
    if(start >= end)
      return NULL;
    else {
      GtkTextIter s, e;
      gtk_text_buffer_get_iter_at_offset(Shiki_CURRENT_TEXT_BUFFER, &s, start); 
      gtk_text_buffer_get_iter_at_offset(Shiki_CURRENT_TEXT_BUFFER, &e, end);

      return gtk_text_buffer_get_text(Shiki_CURRENT_TEXT_BUFFER, &s, &e, FALSE);
    }
  }

  void Shiki_delete_region(gint start, gint end) {
    if(start >= end)
      return;
    else {
      GtkTextIter s, e;
      gtk_text_buffer_get_iter_at_offset(Shiki_CURRENT_TEXT_BUFFER, &s, start); 
      gtk_text_buffer_get_iter_at_offset(Shiki_CURRENT_TEXT_BUFFER, &e, end);

      return gtk_text_buffer_delete(Shiki_CURRENT_TEXT_BUFFER, &s, &e);
    }
  }

gint Shiki_point() {
  GtkTextIter p;
  gtk_text_buffer_get_iter_at_mark(Shiki_CURRENT_TEXT_BUFFER,&p, gtk_text_buffer_get_insert(Shiki_CURRENT_TEXT_BUFFER));
  return gtk_text_iter_get_offset(&p);
}

gint Shiki_point_max() {
  GtkTextIter p;
  gtk_text_buffer_get_end_iter(Shiki_CURRENT_TEXT_BUFFER, &p);
  return gtk_text_iter_get_offset(&p);
}

gint Shiki_point_min() {
  return 0;
}

void Shiki_goto_char(gint offset) {
  GtkTextIter p;
  gtk_text_buffer_get_iter_at_offset(Shiki_CURRENT_TEXT_BUFFER, &p, offset);
  gtk_text_buffer_place_cursor(Shiki_CURRENT_TEXT_BUFFER, &p);
}

void Shiki_forward_char() {
  GtkTextIter p;
  gtk_text_buffer_get_iter_at_mark(Shiki_CURRENT_TEXT_BUFFER,&p, gtk_text_buffer_get_insert(Shiki_CURRENT_TEXT_BUFFER));
  gtk_text_iter_forward_char(&p);
  gtk_text_buffer_place_cursor(Shiki_CURRENT_TEXT_BUFFER, &p);
}

void Shiki_backward_char() {
  GtkTextIter p;
  gtk_text_buffer_get_iter_at_mark(Shiki_CURRENT_TEXT_BUFFER,&p, gtk_text_buffer_get_insert(Shiki_CURRENT_TEXT_BUFFER));
  gtk_text_iter_backward_char(&p);
  gtk_text_buffer_place_cursor(Shiki_CURRENT_TEXT_BUFFER, &p);
}

void Shiki_goto_line(gint line) {
  GtkTextIter p;
  gtk_text_buffer_get_iter_at_line(Shiki_CURRENT_TEXT_BUFFER, &p, line);
  gtk_text_buffer_place_cursor(Shiki_CURRENT_TEXT_BUFFER, &p);  
}

void Shiki_goto_bol() {
  GtkTextIter p;
  gtk_text_buffer_get_iter_at_mark(Shiki_CURRENT_TEXT_BUFFER,&p, gtk_text_buffer_get_insert(Shiki_CURRENT_TEXT_BUFFER));
  gtk_text_buffer_get_iter_at_line_offset(Shiki_CURRENT_TEXT_BUFFER, &p, gtk_text_iter_get_line(&p), 0);
  gtk_text_buffer_place_cursor(Shiki_CURRENT_TEXT_BUFFER, &p);
}

void Shiki_goto_eol() {
  GtkTextIter p;
  gtk_text_buffer_get_iter_at_mark(Shiki_CURRENT_TEXT_BUFFER,&p, gtk_text_buffer_get_insert(Shiki_CURRENT_TEXT_BUFFER));
  gtk_text_iter_forward_to_line_end(&p);
  gtk_text_iter_backward_char(&p);
  gtk_text_buffer_place_cursor(Shiki_CURRENT_TEXT_BUFFER, &p);
}

void Shiki_forward_line(gint count) {
  GtkTextIter p;
  gint i;
  gtk_text_buffer_get_iter_at_mark(Shiki_CURRENT_TEXT_BUFFER,&p, gtk_text_buffer_get_insert(Shiki_CURRENT_TEXT_BUFFER));

  if(count >= 0) {
    for(i = count; i != 0; i--)
      gtk_text_view_forward_display_line(Shiki_CURRENT_TEXT_VIEW, &p);
  } else {
    for(i = count; i != 0; i++)
      gtk_text_view_backward_display_line(Shiki_CURRENT_TEXT_VIEW, &p);
  }
  gtk_text_buffer_place_cursor(Shiki_CURRENT_TEXT_BUFFER, &p);
}

const char *Shiki_buffer_name(GtkTextBuffer *buffer) {
  GList *l = get_ShikiBufferListElement_By_GtkTextBuffer(buffer);
  if(l)
    return ((ShikiBuffer *)(l->data))->name;
  else
    return NULL;
}

gboolean Shiki_deleted_buffer_p(GtkTextBuffer *buffer) {
  GList *l = get_ShikiBufferListElement_By_GtkTextBuffer(buffer);
  if(l)
    return FALSE;
  else
    return TRUE;
}

GtkTextBuffer *Shiki_get_next_buffer(GtkTextBuffer *buffer) {
  GList *l = get_ShikiBufferListElement_By_GtkTextBuffer(buffer);
  if(l && l->next)
    return ((ShikiBuffer *)(l->next->data))->text_buffer;
  else
    return NULL;
}

GtkTextBuffer *Shiki_get_previous_buffer(GtkTextBuffer *buffer) {
  GList *l = get_ShikiBufferListElement_By_GtkTextBuffer(buffer);
  if(l && l->prev)
    return ((ShikiBuffer *)(l->prev->data))->text_buffer;
  else
    return NULL;
}

ScmObj Shiki_buffer_list() {
  GList *l;
  GtkTextBuffer *b;
  ScmObj bl = SCM_NIL;

  for(l = Shiki_EDITOR_BUFFER_LIST; l != NULL; l = l->next) {
    b= ((ShikiBuffer *)(l->data))->text_buffer;
    bl = Scm_Cons(SHIKI_BUFFER_BOX(g_object_ref(b)), bl);
  }
  return bl;
}

void Shiki_erase_buffer(GtkTextBuffer *buffer) {
  GtkTextIter start, end;
  gtk_text_buffer_get_start_iter(buffer, &start);
  gtk_text_buffer_get_end_iter(buffer, &end);
  gtk_text_buffer_delete(buffer, &start, &end);
}

const gchar *Shiki_file_name_dialog(const gchar *msg) {

  GtkWidget *dialog = gtk_file_selection_new(msg);
  gint resp = gtk_dialog_run(GTK_DIALOG(dialog));
  const gchar *filename = NULL;

  if(resp == GTK_RESPONSE_OK)
    filename = gtk_file_selection_get_filename(GTK_FILE_SELECTION(dialog));

  gtk_widget_destroy(dialog);
  return filename;
}

gboolean Shiki_yes_or_no_p(const gchar *msg) {
  GtkWidget *dialog = gtk_message_dialog_new(GTK_WINDOW(Shiki_EDITOR_WINDOW),
      GTK_DIALOG_DESTROY_WITH_PARENT, GTK_MESSAGE_QUESTION,
      GTK_BUTTONS_YES_NO, msg);
  gint resp;
  gtk_dialog_set_default_response(GTK_DIALOG(dialog), GTK_RESPONSE_YES);
  resp = gtk_dialog_run(GTK_DIALOG(dialog));
  gtk_widget_destroy(dialog);
  if(GTK_RESPONSE_YES == resp)
    return TRUE;
  return FALSE;
}

gboolean Shiki_no_or_yes_p(const gchar *msg) {
  GtkWidget *dialog = gtk_message_dialog_new(GTK_WINDOW(Shiki_EDITOR_WINDOW),
      GTK_DIALOG_DESTROY_WITH_PARENT, GTK_MESSAGE_QUESTION,
      GTK_BUTTONS_YES_NO, msg);
  gint resp;
  gtk_dialog_set_default_response(GTK_DIALOG(dialog), GTK_RESPONSE_NO);  
  resp = gtk_dialog_run(GTK_DIALOG(dialog));
  gtk_widget_destroy(dialog);
  if(GTK_RESPONSE_YES == resp)
    return TRUE;
  return FALSE;
}

gboolean Shiki_need_buffer_save_p(GtkTextBuffer *buffer) {
  return gtk_text_buffer_get_modified(buffer);
}

/* 確認あり */
void Shiki_kill_buffer(GtkTextBuffer *buffer) {
  if(!Shiki_need_buffer_save_p(buffer) || Shiki_yes_or_no_p("バッファは変更されています．変更内容を破棄しますか ?"))
    Shiki_delete_buffer(buffer);
}

void Shiki_msgbox(const gchar *msg) {
  GtkWidget *dialog;
  dialog = gtk_message_dialog_new(GTK_WINDOW(Shiki_EDITOR_WINDOW),
      GTK_DIALOG_DESTROY_WITH_PARENT, GTK_MESSAGE_INFO, GTK_BUTTONS_OK,msg);
  gtk_dialog_run(GTK_DIALOG(dialog));
  gtk_widget_destroy(dialog);
}

/* GtkTextCharPredicate */
static gboolean is_kakko_or_kokka(gunichar ch, gpointer p) {
  return ch == '(' || ch == ')';
}
static gboolean is_kakko(gunichar ch, gpointer p) {return ch == '(';}
static gboolean is_kokka(gunichar ch, gpointer p) {return ch == ')';}
static gboolean search_sexp_kokka(GtkTextIter *end) {
  gint nest_level = 0;

  /* 対応する ')' を探す */
  while(1) {
    if(!gtk_text_iter_forward_find_char(end, is_kakko_or_kokka, NULL, NULL))
      return FALSE;

    if(gtk_text_iter_get_char(end) == '(')
      nest_level++;
    else {
      if(!nest_level)
        break;
      else
        nest_level--;
    }
  }
  return TRUE;
}


/* ')' に対応する '(' までの文字列 (S 式) を切り出す */
static gboolean search_last_sexp_kakko(GtkTextIter *start) {
  gint nest_level = 0;
  /* ネストレベルを計算しながら ')' を探す */
  while(1) {
    if(!gtk_text_iter_backward_find_char(start, is_kakko_or_kokka, NULL, NULL))
      return FALSE;

    if(gtk_text_iter_get_char(start) == ')')
      nest_level++;
    else {
      if(!nest_level)
        break;
      else
        nest_level--;
    }
  }
  return TRUE;
}

/* カーソル以降の '(' に対応する ')' までの文字列 (S 式) を切り出す */
static gboolean search_sexp(GtkTextIter *start, GtkTextIter *end) {

  /* カーソルの位置を取得 */
  gtk_text_buffer_get_iter_at_mark(Shiki_CURRENT_TEXT_BUFFER, start, gtk_text_buffer_get_insert(Shiki_CURRENT_TEXT_BUFFER));

  if(gtk_text_iter_get_char(start) != '(')
    gtk_text_iter_forward_find_char(start, is_kakko, NULL, NULL);

  *end = *start;

  /* カーソル位置の前にある S 式を切り出す */
  if(!search_sexp_kokka(end)) return FALSE;
  gtk_text_iter_forward_char(end);
  return TRUE;
}

/* カーソル以前の ')' に対応する '(' までの文字列 (S 式) を切り出す */
static gboolean search_last_sexp(GtkTextIter *start, GtkTextIter *end) {

  /* カーソルの位置を取得 */
  gtk_text_buffer_get_iter_at_mark(Shiki_CURRENT_TEXT_BUFFER, end, gtk_text_buffer_get_insert(Shiki_CURRENT_TEXT_BUFFER));

  gtk_text_iter_backward_char(end);

  if(gtk_text_iter_get_char(end) != ')')
    gtk_text_iter_backward_find_char(end, is_kokka, NULL, NULL);
  *start = *end;
  gtk_text_iter_forward_char(end);

  /* カーソル位置の前にある S 式を切り出す */
  if(!search_last_sexp_kakko(start)) return FALSE;

  return TRUE;
}

/* gauche を起動して文字列を評価 */
static gchar *eval_cstring_by_gauche(gchar *s) {
  gchar *msg;
  ScmEvalPacket packet;
  /* 出力文字列ポート開く */
  ScmObj os = Scm_MakeOutputStringPort(TRUE);
  gint i, result = Scm_EvalCString(s, SCM_OBJ(Shiki_CURRENT_BUFFER_ENV), &packet);
  
  if(result == -1)
    Scm_Write(packet.exception, os, SCM_WRITE_DISPLAY);
  else
    for(i = 0; i < packet.numResults; i++)
      Scm_Printf(SCM_PORT(os), "%S\n", packet.results[i]);

  msg = Scm_GetString(SCM_STRING(Scm_GetOutputString(SCM_PORT(os))));
  /* ポート閉じる */
  Scm_ClosePort(SCM_PORT(os));

  return msg;
}

void Shiki_load_file(const gchar *filename) {
  gchar *result, *utf8filename = g_filename_to_utf8(filename, -1, NULL, NULL, NULL);
  GtkTextIter p;

  Scm_Define(SCM_MODULE(Shiki_CURRENT_BUFFER_ENV), SCM_SYMBOL(SCM_INTERN("*filename*")), SCM_MAKE_STR_COPYING(utf8filename));
  result = eval_cstring_by_gauche("(load *filename*)");

  gtk_text_buffer_get_end_iter(Shiki_CURRENT_TEXT_BUFFER, &p);
  gtk_text_buffer_insert(Shiki_CURRENT_TEXT_BUFFER, &p, "\n", -1);
  gtk_text_buffer_insert(Shiki_CURRENT_TEXT_BUFFER, &p, result, -1);
}

void Shiki_eval_expression() {

  gchar *code, *result;
  GtkTextIter start, end;

  if(!search_sexp(&start, &end)) return;

  code = gtk_text_buffer_get_text(Shiki_CURRENT_TEXT_BUFFER, &start, &end, FALSE);
  result = eval_cstring_by_gauche(code);
  g_free(code);

  gtk_text_buffer_get_iter_at_mark(Shiki_CURRENT_TEXT_BUFFER, &end, gtk_text_buffer_get_insert(Shiki_CURRENT_TEXT_BUFFER));

  gtk_text_buffer_insert(Shiki_CURRENT_TEXT_BUFFER, &end, "\n", -1);
  gtk_text_buffer_insert(Shiki_CURRENT_TEXT_BUFFER, &end, result, -1);
  gtk_text_buffer_insert(Shiki_CURRENT_TEXT_BUFFER, &end, "\n", -1);
}

void Shiki_eval_last_sexp() {
  gchar *code, *result;
  GtkTextIter start, end;

  if(!search_last_sexp(&start, &end)) return;

  code = gtk_text_buffer_get_text(Shiki_CURRENT_TEXT_BUFFER, &start, &end, FALSE);
  result = eval_cstring_by_gauche(code);
  g_free(code);

  gtk_text_buffer_get_iter_at_mark(Shiki_CURRENT_TEXT_BUFFER, &end, gtk_text_buffer_get_insert(Shiki_CURRENT_TEXT_BUFFER));

  gtk_text_buffer_insert(Shiki_CURRENT_TEXT_BUFFER, &end, "\n", -1);
  gtk_text_buffer_insert(Shiki_CURRENT_TEXT_BUFFER, &end, result, -1);
  gtk_text_buffer_insert(Shiki_CURRENT_TEXT_BUFFER, &end, "\n", -1);
}

typedef enum {
  SHIKI_SEARCH_FORWARD,
  SHIKI_SEARCH_BACKWARD
} ShikiSearchDirection;

static struct {
  GtkWidget *input;
  gboolean ci;
  gboolean word;
  gboolean regexp;
  gboolean escape;
  gboolean loop;
} ShikiSearchBufferInfo;

gboolean Shiki_search_string(const gchar *pattern, gboolean no_dup,
    ShikiSearchDirection direction) {
  GtkTextIter p, match_start, match_end, start, end;
  gboolean result;

  gtk_text_buffer_get_start_iter(Shiki_CURRENT_TEXT_BUFFER, &start);
  gtk_text_buffer_get_end_iter(Shiki_CURRENT_TEXT_BUFFER, &end);

  gtk_text_buffer_get_iter_at_mark(Shiki_CURRENT_TEXT_BUFFER, &p, gtk_text_buffer_get_insert(Shiki_CURRENT_TEXT_BUFFER));

  if(direction == SHIKI_SEARCH_FORWARD) {
    if(no_dup)
      gtk_text_iter_forward_char(&p);

    result = gtk_text_iter_forward_search(&p, pattern,
        GTK_TEXT_SEARCH_TEXT_ONLY, &match_start, &match_end, &end);
  } else {
    if(no_dup)
      gtk_text_iter_backward_char(&p);

    result = gtk_text_iter_backward_search(&p, pattern,
        GTK_TEXT_SEARCH_TEXT_ONLY, &match_start, &match_end, &start);
  }

  if(result) {
    gtk_text_buffer_remove_tag_by_name(Shiki_CURRENT_TEXT_BUFFER, "match_highlighting", &start, &end);
    gtk_text_buffer_apply_tag_by_name(Shiki_CURRENT_TEXT_BUFFER, "match_highlighting", &match_start, &match_end);
    gtk_text_buffer_place_cursor(Shiki_CURRENT_TEXT_BUFFER, &match_start);
    gtk_text_view_scroll_to_iter(Shiki_CURRENT_TEXT_VIEW,  &match_start, 
        0.0, FALSE, FALSE, FALSE);
  }
  return result;
}

static void destroy_handler(GtkWidget *button, GtkWidget *widget) {gtk_widget_destroy(widget);}

static void toggled_handler(GtkToggleButton *togglebutton, gboolean *flag) {
  *flag = !*flag;
}

static void search_forward_handler() {
  Shiki_search_string(gtk_entry_get_text(GTK_ENTRY(ShikiSearchBufferInfo.input)), TRUE, SHIKI_SEARCH_FORWARD);
}
static void search_backward_handler() {
  Shiki_search_string(gtk_entry_get_text(GTK_ENTRY(ShikiSearchBufferInfo.input)), TRUE, SHIKI_SEARCH_BACKWARD);
}

void Shiki_search_buffer() {
  static GtkWidget *input = NULL;
  GtkWidget *dialog = gtk_dialog_new_with_buttons ("文字列の検索", GTK_WINDOW(Shiki_EDITOR_WINDOW), GTK_DIALOG_DESTROY_WITH_PARENT, NULL);
  GtkWidget *table  = gtk_table_new(6, 3, FALSE);
  GtkWidget *label  = gtk_label_new("検索 : ");
  GtkWidget *check1 = gtk_check_button_new_with_label("大文字小文字を区別する");
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check1), TRUE);
  GtkWidget *check2 = gtk_check_button_new_with_label("単語単位で検索する");
  GtkWidget *check3 = gtk_check_button_new_with_label("正規表現");
  GtkWidget *check4 = gtk_check_button_new_with_label("エスケープシーケンスを理解する");
  GtkWidget *check5 = gtk_check_button_new_with_label("見つからなければ戻って");
  GtkWidget *prev   = gtk_button_new_with_label ("上検索");
  g_signal_connect (prev, "clicked", G_CALLBACK(search_backward_handler), NULL);  
  GtkWidget *next   = gtk_button_new_with_label ("下検索");
  g_signal_connect (next, "clicked", G_CALLBACK(search_forward_handler), NULL);
  GtkWidget *cancel = gtk_button_new_with_label ("キャンセル");

  if(!input)
    ShikiSearchBufferInfo.input = input = g_object_ref(gtk_entry_new());
  ShikiSearchBufferInfo.ci  =
    ShikiSearchBufferInfo.word  =
    ShikiSearchBufferInfo.regexp  =
    ShikiSearchBufferInfo.escape  =
    ShikiSearchBufferInfo.loop = FALSE;

  g_signal_connect (check1, "toggled", G_CALLBACK (toggled_handler), &(ShikiSearchBufferInfo.ci));
  g_signal_connect (check2, "toggled", G_CALLBACK (toggled_handler), &(ShikiSearchBufferInfo.word));
  g_signal_connect (check3, "toggled", G_CALLBACK (toggled_handler), &(ShikiSearchBufferInfo.regexp));
  g_signal_connect (check4, "toggled", G_CALLBACK (toggled_handler), &(ShikiSearchBufferInfo.escape));
  g_signal_connect (check5, "toggled", G_CALLBACK (toggled_handler), &(ShikiSearchBufferInfo.loop));

  g_signal_connect (G_OBJECT(dialog), "delete_event", G_CALLBACK(gtk_widget_destroy), NULL);
  g_signal_connect (G_OBJECT(cancel), "clicked", G_CALLBACK(destroy_handler), dialog);
  gtk_table_set_row_spacings(GTK_TABLE(table), 10);
  gtk_table_set_col_spacings(GTK_TABLE(table), 10);
  gtk_container_border_width (GTK_CONTAINER (dialog), 10);
  gtk_table_attach_defaults (GTK_TABLE(table), label,  0, 1, 0, 1);
  gtk_table_attach_defaults (GTK_TABLE(table), input,  1, 2, 0, 1);
  gtk_table_attach_defaults (GTK_TABLE(table), prev,   2, 3, 0, 1);
  gtk_table_attach_defaults (GTK_TABLE(table), check1, 1, 2, 1, 2);
  gtk_table_attach_defaults (GTK_TABLE(table), check2, 1, 2, 2, 3);
  gtk_table_attach_defaults (GTK_TABLE(table), check3, 1, 2, 3, 4);
  gtk_table_attach_defaults (GTK_TABLE(table), check4, 1, 2, 4, 5);
  gtk_table_attach_defaults (GTK_TABLE(table), check5, 1, 2, 5, 6);
  gtk_table_attach_defaults (GTK_TABLE(table), next,   2, 3, 1, 2);
  gtk_table_attach_defaults (GTK_TABLE(table), cancel, 2, 3, 2, 3);
  gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), table);
  gtk_widget_show_all(table);
  gtk_dialog_run(GTK_DIALOG(dialog));
}

static struct {
  GtkWidget *find;
  GtkWidget *replace;
  gboolean ci;
  gboolean word;
  gboolean regexp;
  gboolean escape;
  gboolean from_first;
} ShikiReplaceBufferInfo;

gboolean Shiki_replace_string(const gchar *find, const gchar *replace, gboolean no_dup, gboolean interactive_p, gboolean from_first_p) {
  GtkTextIter start, end, match_start, match_end;
  gboolean result = FALSE;

  if(from_first_p)
    gtk_text_buffer_get_start_iter(Shiki_CURRENT_TEXT_BUFFER, &start);
  else
    gtk_text_buffer_get_iter_at_mark(Shiki_CURRENT_TEXT_BUFFER, &start, gtk_text_buffer_get_insert(Shiki_CURRENT_TEXT_BUFFER));

  gtk_text_buffer_get_end_iter(Shiki_CURRENT_TEXT_BUFFER, &end);

  while((result = gtk_text_iter_forward_search(&start, find,
          GTK_TEXT_SEARCH_TEXT_ONLY, &match_start, &match_end, &end))) {

    gtk_text_buffer_remove_tag_by_name(Shiki_CURRENT_TEXT_BUFFER, "match_highlighting", &start, &end);
    gtk_text_buffer_apply_tag_by_name(Shiki_CURRENT_TEXT_BUFFER, "match_highlighting", &match_start, &match_end);
    gtk_text_buffer_place_cursor(Shiki_CURRENT_TEXT_BUFFER, &match_start);
    gtk_text_view_scroll_to_iter(Shiki_CURRENT_TEXT_VIEW,  &match_start, 
        0.0, FALSE, FALSE, FALSE);
    if(!interactive_p
        ||
        (interactive_p && Shiki_yes_or_no_p("置換しますか ?"))) {
      gtk_text_buffer_delete(Shiki_CURRENT_TEXT_BUFFER, &match_start, &match_end);
      gtk_text_buffer_insert_at_cursor(Shiki_CURRENT_TEXT_BUFFER, replace, -1);
    }
    gtk_text_buffer_get_iter_at_mark(Shiki_CURRENT_TEXT_BUFFER, &start, gtk_text_buffer_get_insert(Shiki_CURRENT_TEXT_BUFFER));
    gtk_text_iter_forward_char(&start);
    gtk_text_buffer_get_end_iter(Shiki_CURRENT_TEXT_BUFFER, &end);
  }

  return result;
}

static void replace_interactive_handler() {
  Shiki_replace_string(gtk_entry_get_text(GTK_ENTRY(ShikiReplaceBufferInfo.find)), gtk_entry_get_text(GTK_ENTRY(ShikiReplaceBufferInfo.replace)), TRUE, TRUE, ShikiReplaceBufferInfo.from_first);  
}
static void replace_all_handler() {
  Shiki_replace_string(gtk_entry_get_text(GTK_ENTRY(ShikiReplaceBufferInfo.find)), gtk_entry_get_text(GTK_ENTRY(ShikiReplaceBufferInfo.replace)), TRUE, FALSE, ShikiReplaceBufferInfo.from_first);
}

void Shiki_replace_buffer() {
  static GtkWidget *find = NULL;
  static GtkWidget *replace = NULL;
  GtkWidget *dialog = gtk_dialog_new_with_buttons ("文字列の置換", GTK_WINDOW(Shiki_EDITOR_WINDOW), GTK_DIALOG_DESTROY_WITH_PARENT, NULL);
  GtkWidget *table       = gtk_table_new(7, 3, FALSE);
  GtkWidget *find_label  = gtk_label_new("検索 : ");
  GtkWidget *rep_label   = gtk_label_new("置換 : ");
  GtkWidget *check1      = gtk_check_button_new_with_label("大文字小文字を区別する");
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check1), TRUE);
  GtkWidget *check2 = gtk_check_button_new_with_label("単語単位で検索する");
  GtkWidget *check3 = gtk_check_button_new_with_label("正規表現");
  GtkWidget *check4 = gtk_check_button_new_with_label("エスケープシーケンスを理解する");
  GtkWidget *check5 = gtk_check_button_new_with_label("バッファの先頭から");
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check5), TRUE);  
  GtkWidget *interactive   = gtk_button_new_with_label ("確認あり");
  g_signal_connect (interactive, "clicked", G_CALLBACK(replace_interactive_handler), NULL); 
  GtkWidget *all   = gtk_button_new_with_label ("全て置換");
  g_signal_connect (all, "clicked", G_CALLBACK(replace_all_handler), NULL); 
  GtkWidget *cancel = gtk_button_new_with_label ("キャンセル");

  if(!find && !replace) {
    ShikiReplaceBufferInfo.find = find = g_object_ref(gtk_entry_new());
    ShikiReplaceBufferInfo.replace = replace = g_object_ref(gtk_entry_new());
  }

  ShikiReplaceBufferInfo.ci  =
    ShikiReplaceBufferInfo.word  =
    ShikiReplaceBufferInfo.regexp  =
    ShikiReplaceBufferInfo.escape  = FALSE;
  ShikiReplaceBufferInfo.from_first = TRUE;

  g_signal_connect (check1, "toggled", G_CALLBACK (toggled_handler), &(ShikiReplaceBufferInfo.ci));
  g_signal_connect (check2, "toggled", G_CALLBACK (toggled_handler), &(ShikiReplaceBufferInfo.word));
  g_signal_connect (check3, "toggled", G_CALLBACK (toggled_handler), &(ShikiReplaceBufferInfo.regexp));
  g_signal_connect (check4, "toggled", G_CALLBACK (toggled_handler), &(ShikiReplaceBufferInfo.escape));
  g_signal_connect (check5, "toggled", G_CALLBACK (toggled_handler), &(ShikiReplaceBufferInfo.from_first));

  g_signal_connect (G_OBJECT(dialog), "delete_event", G_CALLBACK(gtk_widget_destroy), NULL);
  g_signal_connect (G_OBJECT(cancel), "clicked", G_CALLBACK(destroy_handler), dialog);
  gtk_table_set_row_spacings(GTK_TABLE(table), 10);
  gtk_table_set_col_spacings(GTK_TABLE(table), 10);
  gtk_container_border_width (GTK_CONTAINER (dialog), 10);

  gtk_table_attach_defaults (GTK_TABLE(table), find_label,  0, 1, 0, 1);
  gtk_table_attach_defaults (GTK_TABLE(table), find,        1, 2, 0, 1);
  gtk_table_attach_defaults (GTK_TABLE(table), interactive, 2, 3, 0, 1);

  gtk_table_attach_defaults (GTK_TABLE(table), rep_label,   0, 1, 1, 2);
  gtk_table_attach_defaults (GTK_TABLE(table), replace,     1, 2, 1, 2);
  gtk_table_attach_defaults (GTK_TABLE(table), all,         2, 3, 1, 2);

  gtk_table_attach_defaults (GTK_TABLE(table), check1, 1, 2, 2, 3);
  gtk_table_attach_defaults (GTK_TABLE(table), cancel, 2, 3, 2, 3);

  gtk_table_attach_defaults (GTK_TABLE(table), check2, 1, 2, 3, 4);
  gtk_table_attach_defaults (GTK_TABLE(table), check3, 1, 2, 4, 5);
  gtk_table_attach_defaults (GTK_TABLE(table), check4, 1, 2, 5, 6);
  gtk_table_attach_defaults (GTK_TABLE(table), check5, 1, 2, 6, 7);

  gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), table);
  gtk_widget_show_all(table);
  gtk_dialog_run(GTK_DIALOG(dialog));
}


