/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.ginkgo.nodes;

import shohaku.core.helpers.Eval;
import shohaku.ginkgo.GinkgoException;
import shohaku.ginkgo.Node;

/**
 * デフォルトノードを構成するための機能を提供します。 <br>
 * このクラスは<code>NodeCompositeRule.getFeature(Class)</code>から<code>NodeCompositeFeature.class</code>
 * をキーとして取得し各ノードが利用します。 <br>
 * デフォルトノードを使用する場合はこのクラスまたはサブクラスが必ず登録されている必要があります。 <br>
 * サブクラスを定義して登録する事でデフォルトノードの振る舞いを変更する事が出来ます。
 */
public class NodeCompositeFeature {

    /**
     * 指定された名前を持つ属性の値を参照構文として解析して参照値を返却します。 <br>
     * 指定された名前の属性又は属性値の示す参照値が存在しない場合は<code>null</code>が返されます。
     * 
     * @param node
     *            ノードのコンテキスト情報
     * @return 指定された名前の属性値が示す参照値
     */
    public Class getTextToClass(Node node) {
        String className = node.getText();
        if (Eval.isBlank(className)) {
            return null;
        }
        return NodeCompositeUtilities.loadClass(node.getContext(), className);
    }

    /**
     * 指定された名前を持つ属性の値を参照構文として解析して参照値を返却します。 <br>
     * 指定された名前の属性又は属性値の示す参照値が存在しない場合は<code>null</code>が返されます。
     * 
     * @param node
     *            ノードのコンテキスト情報
     * @param name
     *            属性名
     * @return 指定された名前の属性値が示す参照値
     */
    public Class getAttributeToClass(Node node, String name) {
        String className = node.getAttribute(name);
        if (Eval.isBlank(className)) {
            return null;
        }
        return NodeCompositeUtilities.loadClass(node.getContext(), className);
    }

    /**
     * 指定された名前を持つ属性の値を参照構文として解析して参照値を返却します。 <br>
     * 指定された名前の属性又は属性値の示す参照値が存在しない場合は<code>defaultValue</code>が返されます。
     * 
     * @param node
     *            ノードのコンテキスト情報
     * @param name
     *            属性名
     * @param defaultValue
     *            指定の属性が存在しない場合に返却される値
     * @return 指定された名前の属性値が示す参照値又は<code>defaultValue</code>
     */
    public Class getAttributeToClass(Node node, String name, Class defaultValue) {
        Class o = getAttributeToClass(node, name);
        if (null == o) {
            return defaultValue;
        }
        return o;
    }

    /**
     * 指定された名前を持つ属性の値を参照構文として解析して参照値を返却します。 <br>
     * 指定された名前の属性又は属性値の示す参照値が存在しない場合は<code>null</code>が返されます。
     * 
     * @param node
     *            ノードのコンテキスト情報
     * @return 指定された名前の属性値が示す参照値
     */
    public Object getTextRef(Node node) {
        String refId = node.getText();
        return getReferenceValue(node, refId);
    }

    /**
     * 指定された名前を持つ属性の値を参照構文として解析して参照値を返却します。 <br>
     * 指定された名前の属性又は属性値の示す参照値が存在しない場合は<code>null</code>が返されます。
     * 
     * @param node
     *            ノードのコンテキスト情報
     * @param name
     *            属性名
     * @return 指定された名前の属性値が示す参照値
     */
    public Object getAttributeRef(Node node, String name) {
        String refId = node.getAttribute(name);
        return getReferenceValue(node, refId);
    }

    /**
     * 指定された名前を持つ属性の値を参照構文として解析して参照値を返却します。 <br>
     * 指定された名前の属性又は属性値の示す参照値が存在しない場合は<code>defaultValue</code>が返されます。
     * 
     * @param node
     *            ノード
     * @param name
     *            属性名
     * @param defaultValue
     *            指定の属性が存在しない場合に返却される値
     * @return 指定された名前の属性値が示す参照値又は<code>defaultValue</code>
     */
    public Object getAttributeRef(Node node, String name, Object defaultValue) {
        Object o = getAttributeRef(node, name);
        if (null == o) {
            return defaultValue;
        }
        return o;
    }

    /**
     * 指定された識別子に対応するノードの値、又はコンテキスト属性を返却します。<br>
     * 識別子に対応するノードの値が存在しない場合にコンテキスト属性を検索し双方に存在しない場合は <code>null</code> を返却します。
     * 
     * @param node
     *            ノード
     * @param id
     *            識別子
     * @return 識別子が示すノードの値、又はコンテキスト属性
     */
    public Object getReferenceValue(Node node, String id) {
        return NodeCompositeUtilities.getReferenceValue(node.getContext(), id);
    }

    /**
     * 指定された識別子に対応するノードの値、又はコンテキスト属性が存在するか検証します。
     * 
     * @param node
     *            ノード
     * @param id
     *            識別子
     * @return 識別子が示すノードの値、又はコンテキスト属性が存在する場合 <code>true</code>
     */
    public boolean isReferenceContainsName(Node node, String id) {
        return NodeCompositeUtilities.isReferenceContainsName(node.getContext(), id);
    }

    /**
     * 指定された値をCEL式として解釈し、その結果の値を返却します。
     * 
     * @param node
     *            ノード
     * @param value
     *            解析する文字列
     * @return CEL式の実行結果の値
     */
    public Object getCELValue(Node node, String value) {
        return NodeCompositeUtilities.toCELValue(node.getContext(), value);
    }

    /**
     * テキスト情報を評価して返却します。
     * 
     * @param node
     *            ノード
     * @return テキスト情報
     */
    public String getTextValue(Node node) {
        return NodeCompositeUtilities.getTextValueToString(node.getContext());
    }

    /**
     * 文字列に対応するBOOL値を返却します。
     * 
     * @param value
     *            解析する文字列
     * @return BOOL型のTRUEを示す文字列表記の場合のみ <code>true</code>
     * @throws GinkgoException
     *             BOOL値として解析出来ない場合
     */
    public boolean getBoolean(String value) {
        if (!NodeCompositeUtilities.isBoolean(value)) {
            throw new GinkgoException("illegal format of Boolean. " + value);
        }
        return NodeCompositeUtilities.getBoolean(value);
    }

    /**
     * 指定された数値文字列をJavaで解析可能な数値文字列に変換して返却します。
     * 
     * @param value
     *            解析する文字列
     * @return Javaで解析可能な数値文字列
     */
    public String toNumberString(CharSequence value) {
        return NodeCompositeUtilities.toJavaNumberString(value);
    }

}
