require 'test_helper'

class UsersControllerTest < ActionController::TestCase
  setup do
    @input_attributes = {
      email: "new_user@mail.address.com",
      password: "private",
      password_confirmation: "private",
    }

    @user = users(:one)
  end

  test "should get new" do
    get :new
    assert_response :success
  end

  test "should create user" do
    assert_difference('User.count') do
      post :create, user: @input_attributes
    end
    assert_redirected_to login_url
  end

  test "should generate handle from email" do
    assert_difference('User.count') do
      post :create, user: @input_attributes
    end
    assert_redirected_to login_url
    created_user = assigns(:user)
    assert_equal 'new_user', created_user.handle
  end

  test "should fail creating user (no email)" do
    @input_attributes[:email] = ""
    assert_no_difference('User.count') do
      post :create, user: @input_attributes
    end
    assert_template 'new'
  end

  test "should fail creating user (no password)" do
    @input_attributes[:password] = ""
    @input_attributes[:password_confirmation] = ""
    assert_no_difference('User.count') do
      post :create, user: @input_attributes
    end
    assert_template 'new'
  end

  test "should fail creating user (invalid email)" do
    @input_attributes[:email] = "no_atmark_email_address"
    assert_no_difference('User.count') do
      post :create, user: @input_attributes
    end
    assert_template 'new'
  end

  test "should fail creating user (password confirmation fail)" do
    @input_attributes[:password_confirmation] = "wrong"
    assert_no_difference('User.count') do
      post :create, user: @input_attributes
    end
    assert_template 'new'
  end

  test "should fail creating user (already user created)" do
    @input_attributes[:email] = "someone@mail.address"
    assert_no_difference('User.count') do
      post :create, user: @input_attributes
    end
    assert_template 'new'
  end

  test "should get edit" do
    session[:user_id] = @user.id
    get :edit, id: @user.to_param
    assert_response :success
  end

  test "should update handle" do
    session[:user_id] = @user.id
    @user.handle = 'nobody'
    user_attributes = @user.attributes
    user_attributes.delete('is_resolved_show')

    put :update, id: @user.id, user: user_attributes
    assert_redirected_to edit_user_path(assigns(:user))
  end

  test "should fail updating handle (no handle)" do
    session[:user_id] = @user.id
    @user.handle = ''
    user_attributes = @user.attributes
    user_attributes.delete('is_resolved_show')
    user_attributes.delete('is_assigned_show')

    put :update, id: @user.id, user: user_attributes
    assert_response :success
    assert_select "#user_handle" do
      assert_select "[value=?]", 'someone'
    end
  end
end
