

--
-- Copyright (C) 2022  <fastrgv@gmail.com>
--
-- This program is free software: you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation, either version 3 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You may read the full text of the GNU General Public License
-- at <http://www.gnu.org/licenses/>.
--


-- Breadth First Search block slider puzzle solver...
-- a brute-force solver for *.blok
--
-- Uses a splaytree, to test whether a given config was seen before.  
-- Extremely fast access.
--
-- Uses a fullproof Key-type, for which we must define
-- operators "<" and ">".









with splaypq0;
with splaytree;
with text_io;

with Ada.Strings.Unbounded;
with Ada.Strings.Unbounded.Text_IO;

with ada.command_line;
with ada.calendar;


package body bfs26 is


procedure bfs (
	infilname: unbounded_string;
	solutionPath : out unbounded_string
) is



	use Ada.Strings.Unbounded;
	use Ada.Strings.Unbounded.Text_IO;

	use text_io;


	package myint_io is new text_io.integer_io(integer);
	package myfloat_io is new text_io.float_io(float);


	procedure myassert( 
		condition : boolean;  
		flag: integer:=0;
		msg: string := ""
		) is
	begin
	  if condition=false then
			put("ASSERTION Failed!  ");
			if flag /= 0 then
				put( "@ " & integer'image(flag) &" : " );
			end if;
			put_line(msg);
			new_line;
			raise program_error;
	  end if;
	end myassert;


--------------- begin types for hashtable --------------------------

type ubyte is range 0..255; -- 2**8-1 (1-byte)
type ushort is range 0..65_535; -- 2**16-1 (2-bytes)

subtype azrange is integer range 0..27;
type keytype is array(azrange) of azrange;


	type hashrectype is
	record
		tchr : character;
		tsel : ubyte;
		prevkey : keytype;
	end record;



	function "<" (k1, k2: in keytype ) return boolean is
	begin

		for i in azrange loop
			if k1(i)<k2(i) then return true;
			elsif k1(i)>k2(i) then return false;
			end if;
		end loop;
		return false;

	end "<";

	function ">" (k1, k2: in keytype ) return boolean is
	begin

		for i in azrange loop
			if k1(i)>k2(i) then return true;
			elsif k1(i)<k2(i) then return false;
			end if;
		end loop;
		return false;

	end ">";

	package mytree is new splaytree( keytype, hashrectype, "<", ">" );
	explored : mytree.treetype;
	estatus : mytree.statustype; -- Ok, found, ...


	package mypq is new splaypq0( keytype, hashrectype, "<", ">" );
	frontier : mypq.listtype;
	fstatus : mypq.statustype; -- Ok, found, ...



--------------- end types for hashtable --------------------------






-- 1<=r<=3, 1<=c<=3, 1<=l<=3
function endx(r,c,l : ushort) return integer is -- returns 1..27
begin
	return  integer( (r-1)*9 +(c-1)*3+(l-1) +1 );
end endx;

-- in rufascube the indx() ftn = this endx() ftn ...
--
-- and the permute array perm(r,c,l) = endx(r,c,l)
-- at the beginning and in solved position,
-- where indx(2,2,2)=14 [symbol(14)=" "], and
-- indx(3,3,3)=27 [symbol(27)="z"]
-- indx(1,1,1)= 1 [symbol(1)="a"]
-- indx(1,1,2)= 2 [symbol(2)="b"]
-- etc.












	grow,gcol,glev : array(azrange) of float;

	winner  : boolean := false;

	nrow,ncol,nlev,
	dblk, nblk, gblk : integer;

	-- these arrays track the current positions of
	-- each of 26 cubelets + 1 blank at index=27:
	rowcen0, colcen0, levcen0,
	rowcen, colcen, levcen : array(azrange) of float;

	idchar : array(azrange) of character := (others=>' ');

	blank: integer;

	depth: integer := 0;





	trailmax: constant integer := 3000; --max #moves to solve
	ntrail : integer := 0;
	trailsel : array(1..trailmax) of integer := (others=>0);
	trailchr : array(1..trailmax) of character := (others=>'X');

	trailenc : array(1..trailmax) of keytype;


-- procedure to print out the solution path;
--
procedure dump is
--azrange: 0..27
	letters: array(azrange) of character :=
		('0','a','b','c','d','e','f','g','h','i','j','k','l','m',
		     'n','o','p','q','r','s','t','u','v','w','x','y','z',' ');

begin
-- note that trailchr alone has enough info to define soln
	set_unbounded_string(solutionPath, "");
	for i in 1..ntrail loop
		append(solutionPath,
			letters(trailsel(i)) &"-"& trailchr(i)
		);
	end loop;

end dump;





--sum of manhattan 3D distances between
--each cubelet and its goal:
function dist2sol return integer is
	d,dr,dc,dl: integer := 0;
begin
	for g in 1..26 loop
		dr:=integer( rowcen(g) - grow(g) );
		dc:=integer( colcen(g) - gcol(g) );
		dl:=integer( levcen(g) - glev(g) );
		d := d + abs(dr) + abs(dc) + abs(dl);
	end loop;
	return d;
end dist2sol;


procedure init( fname: string ) is
	fin : text_io.file_type;
	len : natural := 1;
	rcd : string(1..99);
	rr,cc,ll: float;
	g: integer := 0;
begin


	nrow:=3; ncol:=3; nlev:=3;
	dblk:=26; gblk:=26;

	nblk:=dblk+1; --27

	-- define 26 goal positions:
	for r in 1..3 loop
	for c in 1..3 loop
	for l in 1..3 loop
	if r=3 and c=3 and l=3 then --skip blank pos
		null;
	else
		g:=g+1;
		grow(g):=float(r)-0.5; 
		gcol(g):=float(c)-0.5; 
		glev(g):=float(l)-0.5;
	end if;
	end loop;
	end loop;
	end loop;



-- expect 27 lines with (row,col,layer) in each,
-- defining the current cubelet positions a..z,
-- the final line is the blank space position.
-- This is the format dumped from rcube.

	text_io.open(fin, in_file, fname);

-- read the file that contains the
-- (row,column,layer) positions of 
-- each cubelet written as integers:

	for i in 1..nblk loop
		myfloat_io.get(fin, rr);
		myfloat_io.get(fin, cc);
		myfloat_io.get(fin, ll);
		rowcen(i):=rr-0.5;
		colcen(i):=cc-0.5;
		levcen(i):=ll-0.5;
		text_io.get_line(fin, rcd, len); -- ignore any text
		idchar(i):=character'val(96+i); --a=97...z=122
		rowcen0(i):=rowcen(i);
		colcen0(i):=colcen(i);
		levcen0(i):=levcen(i); --last one (i=27) is blank pos
	end loop;

	text_io.close(fin);

	blank:=dblk+1; --27=index of blank space

	ntrail:=0;
	winner:=false;

end init;






--decrease column
function moveleft( selBlock: integer; track: boolean ) return integer is

	s: keytype := (others=>0);
	r,c,l : ushort;

	ret: integer := 0;

	obr,br: float := rowcen(blank);
	obc,bc: float := colcen(blank);
	obl,bl: float := levcen(blank);

	sr : float := rowcen(selBlock);
	sc : float := colcen(selBlock);
	sl : float := levcen(selBlock);

begin


	if
		abs(br-sr) < 0.1
			and
		abs(bc-sc+1.0)<0.1
			and
		abs(bl-sl) < 0.1
	then -- space allows moveleft
		colcen(selBlock) := bc;
		colcen(blank) := sc;
	end if;



	if
		abs(obr-rowcen(blank))<0.1
			and
		abs(obc-colcen(blank))<0.1
			and
		abs(obl-levcen(blank))<0.1
	then
		ret := 0; -- no change
	else
		ret := 1; -- change
	end if;


	if track and ret>0 then
		for j in 1..dblk loop
			r := ushort( float'rounding( rowcen(j)+0.5 ) );
			c := ushort( float'rounding( colcen(j)+0.5 ) );
			l := ushort( float'rounding( levcen(j)+0.5 ) );
			s(j) := ( endx(r,c,l) );
		end loop;

		ntrail:=ntrail+1;
		trailenc(ntrail):=s;
		trailsel(ntrail):=selblock;
		trailchr(ntrail):='l';

	end if;


	return ret;

end moveleft;










function moveright( selBlock: integer; track: boolean ) return integer is

	s: keytype := (others=>0);
	r,c,l : ushort;

	ret: integer := 0;

	obr,br: float := rowcen(blank);
	obc,bc: float := colcen(blank);
	obl,bl: float := levcen(blank);


	sr : float := rowcen(selBlock);
	sc : float := colcen(selBlock);
	sl : float := levcen(selBlock);

begin


	if
		abs(br-sr) < 0.1
			and
		abs(bc-sc-1.0)<0.1
			and
		abs(bl-sl) < 0.1
	then -- space allows moveright
		colcen(selBlock) := bc;
		colcen(blank) := sc;
	end if;



	if
		abs(obr-rowcen(blank))<0.1
			and
		abs(obc-colcen(blank))<0.1
			and
		abs(obl-levcen(blank))<0.1
	then
		ret := 0; -- no change
	else
		ret := 1; -- change
	end if;


	if track and ret>0 then
		for j in 1..dblk loop
			r := ushort( float'rounding( rowcen(j)+0.5 ) );
			c := ushort( float'rounding( colcen(j)+0.5 ) );
			l := ushort( float'rounding( levcen(j)+0.5 ) );
			s(j) := ( endx(r,c,l) );
		end loop;

		ntrail:=ntrail+1;
		trailenc(ntrail):=s;
		trailsel(ntrail):=selblock;
		trailchr(ntrail):='r';

	end if;




	return ret;


end moveright;








function moveup( selBlock: integer; track: boolean ) return integer is

	s: keytype := (others=>0);
	r,c,l : ushort;

	ret: integer := 0;

	obr,br: float := rowcen(blank);
	obc,bc: float := colcen(blank);
	obl,bl: float := levcen(blank);

	sr : float := rowcen(selBlock);
	sc : float := colcen(selBlock);
	sl : float := levcen(selBlock);

begin


		if
			abs(br-sr+1.0)<0.1
				and
			abs(bc-sc)<0.1
				and
			abs(bl-sl)<0.1
		then -- space allows moveup
			rowcen(selBlock) := br;
			rowcen(blank) := sr;
		end if;




	if
		abs(obr-rowcen(blank))<0.1
			and
		abs(obc-colcen(blank))<0.1
			and
		abs(obl-levcen(blank))<0.1
	then
		ret := 0; -- no change
	else
		ret := 1; -- change
	end if;


	if track and ret>0 then
		for j in 1..dblk loop
			r := ushort( float'rounding( rowcen(j)+0.5 ) );
			c := ushort( float'rounding( colcen(j)+0.5 ) );
			l := ushort( float'rounding( levcen(j)+0.5 ) );
			s(j) := ( endx(r,c,l) );
		end loop;

		ntrail:=ntrail+1;
		trailenc(ntrail):=s;
		trailsel(ntrail):=selblock;
		trailchr(ntrail):='u';

	end if;



	return ret;

end moveup;






function movedown( selBlock: integer; track: boolean ) return integer is

	s: keytype := (others=>0);
	r,c,l : ushort;

	ret: integer := 0;

	obr,br: float := rowcen(blank);
	obc,bc: float := colcen(blank);
	obl,bl: float := levcen(blank);

	sr : float := rowcen(selBlock);
	sc : float := colcen(selBlock);
	sl : float := levcen(selBlock);

begin


		if
			abs(br-sr-1.0)<0.1
				and
			abs(bc-sc)<0.1
				and
			abs(bl-sl)<0.1
		then -- space allows movedown
			rowcen(selBlock) := br;
			rowcen(blank) := sr;
		end if;




	if
		abs(obr-rowcen(blank))<0.1
			and
		abs(obc-colcen(blank))<0.1
			and
		abs(obl-levcen(blank))<0.1
	then
		ret := 0; -- no change
	else
		ret := 1; -- change
	end if;


	if track and ret>0 then
		for j in 1..dblk loop
			r := ushort( float'rounding( rowcen(j)+0.5 ) );
			c := ushort( float'rounding( colcen(j)+0.5 ) );
			l := ushort( float'rounding( levcen(j)+0.5 ) );
			s(j) := ( endx(r,c,l) );
		end loop;

		ntrail:=ntrail+1;
		trailenc(ntrail):=s;
		trailsel(ntrail):=selblock;
		trailchr(ntrail):='d';

	end if;


	return ret;


end movedown;









--level decreases
function movebackward( selBlock: integer; track: boolean ) return integer is

	s: keytype := (others=>0);
	r,c,l : ushort;

	ret: integer := 0;

	obr,br: float := rowcen(blank);
	obc,bc: float := colcen(blank);
	obl,bl: float := levcen(blank);

	sr : float := rowcen(selBlock);
	sc : float := colcen(selBlock);
	sl : float := levcen(selBlock);

begin


		if
			abs(br-sr)<0.1
				and
			abs(bc-sc)<0.1
				and
			abs(bl-sl+1.0)<0.1  -- space lower than level
		then -- space allows movebackward (level decreases)
			levcen(selBlock) := bl;
			levcen(blank) := sl;
		end if;




	if
		abs(obr-rowcen(blank))<0.1
			and
		abs(obc-colcen(blank))<0.1
			and
		abs(obl-levcen(blank))<0.1
	then
		ret := 0; -- no change
	else
		ret := 1; -- change
	end if;


	if track and ret>0 then
		for j in 1..dblk loop
			r := ushort( float'rounding( rowcen(j)+0.5 ) );
			c := ushort( float'rounding( colcen(j)+0.5 ) );
			l := ushort( float'rounding( levcen(j)+0.5 ) );
			s(j) := ( endx(r,c,l) );
		end loop;

		ntrail:=ntrail+1;
		trailenc(ntrail):=s;
		trailsel(ntrail):=selblock;
		trailchr(ntrail):='b';

	end if;


	return ret;


end movebackward;






--level increases
function moveforward( selBlock: integer; track: boolean ) return integer is

	s: keytype := (others=>0);
	r,c,l : ushort;

	ret: integer := 0;

	obr,br: float := rowcen(blank);
	obc,bc: float := colcen(blank);
	obl,bl: float := levcen(blank);

	sr : float := rowcen(selBlock);
	sc : float := colcen(selBlock);
	sl : float := levcen(selBlock);

begin


		if
			abs(br-sr)<0.1
				and
			abs(bc-sc)<0.1
				and
			abs(bl-sl-1.0)<0.1  -- space bigger than level
		then -- space allows moveforward (lev increases)
			levcen(selBlock) := bl;
			levcen(blank) := sl;
		end if;




	if
		abs(obr-rowcen(blank))<0.1
			and
		abs(obc-colcen(blank))<0.1
			and
		abs(obl-levcen(blank))<0.1
	then
		ret := 0; -- no change
	else
		ret := 1; -- change
	end if;


	if track and ret>0 then
		for j in 1..dblk loop
			r := ushort( float'rounding( rowcen(j)+0.5 ) );
			c := ushort( float'rounding( colcen(j)+0.5 ) );
			l := ushort( float'rounding( levcen(j)+0.5 ) );
			s(j) := ( endx(r,c,l) );
		end loop;

		ntrail:=ntrail+1;
		trailenc(ntrail):=s;
		trailsel(ntrail):=selblock;
		trailchr(ntrail):='f';

	end if;


	return ret;


end moveforward;
















procedure undo is
 res, selBlock: integer;
 chr: character;
begin

	if ntrail>0 then

 		chr := trailchr(ntrail);
		selBlock := trailsel(ntrail);
		ntrail := ntrail-1;

		case chr is

			when 'd' =>
				res := moveup(selBlock,false);
				myassert(res>0,11,"undo 1");

			when 'u' =>
				res := movedown(selBlock,false);
				myassert(res>0,12, "undo 2");

			when 'r' =>
				res := moveleft(selBlock,false);
				myassert(res>0,13, "undo 3");

			when 'l' =>
				res := moveright(selBlock,false);
				myassert(res>0,14, "undo 4");

			when 'b' => --level increase
				res := moveforward(selBlock,false);
				myassert(res>0,15, "undo 5");

			when 'f' => --level decrease
				res := movebackward(selBlock,false);
				myassert(res>0,16, "undo 6");


			when others => null;
		end case;


	end if;

end undo;













procedure addifnew( okey: keytype ) is
	rec : hashrectype;
	nt: constant integer := ntrail;
	nukey : keytype := trailenc(nt);
	pri: integer;
	use mypq;
	use mytree;
begin

	mytree.search(nukey,explored,rec,estatus);
	if estatus=notfound then -- not already in {explored}

		mypq.search( nukey, frontier, rec, pri, fstatus );

		-- if found, we have reached this config earlier, so ignore

		if fstatus=notfound then -- not already in {frontier}

			rec.prevkey := okey;
			rec.tsel := ubyte(trailsel(nt));
			rec.tchr := trailchr(nt);

			pri := dist2sol;

			mypq.addnode( nukey, rec, pri, frontier, fstatus );
			myassert( fstatus=ok, 15, "addnode error" );

		end if; -- not seen

	end if; --not in explored

end addifnew;






-- recursive ftn to load trail* from database
function getrail( pkey: keytype ) return integer is
	k: integer := 0;
	rec : hashrectype;
	use mytree;
begin

	mytree.search( pkey, explored, rec, estatus );

	if rec.tchr = 's' or rec.tsel=0 then
		return 0;

	elsif estatus=notfound then
		return 0;

	else

		k := getrail( rec.prevKey );
		myassert(k>=0,16, "getrail error");

		k := k+1;
		trailchr(k) := rec.tchr;
		trailsel(k) := integer(rec.tsel);

	end if;

	return k;

end getrail;




procedure restore( okey: keytype ) is
 res, selblock : integer;
 chr : character;
begin

	-- restore original block positions:
	for i in 1..nblk loop
		rowcen(i):=rowcen0(i);
		colcen(i):=colcen0(i);
		levcen(i):=levcen0(i);
	end loop;

-- now, restore block configuration

	ntrail:=getrail(okey);
	for i in 1..ntrail loop
		selblock := trailsel(i);
		chr := trailchr(i);
		case chr is
			when 'u' =>
				res := moveup(selblock,false);
				myassert(res>0,101,"restore 1");

			when 'd' =>
				res := movedown(selblock,false);
				myassert(res>0,102,"restore 2");

			when 'l' =>
				res := moveleft(selblock,false);
				myassert(res>0,103,"restore 3");

			when 'r' =>
				res := moveright(selblock,false);
				myassert(res>0,104,"restore 4");



			when 'f' => -- level increase
				res := moveforward(selblock,false);
				myassert(res>0,105,"restore 5");

			when 'b' => -- level decrease
				res := movebackward(selblock,false);
				myassert(res>0,106,"restore 6");




			when others => 
				null;
				put_line("ERROR in restore...bad trailchr");
				myassert(false);
		end case;
	end loop;
end restore;


bestSolnLen: integer := 9000;

procedure test4winner( key: keytype ) is
	canwin: boolean := true;
	len: integer;
begin

	if dist2sol<1 then
		winner:=true;

		len := getrail(key);
		if len < bestSolnLen then
			bestSolnLen := len;
			dump;
		end if;

	end if;

end test4winner;







procedure trymove is
	len: integer := 0;
	okey: keytype;
	orec: hashrectype;
	res,pri: integer;
begin --trymove


	--while not winner loop
	loop

		depth:=depth+1;

		if depth>80_000 then
			put("Sol-Len=");
			put(integer'image(bestSolnLen));
			put(", depth-exit"); new_line;
			exit;
		end if;

		len:=mypq.length(frontier);

		if len=0 then
			put("Sol-Len=");
			put(integer'image(bestSolnLen));
			put(", len-exit"); new_line;
			put(integer'image(depth));
			put(integer'image(len));
			new_line;
			exit;
		end if;


		mypq.popNode(frontier,okey,orec,pri,fstatus);

		mytree.addNode(okey,orec,explored,estatus);

		restore(okey);

		test4winner(okey);
		if winner then
			--put("Sol-Len=");
			--put(integer'image(bestSolnLen));
			--put(", winner-exit"); new_line;
			exit;
		end if;


		for ii in 1..dblk loop

			res := moveup(ii,true);
			if res>0 then
				addifnew(okey);
				undo;
			end if;

			res := movedown(ii,true);
			if res>0 then
				addifnew(okey);
				undo;
			end if;

			res := moveright(ii,true);
			if res>0 then
				addifnew(okey);
				undo;
			end if;

			res := moveleft(ii,true);
			if res>0 then
				addifnew(okey);
				undo;
			end if;


			res := moveforward(ii,true); --level increase
			if res>0 then
				addifnew(okey);
				undo;
			end if;

			res := movebackward(ii,true); --level decrease
			if res>0 then
				addifnew(okey);
				undo;
			end if;


		end loop;

	end loop; -- while::940



end trymove;


	key0 : keytype := (others=>0);
	rec0 : hashrectype;
	pri: integer;

	use mypq;

begin -- bfs


	init( to_string(infilname) ); -- read puzzle file

	rec0.prevKey := key0;
	rec0.tsel := 0;
	rec0.tchr := 's';

	pri := dist2sol;
	mypq.addnode( key0, rec0, pri, frontier, fstatus );
	myassert( fstatus=ok, 114, "bfs addnode error" );

	--put("initial dist2sol = "); put(integer'image(pri)); new_line;

	trymove;

end bfs; --proc

end bfs26; --package
