/*
 * FTGL - OpenGL font library
 *
 * Copyright (c) 2001-2004 Henry Maddocks <ftgl@opengl.geek.nz>
 * Copyright (c) 2008 Éric Beets <ericbeets@free.fr>
 * Copyright (c) 2008 Sam Hocevar <sam@zoy.org>
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include <cassert>
#include "mgGL/VBO.h"

//#include "config.h"

#include "FTGL/ftgl.h"

#include "FTInternals.h"
#include "FTPolygonGlyphImpl.h"
#include "FTVectoriser.h"
#include "mgGL/Appearance.h"


//
//  FTGLPolyGlyph
//


FTPolygonGlyph::FTPolygonGlyph(FT_GlyphSlot glyph, float outset) :
    FTGlyph(new FTPolygonGlyphImpl(glyph, outset))
{}

FTPolygonGlyph::~FTPolygonGlyph()
{}

const FTPoint& FTPolygonGlyph::Render(mgVBO& vbo, const FTPoint& pen, int renderMode)
{
    FTPolygonGlyphImpl *myimpl = dynamic_cast<FTPolygonGlyphImpl *>(impl);
    return myimpl->RenderImpl(vbo, pen, renderMode);
}


//
//  FTGLPolyGlyphImpl
//


FTPolygonGlyphImpl::FTPolygonGlyphImpl(FT_GlyphSlot glyph, float _outset)
:   FTGlyphImpl(glyph),vectoriser(NULL)
{
    if(ft_glyph_format_outline != glyph->format)
    {
        err = 0x14; // Invalid_Outline
        return;
    }

    vectoriser = new FTVectoriser(glyph);

    if((vectoriser->ContourCount() < 1) || (vectoriser->PointCount() < 3))
    {
        delete vectoriser;
        vectoriser = NULL;
        return;
    }


    hscale = glyph->face->size->metrics.x_ppem * 64;
    vscale = glyph->face->size->metrics.y_ppem * 64;
    outset = _outset;
}


FTPolygonGlyphImpl::~FTPolygonGlyphImpl()
{
  if(vectoriser!=NULL)
    {
        delete vectoriser;
    }
}

const FTPoint& FTPolygonGlyphImpl::RenderImpl(mgVBO& vbo, const FTPoint& pen,
                                              int renderMode)
{
   if(vectoriser)
    {
        DoRender(vbo, pen);
    }

    return advance;
}

void FTPolygonGlyphImpl::DoRender(mgVBO& vbo, const FTPoint& pen)
{
	vectoriser->MakeMesh(1.0, 1, outset);

	const FTMesh *mesh = vectoriser->GetMesh();

	for(unsigned int t = 0; t < mesh->TesselationCount(); ++t)
	{
		const FTTesselation* subMesh = mesh->Tesselation(t);
		unsigned int polygonType = subMesh->PolygonType();

		vbo.Begin(polygonType);
		for(unsigned int i = 0; i < subMesh->PointCount(); ++i)
		{
			FTPoint point = subMesh->Point(i);
			vbo.TexCoord2d(point.Xf() / hscale, point.Yf() / vscale);
			vbo.Vertex3d(point.Xf() / 64.0f + pen.Xf(), point.Yf() / 64.0f + pen.Yf(), 0.0f + pen.Zf());
		}
		vbo.End();
	}
}
