/********************************************************************/
/* Copyright (c) 2019 System fugen G.K. and Yuzi Mizuno          */
/* All rights reserved.                                             */
/********************************************************************/
#include "StdAfx.h"
#include "mg/Box.h"
#include "mg/KnotVector.h"
#include "mg/CCisects.h"
#include "mg/CSisects.h"
#include "mg/CParam_list.h"
#include "mg/Position_list.h"
#include "mg/Straight.h"
#include "mg/Ellipse.h"
#include "mg/SurfCurve.h"
#include "mg/TrimmedCurve.h"
#include "mg/BSumCurve.h"
#include "mg/CompositeCurve.h"
#include "mg/RLBRep.h"
#include "mg/Plane.h"
#include "mg/Sphere.h"
#include "mg/Cylinder.h"
#include "mg/SBRep.h"
#include "mg/RSBRep.h"
#include "mg/BSumSurf.h"
#include "mg/Tolerance.h"

#if defined(_DEBUG)
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

//
//Implements MGTrimmedCurve Class.
//MGTrimmedCurve is a part of original curve that has limitted parameter
//range.
//MGTrimmedCurve is a temporal curve, and does not have update functions.
//

///////Constructor///////

MGTrimmedCurve::MGTrimmedCurve()
:MGCurve(),m_sameRange(false),m_curve(nullptr){;}

//Copy constructor.
MGTrimmedCurve::MGTrimmedCurve(const MGTrimmedCurve& tc)
:MGCurve(tc), m_curve(tc.m_curve),
m_sameRange(tc.m_sameRange),m_range(tc.m_range){
	if(tc.m_knotV)
		m_knotV.reset(new MGKnotVector(*(tc.m_knotV)));
}

MGTrimmedCurve::MGTrimmedCurve(const MGCurve& crv, double t1, double t2)
:m_curve(&crv),m_sameRange(false),m_range(t1,t2){
	invalidateBox();
	if(t1>t2)
		m_range=MGInterval(t2,t1);
	const MGInterval& crange=crv.param_range();
	m_range&=crange;
	if(m_range==crange){
		m_range=crange;
		m_sameRange=true;
	}
	const MGTrimmedCurve* tc=dynamic_cast<const MGTrimmedCurve*>(&crv);
	if(tc) m_curve=tc->m_curve;
}

MGTrimmedCurve::MGTrimmedCurve(const MGCurve& crv, const MGInterval range)
:m_curve(&crv),m_sameRange(false),m_range(range){
	invalidateBox();
	const MGInterval& crange=crv.param_range();
	m_range&=crange;
	if(m_range==crange){
		m_range=crange;
		m_sameRange=true;
	}
	const MGTrimmedCurve* tc=dynamic_cast<const MGTrimmedCurve*>(&crv);
	if(tc) m_curve=tc->m_curve;
}

//Assignment.
//When the leaf object of this and obj2 are not equal, this assignment
//does nothing.
MGTrimmedCurve& MGTrimmedCurve::operator=(const MGTrimmedCurve& tc){
	if(this==&tc)
		return *this;

	MGCurve::operator=(tc);
	if(m_knotV.get()){
		m_knotV=nullptr;
	}
	if(tc.m_knotV.get())
		m_knotV.reset(new MGKnotVector(*(tc.m_knotV)));
	m_curve=tc.m_curve;
	m_sameRange=tc.m_sameRange;
	m_range=tc.m_range;
	return *this;
}
MGTrimmedCurve& MGTrimmedCurve::operator=(const MGGel& gel2){
	const MGTrimmedCurve* gel2_is_this=dynamic_cast<const MGTrimmedCurve*>(&gel2);
	if(gel2_is_this)
		operator=(*gel2_is_this);
	return *this;
}
MGTrimmedCurve& MGTrimmedCurve::operator=(MGGel&& gel2){
	MGTrimmedCurve* gel2_is_this=dynamic_cast<MGTrimmedCurve*>(&gel2);
	if(gel2_is_this)
		operator=(std::move(*gel2_is_this));
	return *this;
}

///Approximate this curve as a MGLBRep curve
///within the tolerance MGTolerance::line_zero().
///When parameter_normalization=0, reparameterization will not done, and
///the evaluation at the same parameter has the same values before and after
///of approximate_as_LBRep.
void MGTrimmedCurve::approximate_as_LBRep(
	MGLBRep& lb,		///<Approximated obrep will be set.
	int ordr,		///<new order
	int parameter_normalization,
		//Indicates how the parameter normalization be done:
		//=0: no parameter normalization.
		//=1: normalize to range=(0., 1.);
		//=2: normalize to make the average length of the 1st derivative 
		//    is as equal to 1. as possible.
	bool neglectMulti///<Indicates if multiple knots be kept.
		///< true: multiplicity is removed.
		///< false: multiplicity is kept.
)const{
	const MGCompositeCurve* compo=dynamic_cast<const MGCompositeCurve*>(base_curve());
	if(compo){
		compo->approximate_as_LBRep(lb,ordr,parameter_normalization,neglectMulti);
		lb.limit(m_range);
	}else{
		int odr2=ordr;
		const MGLBRep* bcrv = dynamic_cast<const MGLBRep*>(base_curve());
		if(bcrv){
			int k=bcrv->order();

			if(!ordr)
				ordr=k;
			if(m_sameRange){
				bcrv->approximate_as_LBRep(lb,ordr,parameter_normalization,neglectMulti);
			}else{
				MGLBRep lbtemp;
				bcrv->shrinkToParameters(m_range.low_point(), m_range.high_point(), lbtemp);
				lbtemp.approximate_as_LBRep(lb,ordr,parameter_normalization,neglectMulti);
			}
		}else
			MGCurve::approximate_as_LBRep(lb,ordr,parameter_normalization,neglectMulti);
	}
}

//Returns B-Rep Dimension.
int MGTrimmedCurve::bdim() const{
	const MGKnotVector& t=knot_vector();
	return t.bdim();
}

//Return minimum box that includes the curve of parameter interval.
// ͂̃p[^͈͂̋Ȑ͂ރ{bNXԂB
MGBox MGTrimmedCurve::box_limitted(
	const MGInterval& rng // Parameter Range of the curve.
)const{
	MGInterval prng(param_range()&rng);
	return m_curve->box_limitted(prng);
}

//Compute the box of the whole of the curve.
//Returned is a newed object pointer.
void MGTrimmedCurve::compute_box(MGBox& bx)const{
	bx=m_curve->box_limitted(m_range);
}

//Construct new curve object by copying to newed area.
//User must delete this copied object by "delete"
MGCurve* MGTrimmedCurve::clone() const{
	if(m_sameRange)
		return m_curve->clone();
	return m_curve->part(m_range.low_point(),m_range.high_point());
}

///Divide this curve at the designated knot multiplicity point.
///Function's return value is the number of the curves after divided.
int MGTrimmedCurve::divide_multi(
	std::vector<UniqueCurve>& crv_list,	//divided curves are appended.
	int multiplicity	///<designates the multiplicity of the knot to divide at,
						///<When multiplicity<=0, order()-1 is assumed,
						///<When multiplicity>=order(), order() is assumed.
)const{
	if(m_sameRange)
		return m_curve->divide_multi(crv_list,multiplicity);

	std::vector<UniqueCurve> crv_list2;
	int ncurve=m_curve->divide_multi(crv_list2,multiplicity);
	double t0=param_s(), t1=param_e();
	int ncurveNew=0;
	for(int i=0; i<ncurve; i++){
		MGCurve* ci=crv_list2[i].release();
		double ti0=ci->param_s(), ti1=ci->param_e();
		if(ti1<=t0){
			delete ci;
			continue;
		}
		if(t1<=ti0)
			break;
		if(ti0<t0 || t1<ti1)
			ci->limit(t0,t1);
		crv_list.emplace_back(ci);
		ncurveNew++;
	}
	return ncurveNew;
}

//Exclusive function for common.
//When this curve is a TrimmedCurve of MGComposite, narrow the parameter range
//by this m_range.
void MGTrimmedCurve::narrow_into_range(
	const MGCurve& curve2,
	std::vector<double>& vecComSpan
)const{
	std::vector<double> paramVec;
	int n=(int)vecComSpan.size();
	for(int i=0; i<n;){
		double& s0=vecComSpan[i++];
		double& s1=vecComSpan[i++];
		double& t0=vecComSpan[i++];
		double& t1=vecComSpan[i++];
		if(s1<=m_range)
			continue;
		if(m_range<=s0)
			continue;
		if(m_range.includes(s0) && m_range.includes(s1)){//when span [s0, s1] is included in m_range
			paramVec.push_back(s0);
			paramVec.push_back(s1);
			paramVec.push_back(t0);
			paramVec.push_back(t1);
			continue;
		}

		double s0d=m_range.low_point();
		double s1d=m_range.high_point();
		double t0d=t0, t1d=t1;
		if(s0<=s0d && s0d<=s1){
			double tg=t0+(t1-t0)*((s0d-s0)/(s1-s0));
			curve2.perp_guess(t0,t1,start_point(),tg,t0d);
		}
		if(s0<=s1d && s1d<=s1){
			double tg=t0+(t1-t0)*((s1d-s0)/(s1-s0));
			curve2.perp_guess(t0,t1,end_point(),tg,t1d);
		}
		paramVec.push_back(s0d);
		paramVec.push_back(s1d);
		paramVec.push_back(t0d);
		paramVec.push_back(t1d);
	}
	vecComSpan=paramVec;
}

//֐Fcommon
//ړIF^ꂽȐƎg̋ʕ邩ǂׂB
//F
//		const MGCurve&			curve2,		(I/ )	^Ȑ
//		std::vector<double>&	vecComSpan	( /O)	ʕ̃p[^͈
//		 4n̔zŁAvecComSpan(4*i+0),vecComSpan(4*i+1)g̃p[^͈
//					(vecComSpan(4*i+0) < vecComSpan(4*i+1))A
//				 vecComSpan(4*i+2),vecComSpan(4*i+3)curve2̃p[^͈
//		MGCCisects&			isect		( /O)	_
//߂lF
//		3:_ʕ܂
//		2:_݂̂܂
//		1:ʕ݂̂܂
//		0:_ʕȂ
//		-1:ʃGbW̎vZG[
//		-2:ʃGbWSȏ㋁܂(̂ĂȂƌȂ)
//ǋLF
//	Ȑʂǂ̌덷ɂline_zero()Ap[^͈͂̎vZ
//	덷ɂ́Ap[^͈*rc_zero()gp
int MGTrimmedCurve::common(
	const MGCurve& curve2,
	std::vector<double>& vecComSpan,
	MGCCisects& isect
)const{
	int ret;
	const MGCompositeCurve* cmpo=dynamic_cast<const MGCompositeCurve*>(m_curve);
	if(cmpo){
		ret=cmpo->common(curve2,vecComSpan,isect);
	}else{
		ret=m_curve->common(curve2,vecComSpan,isect);
	}
	if(!m_sameRange)
		narrow_into_range(curve2,vecComSpan);

	MGCCisects::iterator j=isect.begin(), jend=isect.end(),j2;
	while(j!=jend){//Remove parameter values that are outside the range.
		j2=j; j2++;
		auto& cci=isectCast<MGCCisect>(j);
		if(!m_range.includes(cci.param1()))
			isect.erase(j);
		j=j2;
	}
	return ret;
}

//֐Fcommon
//ړIF^ꂽȐƎg̋ʕ邩ǂׂB
//F
//		const MGCurve&			curve2,		(I/ )	^Ȑ
//		std::vector<double>&	vecComSpan	( /O)	ʕ̃p[^͈
//		 4n̔zŁAvecComSpan(4*i+0),vecComSpan(4*i+1)g̃p[^͈
//					(vecComSpan(4*i+0) < vecComSpan(4*i+1))A
//				 vecComSpan(4*i+2),vecComSpan(4*i+3)curve2̃p[^͈
//߂lF
//		ʕ̐:	ʕ܂
//		0:				ʕȂ
//		-1:				ʃGbW̎vZG[
//		-2:				ʃGbWSȏ㋁܂(̂ĂȂƌȂ)
//ǋLF
//	Ȑʂǂ̌덷ɂline_zero()Ap[^͈͂̎vŽ덷ɂ́A
//  p[^͈*rc_zero()gp
int MGTrimmedCurve::common(
	const MGCurve& curve2,
	std::vector<double>& vecComSpan
)const{
	int ret;
	const MGCompositeCurve* cmpo=dynamic_cast<const MGCompositeCurve*>(m_curve);
	if(cmpo){
		ret=cmpo->common(curve2,vecComSpan);
	}else{
		ret=m_curve->common(curve2,vecComSpan);
	}
	if(!m_sameRange)
		narrow_into_range(curve2,vecComSpan);
	return ret;
}

//copy as a newed curve. The new curve will be MGLBRep or MGRLBRep.
//When original curve was a MGRLBRep, the new curve will be a MGRLBRep.
//Otherwise,  the new curve will be a MGLBRep.
//Returned object must be deleted.
MGCurve* MGTrimmedCurve::copy_as_nurbs() const{
	MGCurve* tcrv=m_curve->part(m_range.low_point(), m_range.high_point());
	MGCurve* nurbs=tcrv->copy_as_nurbs();
	delete tcrv;
	return nurbs;
}

//Construct new curve object by changing
//the original object's space dimension.
//User must delete this copied object by "delete".
MGCurve* MGTrimmedCurve::copy_change_dimension(
	int sdim,		// new space dimension
	int start1, 		// Destination order of new line.
	int start2 		// Source order of this line.
)const{
	MGCurve* crv=(MGCurve*)
		(m_curve->copy_change_dimension(sdim,start1,start2));
	MGCurve* trimmed=crv->part(m_range.low_point(), m_range.high_point());
	delete crv;
	return trimmed;
}

//Construct new curve object by copying to newed area,
//and limitting the parameter range to prange.
//Returned is newed object and must be deleted.
MGCurve* MGTrimmedCurve::copy_limitted(const MGInterval& prange) const{
	MGInterval prng2(prange&m_range);
	return m_curve->part(prng2.low_point(), prng2.high_point());
}
MGCurve* MGTrimmedCurve::copy_limitted() const{
	return m_curve->part(m_range.low_point(), m_range.high_point());
}

//Compute curvilinear integral of the 1st two coordinates.
//This integral can be used to compute area sorounded by the curve.
//(ϕj߂B
//curvilinear_integral from t1 to t2 can be obtained by
//Integral of (x*dy-y*dx) about t, where curve is expressed by
//f(t)=(x(t),y(t)), dx=dx/dt, and dy=dy/dt.
double MGTrimmedCurve::curvilinear_integral(double t1, double t2) const{
	MGInterval prng2(m_range&MGInterval(t1,t2));
	return m_curve->curvilinear_integral
					(prng2.low_point(), prng2.high_point());
}

//Extrapolate this curve by an (approximate) chord length.
//The extrapolation is C2 continuous.
void MGTrimmedCurve::extend(
	double length,	//approximate chord length to extend. 
	bool start		//Flag of which point to extend, start or end point of the line.
					//If start is true extend on the start point.
){
	double lenPara;
	double t, ts=m_curve->param_s(), te=m_curve->param_e();
	if(start){
		t=param_s();
		lenPara=m_curve->length(ts,t);
	}else{
		t=param_e();
		lenPara=m_curve->length(t,te);
	}
	if(length>=lenPara)
		return;

	if(start)
		length*=-1.;
	double t2=m_curve->length_param(t,length);
	if(start)
		change_range(t2,param_e());
	else
		change_range(param_s(),t2);
}

//Provide divide number of curve span for function intersect.
int MGTrimmedCurve::intersect_dnum() const{
	int n;
	const MGKnotVector& t=knot_vector();
	if(t==mgNULL_KNOT_VECTOR) n=m_curve->intersect_dnum();
	else{
		int k=order();
		int nspan=bdim()+2-k;
		int km2=k-2; if(km2<=0) km2=1;
		double t1=m_range.low_point(), t2=m_range.high_point();
		int i1=t.locate(t1), i2=t.locate(t2,1);
		if(i2<=i1)
			i2=i1+1;
		if(dynamic_cast<const MGRLBRep*>(m_curve)) n=(i2-i1+1)*(km2+1);
		else n=(i2-i1+1)*km2;
	}
	return n;
}

//Intersection of Curve.
MGCCisects MGTrimmedCurve::isect_in_range(const MGCurve& crv2) const{
	MGCCisects list=m_curve->isect(crv2);
	if(m_sameRange)
		return list;
	MGCCisects::iterator i=list.begin(), iend=list.end(),i2;
	while(i!=iend){//Remove parameter values that are outside the range.
		i2=i; i2++;
		auto& cci=isectCast<MGCCisect>(i);
		if(!m_range.includes(cci.param1()))
			list.erase(i);
		i=i2;
	}
	return list;
}

//Compute intersections with MGLBRep curve2 that does not have C0 continuity in it.
MGCCisects MGTrimmedCurve::isect_withC1LB(const MGLBRep& curve2)const{
	return isect_in_range(curve2);
}

///Intersection of Curve and Surface
MGCCisects MGTrimmedCurve::isect(const MGCurve& curve2) const{
	return isect_in_range(curve2); 
}
MGCCisects MGTrimmedCurve::isect(const MGStraight& curve2) const{
	return isect_in_range(curve2); 
}
MGCCisects MGTrimmedCurve::isect(const MGLBRep& curve2) const{
	return isect_in_range(curve2); 
}
MGCCisects MGTrimmedCurve::isect(const MGSurfCurve& curve2) const{
	return isect_in_range(curve2); 
}

//Intersection with a surface.
MGCSisects MGTrimmedCurve::isect(const MGSurface& surf) const{
	return isect_in_range(surf);
}
MGCSisects MGTrimmedCurve::isect(const MGPlane& surf) const{
	return isect_in_range(surf);
}

//Intersection with a surface.
MGCSisects MGTrimmedCurve::isect_in_range(const MGSurface& surf)const{
	MGCSisects list=m_curve->isect(surf);
	if(m_sameRange)
		return list;
	MGCSisects::iterator i=list.begin(), iend=list.end(),i2;
	while(i!=iend){//Remove parameter values that are outside the range.
		i2=i; i2++;
		auto& cci=isectCast<MGCSisect>(i);
		if(!m_range.includes(cci.param_curve()))
			list.removeAt(i);
		i=i2;
	}
	return list;
}

//Compute intersection point of 1D sub curve of original curve.
//Parameter values of intersection point will be returned.
MGCParam_list MGTrimmedCurve::intersect_1D(						
	double f,			// Coordinate value
	int coordinate	// Coordinate kind of the data f(from 0).
)const{
	MGCParam_list list=m_curve->isect_1D(f,coordinate);
	if(m_sameRange) return list;
	MGCParam_list::iterator i=list.begin(), iend=list.end(),i2;
	while(i!=iend){//Remove parameter values that are outside the range.
		i2=i; i2++;
		if(!m_range.includes((*i)))
			list.removeAt(i);
		i=i2;
	}
	return list;
}

//Cmpute curve length of the interval.
//If t1 is greater than t2, return negative value.
// ^ꂽp[^lԂ̋Ȑ̒ԂB
// p[^ŗ^ꂽƂ͐lA~̂Ƃ͕lԂB
double MGTrimmedCurve::length(double t1, double t2) const{
	t1=range(t1); t2=range(t2);
	return m_curve->length(t1,t2);
}

//Inverse function of length. Compute the point that is away from
//the point t by length len.
// length̋t֐Bwp[^tŎ_w苗len
// Ȑɉėꂽ_p[^lԂB
double MGTrimmedCurve::length_param( double t, double len) const{
	t=range(t);
	return m_curve->length_param(t,len);
}

//Update this by limiting the parameter range of the curve.
// gɎw肵p[^͈͂̂B
void MGTrimmedCurve::limit(const MGInterval& nrng){
	if(m_range!=m_curve->param_range()){
		m_knotV=nullptr;
		m_range&=nrng;
		m_sameRange=false;
		invalidateBox();
	}
}

// Return ending parameter value.
double MGTrimmedCurve::param_e() const{
	return m_range.high_point();
}

//Normalize parameter value t to the nearest knot if their distance is
//within tolerance.
double MGTrimmedCurve::param_normalize(double t) const{
	double s1=m_range.low_point(); if(t<=s1) return s1;
	double s2=m_range.high_point(); if(t>=s2) return s2;
	double len=s2-s1;
	if((t-s1)<=(s2-t)){if(MGREqual_base(t,s1,len)) return s1;}
	else {if(MGREqual_base(t,s2,len)) return s2;}
	return m_curve->param_normalize(t);
}

// Return starting parameter value.
double MGTrimmedCurve::param_s() const{
	return m_range.low_point();
}
	
//Compute part of this curve from parameter t1 to t2.
//Returned is the pointer to newed object, and so should be deleted
//by calling program, or memory leaked.
MGCurve* MGTrimmedCurve::part(double t1, double t2, int multiple) const{
	assert(t1<t2);
	MGInterval rng(param_range()&MGInterval(t1,t2));
	return m_curve->part(rng.low_point(),rng.high_point(),multiple);
}

//Compute all foot points of the perpendicular line from point to
//the curve.
// ^|CgȐ։낵̑́CȐ̃p[^l
// ׂċ߂B
MGCParam_list MGTrimmedCurve::perps(
	const MGPosition& P		//Point(w_)
)const{
	if(m_sameRange) return m_curve->perps(P);
	MGCParam_list list=m_curve->perps(P);
	MGCParam_list::iterator i=list.begin(), iend=list.end(),i2;
	while(i!=iend){//Remove parameter values that are outside the range.
		i2=i; i2++;
		if(!m_range.includes((*i)))
			list.removeAt(i);
		i=i2;
	}
	return list;
}

//Compute all the perpendicular points of this curve and the second one.
//That is, if f(s) and g(t) are the points of the two curves f and g,
//then obtains points where the following conditions are satisfied:
//  fs*(f-g)=0.    gt*(g-f)=0.
//Here fs and gt are 1st derivatives at s and t of f and g.
//MGPosition P in the MGPosition_list contains this and crv's parameter
//as:     P(0)=this curve's parameter, P(1)=crv's parameter value.
MGPosition_list MGTrimmedCurve::perps(const MGSurfCurve& crv2)const{
	return perps_in_range(crv2);
}

//Compute all the perpendicular points of this curve and the second one.
//That is, if f(s) and g(t) are the points of the two curves f and g,
//then obtains points where the following conditions are satisfied:
//  fs*(f-g)=0.    gt*(g-f)=0.
//Here fs and gt are 1st derivatives at s and t of f and g.
//MGPosition P in the MGPosition_list contains this and crv's parameter
//as:     P(0)=this curve's parameter, P(1)=crv's parameter value.
MGPosition_list MGTrimmedCurve::perps_in_range(
	const MGCurve& crv2		//The second curve
)const{
	MGPosition_list list=m_curve->perps(crv2);
	if(m_sameRange)
		return list;

	MGPosition_list::iterator i=list.begin(), iend=list.end(),i2;
	while(i!=iend){//Remove parameter values that are outside the range.
		i2=i; i2++;
		if(!m_range.includes(((*i).ref(0))))
			list.removeAt(i);
		i=i2;
	}
	return list;
}

//Round t into curve's parameter range.
// ̓p[^p[^͈͂ł܂߂ĕԋpB
double MGTrimmedCurve::range(double t) const{
	double s=m_range.low_point();
	if(t<=s) return s;
	s=m_range.high_point();
	if(t>=s) return s;
	return t;
}

//Unlimit parameter range of the curve(limit͂)
MGCurve& MGTrimmedCurve::unlimit(){
	m_knotV=nullptr;
	m_range=m_curve->param_range();
	m_sameRange=true;
	invalidateBox();
	return *this;
}

//Unlimit parameter range of the curve to the end point direction
//(I_limit͂)
MGCurve& MGTrimmedCurve::unlimit_end(){
	m_knotV=nullptr;
	m_range(1)=m_curve->param_range()[1];
	if(m_range==m_curve->param_range()){
		m_range=m_curve->param_range();
		m_sameRange=true;
	}else
		m_sameRange=false;
	invalidateBox();
	return *this;
}

//Unlimit parameter range of the curve to the start point direction
//(n_limit͂)
MGCurve& MGTrimmedCurve::unlimit_start(){
	m_knotV=nullptr;

	m_range(0)=m_curve->param_range()[0];
	if(m_range==m_curve->param_range()){
		m_range=m_curve->param_range();
		m_sameRange=true;
	}else
		m_sameRange=false;
	invalidateBox();
	return *this;
}

//Operator overload(Zqd`)

//Logical operator overload(_Zqd`)

// ^ȐƎg̔rsB
bool MGTrimmedCurve::operator==(const MGTrimmedCurve& crv2) const{
	if((crv2.m_range)!=m_range)
		return false;
	if(m_curve==0){
		if(crv2.m_curve==0)
			return true;
		return false;
	}
	if(crv2.m_curve==0){
		if(m_curve==0)
			return true;
		return false;
	}

	return *(crv2.m_curve)==(*m_curve);
}
bool MGTrimmedCurve::operator==(const MGCompositeCurve& crv2) const{
	return is_same_curve(crv2);
}
bool MGTrimmedCurve::operator<(const MGTrimmedCurve& gel2)const{
	if(!m_curve)
		return true;
	if(!(gel2.m_curve))
		return false;
	if(*(gel2.m_curve)==*m_curve)
		return m_range.length()<gel2.m_range.length();
	return *m_curve<*(gel2.m_curve);
}
bool MGTrimmedCurve::operator==(const MGGel& gel2)const{
	const MGTrimmedCurve* gel2_is_this=dynamic_cast<const MGTrimmedCurve*>(&gel2);
	if(gel2_is_this)
		return operator==(*gel2_is_this);
	else{
		const MGCompositeCurve* gel2_is_compo=dynamic_cast<const MGCompositeCurve*>(&gel2);
		if(gel2_is_compo)
			return operator==(*gel2_is_compo);

	}
	return false;
}
bool MGTrimmedCurve::operator<(const MGGel& gel2)const{
	const MGTrimmedCurve* gel2_is_this=dynamic_cast<const MGTrimmedCurve*>(&gel2);
	if(gel2_is_this)
		return operator<(*gel2_is_this);
	return false;
}

bool MGTrimmedCurve::is_same_curve(const MGCurve& crv2)const{
	if(!m_sameRange)
		return false;
	if(m_curve==0)
		return false;
	return (*m_curve)==(crv2);
}

//Test if this cure is planar or not.
//MGPlane expression will be out to plane if this is planar.
//Function's return value is true if planar.
bool MGTrimmedCurve::is_planar(MGPlane& plane)const{
	if(m_sameRange)
		return m_curve->is_planar(plane);
	MGCurve* tcrv=this->part(m_range.low_point(), m_range.high_point());
	bool planar=tcrv->is_planar(plane);
	delete tcrv;
	return planar;
}

//Access to i-th element of knot
double MGTrimmedCurve::knot(int i) const{
	const MGKnotVector& t=knot_vector();
	return t(i);
}

//Returns the knot vector.
const MGKnotVector& MGTrimmedCurve::knot_vector() const{
	if(m_sameRange)
		return m_curve->knot_vector();
	if(!m_knotV.get()){
		m_knotV.reset(new MGKnotVector(m_curve->knot_vector(),
					m_range.low_point(), m_range.high_point()));
	}
	return *m_knotV;
}

//Exchange ordering of the coordinates.
//Exchange coordinates (i) and (j).
void MGTrimmedCurve::coordinate_exchange(int i, int j){
	assert(false);
}

//Negate the curve direction(Ȑ̕𔽓])
void MGTrimmedCurve::negate(){
	assert(false);
}

//Obtain so transformed 1D curve expression of this curve that
//f(t)={sum(xi(t)*g[i]) for i=0(x), 1(y), 2(z)}-g[3], where f(t) is the output
//of oneD and xi(t) is i-th coordinate expression of this curve.
//This is used to compute intersections with a plane g[4].
std::unique_ptr<MGCurve> MGTrimmedCurve::oneD(
	const double g[4]			//Plane expression(a,b,c,d) where ax+by+cz=d.
) const{
	std::unique_ptr<MGCurve> crv=m_curve->oneD(g);
	crv->limit(m_range);
	return crv;
}

//mbg폜֐(B\Ȑ̂)
//gXline_zerogpB̃mbĝׂقǍ폜₷
//߂ĺA폜mbg̐
//removal knot. line_zero tolerance is used.
void MGTrimmedCurve::remove_knot(){assert(false);}

//Update the curve by translation.
// ^xNgȐ𕽍sړĎgƂB
MGTrimmedCurve& MGTrimmedCurve::operator+= (const MGVector& v)
{assert(false); return *this;}

//Update the curve by translation.
// ^xNgȐ}CiXɕsړĎgƂB
MGTrimmedCurve& MGTrimmedCurve::operator-= (const MGVector& v)
{assert(false); return *this;}

//Update the curve by multiplying scale.
// ^ꂽXP[ȐɂB
MGTrimmedCurve& MGTrimmedCurve::operator*= (double scale)
{assert(false); return *this;}

//Update the curve by transformation of matrix.
// ^ꂽϊŒ̕ϊsg̒ƂB
MGTrimmedCurve& MGTrimmedCurve::operator*= (const MGMatrix& mat)
{assert(false); return *this;}

//Update the curve by transformation of transf.
// ^ꂽϊŋȐ̃gXtH[sgƂB
MGTrimmedCurve& MGTrimmedCurve::operator*= (const MGTransf& tr)
{assert(false); return *this;}

MGTrimmedCurve MGTrimmedCurve::operator+ (const MGVector& v) const{
	assert(false); return *this;
}
MGTrimmedCurve operator+ (const MGVector& v, const MGTrimmedCurve& cv2){
	assert(false); return cv2;
}
MGTrimmedCurve MGTrimmedCurve::operator- (const MGVector& v) const{
	assert(false); return *this;
}
MGTrimmedCurve MGTrimmedCurve::operator* (double scale) const{
	assert(false); return *this;
}
MGTrimmedCurve operator* (double scale, const MGTrimmedCurve& cv2){
	assert(false); return cv2;
}
MGTrimmedCurve MGTrimmedCurve::operator* (const MGMatrix& mat) const{
	assert(false); return *this;
}
MGTrimmedCurve MGTrimmedCurve::operator* (const MGTransf& tr) const{
	assert(false); return *this;
}
