module TapKit #:nodoc:

	class GeneralAdapterError < RuntimeError; end #:nodoc:

	# The subclasses must be implemented run() method.
	class LoginPrompt
		def run; end
	end

	class Adapter
		attr_reader :name, :application, :model, :contexts
		attr_accessor :connection

		class << self
			def new_with_name( name, model = nil, application = nil )
				require "tapkit/access/adapters/#{name.downcase.untaint}"
				adapter_class(name).new(model, application)
			end

			def adapter_class( name )
				TapKit.const_get("#{name}Adapter")
			end

			# abstract
			def expression_class; end

			# Implemented by subclasses to return object of the classes used to 
			# save values to database as external type,
			# and to create model files by modeler.
			# Returns nil if mapping for external type can't be found.
			#
			# The default implementation is using internal_types() of class method.
			def internal_type( external_type )
				internal_types[external_type.downcase]
			end

			# Hash of internal types.
			# If you create subclasses of DBIAdapter, use this with adding
			# own internal types for database.
			#
			# The default implementation is returning hash of data types for SQL-92.
			def internal_types
				{
					'char'                       => String,
					'character'                  => String,
					'varchar'                    => String,
					'char varying'               => String,
					'character varying'          => String,
					'nchar'                      => String,
					'national char'              => String,
					'national character'         => String,
					'national character varying' => String,
					'bit'                        => String,
					'bit varying'                => String,
					'int'                        => Integer,
					'integer'                    => Integer,
					'smallint'                   => Integer,
					'numeric'                    => Float,
					'decimal'                    => Float,
					'dec'                        => Float,
					'float'                      => Float,
					'real'                       => Float,
					'double precision'           => Float,
					'date'                       => Date,
					'time'                       => Time,
					'timestamp'                  => Timestamp,
					'interval'                   => Integer
				}
			end

			def internal_type_name( external_type )
				name = internal_type(external_type).to_s
				name.gsub!(/TapKit\:\:/, '')
				name
			end

			# Abstract method. Returns an instance of LoginPrompt for the adapter.
			# A subclass must override this method.
			def login_prompt; end
		end

		def initialize( model, application = nil )
			if model then
				@name       = model.adapter_name
				@connection = model.connection
				@model      = model
			else
				@connection = {}
				name = self.class.to_s
				name.gsub!(/TapKit\:\:/, '')
				name.gsub!(/adapter/i, '')
				@name = name
			end

			@application = application
			@contexts = []
		end

		# abstract
		def expression_factory; end

		# Abstract method. Returns a new context object and
		# adds it into contexts array. A subclass should override.
		def create_context
			context = TapKit.const_get("#{@name}Context").new self
			self.contexts << context
			context
		end

		def run_login_prompt
			self.class.login_prompt.run
		end
	end


	class AdapterChannel
		attr_reader :adapter_context, :attributes_to_fetch, :application

		class << self
			def new_with_name( name, adapter_context )
				TapKit.const_get("#{name}Channel").new adapter_context
			end
		end

		def initialize( adapter_context )
			@adapter_context = adapter_context
			@application     = @adapter_context.adapter.application
			@attributes_to_fetch = []
			@adapter_context.channels << self
		end

		# abstract
		def select_attributes( attributes, lock, fetch_spec, entity ); end

		# abstract
		def evaluate( expression ); end

		# abstract
		def attributes_to_fetch=( attributes ); end

		def perform_adapter_operations( adapter_operations )
			adapter_operations.each do |op|
				perform_adapter_operation op
			end
		end

		def perform_adapter_operation( adapter_operation )
			open

			case adapter_operation.adapter_operator
			when DatabaseOperation::ADAPTER_INSERT_OPERATOR
				insert_row(adapter_operation.changed_values, adapter_operation.entity)
			when DatabaseOperation::ADAPTER_DELETE_OPERATOR
				delete_row(adapter_operation.qualifier, adapter_operation.entity)
			when DatabaseOperation::ADAPTER_UPDATE_OPERATOR
				update_row(adapter_operation.changed_values, \
					adapter_operation.qualifier, adapter_operation.entity)
			else
				raise "Unsupported operation"
			end
		end

		def delete_row( qualifier, entity )
			if delete_rows(qualifier, entity) > 1 then
				raise GeneralAdapterError, "deleted rows more than one"
			end
		end

		def update_row( row, qualifier, entity )
			if update_rows(row, qualifier, entity) > 1 then
				raise GeneralAdapterError, "updated rows more than one"
			end
		end

		# abstract
		def insert_row( row, entity ); end

		# abstract
		def delete_rows( qualifier, entity ); end

		# abstract
		def update_rows( row, qualifier, entity ); end

		def check_transaction
			unless @adapter_context.open_transaction? then
				raise "channel's context has no transaction"
			end
		end
	end


	class AdapterContext
		attr_reader :adapter, :channels

		class << self
			def new_with_name( name, adapter )
				TapKit.const_get("#{name}Context").new adapter
			end
		end

		def initialize( adapter )
			@adapter = adapter
			@channels = []
			@open_transaction = false
		end

		def application
			@adapter.application
		end

		def transaction_did_begin; end

		def transaction_did_commit; end

		def transaction_did_rollback; end

		def open_transaction?
			@open_transaction
		end

		def commit_transaction?
			@commit_transaction
		end

		def rollback_transaction?
			@rollback_transaction
		end

		# abstract
		def create_channel; end

		# abstract
		def begin_transaction; end

		# abstract
		def commit_transaction; end

		# abstract
		def rollback_transaction; end
	end


	class AdapterOperation
		attr_accessor :adapter_operator, :attributes, :changed_values, :exception, :qualifier, :stored_procedure
		attr_reader :entity

		def initialize( entity, adapter_operator = nil )
			@entity = entity
			@adapter_operator = adapter_operator
		end

		def compare( other ); end
	end

end

