require 'openbase'

module TapKit

	class OpenBaseLoginPrompt < LoginPrompt
		def run
			conn = {}
			puts "Login OpenBase database"

			print "Database: "
			conn['database'] = gets.chomp

			print "Host: "
			conn['host'] = gets.chomp

			print "Username: "
			conn['user'] = gets.chomp

			print "Password: "
			conn['password'] = gets.chomp

			conn
		end
	end

	class OpenBaseAdapter < Adapter
		class << self
			def expression_class
				OpenBaseExpression
			end

			def internal_types
				types = super
				types['char']     = String
				types['int']      = Integer
				types['float']    = Float
				types['long']     = Integer
				types['longlong'] = Integer
				types['money']    = Integer
				types['date']     = Date
				types['time']     = Time
				types['datetime'] = Timestamp
				types['object']   = String
				types
			end

			def login_prompt
				OpenBaseLoginPrompt.new
			end
		end

		def expression_factory
			OpenBaseExpressionFactory.new self
		end

		def create_context
			context = OpenBaseContext.new self
			self.contexts << context
			context
		end
	end

	class OpenBaseChannel < AdapterChannel
		attr_reader :openbase

		def evaluate_for_aggregate_spec( agg_spec, expression )
			result = self.evaluate expression
			rows = []
			result.each do |row|
				new_row = {}
				row.each_with_index do |value, index|
					key = result.column_infos[index].name
					new_row[key] = value
				end
				rows << new_row
			end
			rows
		end

		def initialize( adapter_context )
			super
			@pk_cache = Hash.new(0)
			@openbase = nil
		end

		def fetch_progress?
			false
		end

		def select_attributes( attributes, lock, fetch_spec, entity )
			@attributes_to_fetch = attributes
		end

		def open?
			if @openbase then
				true
			else
				false
			end
		end

		def open
			unless open? then
				connection = @adapter_context.adapter.connection
				db   = connection['database']
				host = connection['host']
				user = connection['user']
				pass = connection['password']
				@openbase = OpenBase.new(db, host, user, pass)
			end
		end

		def close
			if @openbase then
				@openbase.invalidate
			end
		end

		def evaluate( expression )
			@entity = expression.entity
			open

			bindings = []
			expression.bind_variables.each do |binding|
				value = binding[expression.class::VALUE_KEY]
				attr = binding[expression.class::ATTRIBUTE_KEY]
				value = attr.adapter_value(value)
				binding[expression.class::VALUE_KEY] = value
				bindings << expression.format_value(value, attr)
			end

			statement = expression.statement.dup
			index = 0
			statement.gsub!('?') do |char|
				index += 1
				bindings[index-1]
			end

			if application then
				if application.log_options[:sql] then
					application.log_options[:out].puts statement
				end
			end

			@pk_cache.clear
			@result = @openbase.execute(statement)
		end

		def fetch_all
			rows = []
			@result.each do |raw_row|
				row = {}

				raw_row.each_with_index do |value, index|
					column = @result.column_infos[index].name

					attr = nil
					@attributes_to_fetch.each do |_attr|
						if _attr.column_name == column then
							attr = _attr
						end
					end

					if attr then
						encoding = @adapter_context.adapter.connection['encoding']
						value = _convert_datetime(attr, value)
						row[attr.name] = attr.new_value(value, encoding)
					else
						row[column] = value
					end
				end
				rows << row
			end
			rows
		end

		private

		def _convert_datetime( attr, value )
			if value == '' then
				return nil
			end

			begin
				case attr.class_name
				when 'Date'
					value = Date.new_with_date ::Date.parse(value)
				when 'Time'
					value = _parse_time value
				when 'Timestamp'
					value = Timestamp.new_with_datetime DateTime.parse(value)
				end
			rescue
				raise ArgumentError, "can't cast to #{attr.class_name} - #{value.class}:#{value}"
			end

			value
		end

		def _parse_time( value )
			value =~ /(\d+):(\d+) (\w+)/

			if $3 == 'am'
				hour = $1.to_i
			else
				hour = $1.to_i + 12
			end

			min = $2.to_i
			sec = 0

			Time.new(hour, min, sec)
		end

		public

		def each
			fetch_all.each do |row|
				yield row
			end
		end

		def insert_row( row, entity )
			_check_transaction
			factory = @adapter_context.adapter.expression_factory
			expr = factory.insert_statement(row, entity)
			evaluate expr
			@result.rows_affected
		end

		def _check_transaction
			unless @adapter_context.open_transaction? then
				raise "channel's context has no transaction"
			end
		end

		def delete_rows( qualifier, entity )
			_check_transaction
			factory = @adapter_context.adapter.expression_factory
			expr = factory.delete_statement(qualifier, entity)
			evaluate expr
			@result.rows_affected
		end

		def update_rows( row, qualifier, entity )
			_check_transaction
			factory = @adapter_context.adapter.expression_factory
			expr = factory.update_statement(row, qualifier, entity)
			evaluate expr
			@result.rows_affected			
		end

		# Creates new rows of primary keys.
		def primary_keys_for_new_row( count, entity )
			attrs = entity.primary_key_attributes
			if attrs.size > 1 then return nil end

			open
			attr = attrs.first
			column_name = attr.column_name

			if (maxnum = @pk_cache[entity]) > 0 then
				@pk_cache[entity] = maxnum + count
				return _primary_keys(attr, maxnum, count)
			end

			sql =  "SELECT #{column_name} FROM #{entity.external_name}"
			sql << " ORDER BY #{column_name} DESC RETURN RESULTS 1"

			factory = @adapter_context.adapter.expression_factory
			expr = factory.create entity
			expr.statement = sql
			result = evaluate expr
			row = result.fetch

			if row.nil? then
				maxnum = 0
			else
				maxnum = row[0]
			end

			@pk_cache[entity] = maxnum + count
			_primary_keys(attr, maxnum, count)
		end

		private

		def _primary_keys( attribute, maxnum, count )
			keys = []
			for x in 1..count
				maxnum += 1
				keys << {attribute.name => maxnum}
			end
			keys
		end

		public

		def describe_table_names
			[]
		end

		def describe_model( table_names )
			open

			connection = {'database'=>@openbase.database, 'host'=>@openbase.host,
				'user'=>@openbase.login, 'password'=>@openbase.password}

			entities = []
			table_names.each { |table| entities << _describe_entity(table) }
			entities.each    { |entity| _describe_attributes(entity) }

			model              = Model.new
			model.entities     = entities
			model.adapter_name = 'OpenBase'
			model.connection   = connection

			model
		end

		private

		def _describe_entity( table )
			entity = Entity.new
			entity.external_name = table
			entity.beautify_name
			entity.class_name = 'GenericRecord'
			entity
		end

		def _describe_attributes( entity )
			sql = "SELECT * FROM #{entity.external_name} RETURN RESULTS 1"
			result = @openbase.execute sql

			attrs = []
			result.column_infos.each do |info|
				if info.name == '_rowid' then
					next
				end

				attr = Attribute.new
				attr.column_name = info.name
				attr.beautify_name
				attr.external_type = info.type
				attr.class_name = OpenBaseAdapter.internal_type_name info.type

				if attr.class_name == 'String' then
					attr.width = 255 # openbase adapter doesn't know width of char
				end

				if info.name =~ /_id$/i then
					attr.allow_null = false
					entity.primary_key_attribute_names << attr.name
				else
					attr.allow_null = true
					entity.class_property_names << attr.name
				end
				entity.add_attribute attr
			end
		end
	end


	class OpenBaseContext < AdapterContext
		def create_channel
			channel = OpenBaseChannel.new self
			@channels << channel
			channel
		end

		def openbase
			if @channel then
				@channel.openbase
			else
				nil
			end
		end

		def begin_transaction
			if open_transaction? then
				raise "already begun"
			else
				@channel = nil
				@channels.each do |channel|
					unless channel.fetch_progress? then
						unless channel.open? then
							channel.open
						end
					end

					@channel = channel
					break
				end

				unless @channel then
					@channel = create_channel
					@channel.open
				end
			end

			openbase.begin

			@open_transaction     = true
			@commit_transaction   = false
			@rollback_transaction = false

			transaction_did_begin
		end

		def commit_transaction
			openbase.commit

			@open_transaction   = false
			@commit_transaction = true

			transaction_did_commit
		end

		def rollback_transaction
			openbase.rollback

			@open_transaction     = false
			@rollback_transaction = true

			transaction_did_rollback
		end
	end

	class OpenBaseExpression < SQLExpression
		STRING_TYPE = 1
		DATE_TYPE   = 2
		NUMBER_TYPE = 3

		class << self
			def adapter_class
				OpenBaseAdapter
			end
		end

		def must_use_bind_variable?( attribute )
			true
		end

		def should_use_bind_variable?( attribute )
			true
		end

		def use_bind_variables?
			true
		end

		def format_value( value, attribute )
			case attribute.adapter_value_type
			when Attribute::ADAPTER_CHARACTERS_TYPE then sql_for_string value
			when Attribute::ADAPTER_DATE_TYPE       then sql_for_date value
			when Attribute::ADAPTER_NUMBER_TYPE     then sql_for_number value
			when Attribute::ADAPTER_BYTES_TYPE      then sql_for_data value
			else
				raise "unsupported type: '#{attribute.external_type}'"
			end
		end

		def sql_for_string( string )
			if string.nil? then
				'NULL'
			else
				if @encoding then
					string = Utilities.encode(string, @encoding)
				end
				escape = Regexp.escape sql_escape_char
				string = string.to_s.gsub(/'/, '"')
				"'#{string}'"
			end
		end

		def sql_for_date( date )
			if date.nil? then
				'NULL'
			elsif date.class == Time then
				"'#{date.to_s(true)}'"
			elsif date.class == Timestamp then
				time = date.to_time.utc
				timestamp = Timestamp.new_with_datetime time
				timestamp.zone = '+0000'
				"'#{timestamp.to_s}'"
			else
				"'#{date.to_s}'"
			end
		end

		def sql_for_data( data )
			if data.nil? then
				'NULL'
			else
				data.to_s
			end
		end

		def sql_pattern( pattern, escape = nil )
			translated = pattern.dup
			translated.gsub!(/(\\\*|\*|\\\?|\?|%|_)/) do
				case $1
				when '\*' then '*'
				when '*'  then '%'
				when '\?' then '?'
				when '?'  then '_'
				when '%'  then "[%]"
				when '_'  then "[_]"
				end
			end
			translated
		end

		def bind_variable( attribute, value )
			binding                  = {}
			binding[NAME_KEY]        = attribute.name
			binding[PLACEHOLDER_KEY] = '?'
			binding[ATTRIBUTE_KEY]   = attribute
			binding[VALUE_KEY]       = value
			binding
		end

		def prepare_select( attributes, lock, fetch_spec )
			sql = super

			if limit = fetch_spec.limit then
				if limit > 0 then
					sql << " RETURN RESULTS #{limit}"
				end
			end

			@statement = sql
		end

		def join_clause
			if @join_entities.empty? then
				return ''
			end

			clause = ''
			@join_entities.each do |entity|
				name    = entity[0].external_name
				another = entity[1]
				clause << ", #{name} #{another}"
			end

			clause
		end

		def add_join( left, right, semantic )
			clause = assemble_join(left, right, semantic)

			if @where_clause.empty? then
				@where_clause = clause
			else
				@where_clause << " AND #{clause}"
			end

			clause
		end

		def assemble_join( left, right, semantic )
			case semantic
			when Relationship::INNER_JOIN
				op = '='
			when Relationship::LEFT_OUTER_JOIN
				op = '*'
			else
				raise "OpenBase adapter supports inner join or left outer join."
			end

			"#{left} #{op} #{right}"
		end
	end

	class OpenBaseExpressionFactory < SQLExpressionFactory
		def expression_class
			OpenBaseExpression
		end
	end

end

