module TapKit

	class ClassDescription
		class ClassDescriptionError < StandardError; end #:nodoc:

		DELETE_RULE_NULLIFY = :delete_rule_nullify
		DELETE_RULE_CASCADE = :delete_rule_cascade
		DELETE_RULE_DENY    = :delete_rule_deny

		class << self
			def new_with_entity_name( application, entity_name )
				entity = application.model_group.entity entity_name
				EntityClassDescription.new entity
			end

			def new_with_destination_key( destination_key )
			end
		end

		def create( editing_context, gid ); end
		def awake_from_fetch( object, editing_context ); end
		def awake_from_insertion( object, editing_context ); end
	end

	class EntityClassDescription < ClassDescription
		attr_reader :entity_name, :entity

		def initialize( entity )
			@entity = entity
			@entity_name = @entity.name
		end

		def create( editing_context, gid = nil )
			begin
				klass = @entity.class_name.split(/::/).inject(TapKit) {|c,name| c.const_get(name)}
				object = klass.new self
				unless object.is_a? GenericRecord
					raise  ClassDescriptionError, "invalid class_name: #{@entity.class_name}"
				end
			rescue NameError
				raise  ClassDescriptionError, "invalid class_name: #{@entity.class_name}"
			end
			object.editing_context = editing_context
			object.application     = editing_context.application

			to_many_relationship_keys.each do |key|
				object[key] = []
			end

			object
		end

		def delete_rule( relationship_key )
			relationship = @entity.relationship relationship_key
			relationship.delete_rule
		end

		def propagate_delete( object, editing_context )
			# to-one
			object.to_one_relationship_keys.each do |key|
				delete_rule = object.delete_rule(key)

				if delete_rule == Relationship::DELETE_RULE_DENY then
					next
				end

				if related = object.retrieve_stored_value(key) then
					related.will_read
				else
					next
				end

				case delete_rule
				when Relationship::DELETE_RULE_NULLIFY
					object.remove_object_from_both_sides_of_relationship(related, key)
				when Relationship::DELETE_RULE_CASCADE
					object.remove_object_from_both_sides_of_relationship(related, key)
					editing_context.delete related
				end
			end

			# to-many
			object.to_many_relationship_keys.each do |key|
				delete_rule = object.delete_rule(key)

				if delete_rule == Relationship::DELETE_RULE_DENY then
					next
				end

				# copy related objects to an other array to delete
				# to-many related objects.
				related_objects = object.retrieve_stored_value key
				related_objects.size # faulting
				copied_objects = []
				related_objects.each do |related_object|
					copied_objects << related_object
				end

				case delete_rule
				when Relationship::DELETE_RULE_NULLIFY
					copied_objects.each do |related_object|
						object.remove_object_from_both_sides_of_relationship(
							related_object, key)
					end
				when Relationship::DELETE_RULE_CASCADE
					copied_objects.each do |related_object|
						object.remove_object_from_both_sides_of_relationship(
							related_object, key)
						editing_context.delete related_object
					end
				end
			end
		end

		def class_description( detail_key )
			if relationship = @entity.relationship(detail_key) then
				destination = relationship.destination_entity
				destination.class_description
			else
				nil
			end
		end

		def inverse( relationship_key )
			if rel = @entity.relationship(relationship_key).inverse_relationship then
				rel.name
			else
				nil
			end
		end


		##
		## property keys
		##

		def all_property_keys
			keys =  all_attribute_keys
			keys += all_to_one_relationship_keys
			keys += all_to_many_relationship_keys
			keys
		end

		def property_keys
			keys =  attribute_keys
			keys += to_one_relationship_keys
			keys += to_many_relationship_keys
			keys
		end

		def all_attribute_keys
			keys = []
			@entity.attributes.each do |attr|
				keys << attr.name
			end
			keys
		end

		def attribute_keys
			keys = []
			@entity.class_properties.each do |property|
				if @entity.attributes.include? property then
					keys << property.name
				end
			end
			keys
		end

		def all_to_many_relationship_keys
			keys = []
			@entity.relationships.each do |re|
				if re.to_many? then
					keys << re.name
				end
			end
			keys
		end

		def to_many_relationship_keys
			keys = []
			@entity.relationships.each do |re|
				if re.to_many? then
					keys << re.name
				end
			end
			keys
		end

		def all_to_one_relationship_keys
			keys = []
			@entity.relationships.each do |re|
				unless re.to_many? then
					keys << re.name
				end
			end
			keys
		end

		def to_one_relationship_keys
			keys = []
			@entity.relationships.each do |re|
				unless re.to_many? then
					keys << re.name
				end
			end
			keys
		end


		##
		## validation
		##

		def validate_object_for_delete( object )
			@entity.relationships.each do |relation|
				if (relation.delete_rule == DELETE_RULE_DENY) and \
					(value = object.retrieve_stored_value(relation.name)) then
					msg = "The delete opration should not proceed."
					key = relation.name
					error_object = {key => object}
					error = ValidationError.new(msg, key, error_object)
					raise error
				end
			end
		end

		def validate_object_for_save( object )
			object.property_keys.each do |key|
				validate_value(key, object[key])
			end
		end

		# Attribute#validate_value, Relationship#validate_value$B$rMxMQ$9$k(B
		def validate_value( key, value )
			if attr = @entity.attribute(key) then
				attr.validate_value value
			elsif relation = @entity.relationship(key) then
				relation.validate_value value
			else
				msg    = "The '#{key}' property of #{@entity_name} is not found"
				object = {key => value}
				error  = ValidationError.new(msg, key, object)
				raise error
			end
		end
	end

end
