class TestEditingContextUpdate < Test::Unit::TestCase
	include TapKit

	def setup
		@app = Application.new MODELS
		@ec  = @app.create_editing_context
	end

	def test_tracking_changes
		qualifier = Qualifier.format "title like '*'"
		fetchspec = FetchSpec.new('cd', qualifier)
		fetchspec.limit = 1
		object    = @ec.fetch(fetchspec).first

		assert_equal(false, @ec.changed?)
		object['title'] = 'test'
		assert_equal(true, @ec.changed?)
	end

	def test_update_single_record
		# @app.log_options[:sql] = true
		qualifier = Qualifier.format "title like '*'"
		fetchspec = FetchSpec.new('cd', qualifier)
		fetchspec.limit = 1
		object    = @ec.fetch(fetchspec).first
		original  = object['title']

		# update the object
		updated = 'test updated'
		object['title'] = updated
		@ec.save

		# get and check update object
		qualifier = Qualifier.format("title like %@", [updated])
		fetchspec = FetchSpec.new('cd', qualifier)
		fetchspec.refresh = true
		object    = @ec.fetch(fetchspec).first
		assert_equal(updated, object['title'])

		# restore updated object
		object['title'] = original
		@ec.save

		qualifier = Qualifier.format("title like %@", [original])
		fetchspec = FetchSpec.new('cd', qualifier)
		fetchspec.refresh = true
		object = @ec.fetch(fetchspec).first
		assert_equal(original, object['title'])
	end

	def test_update_multiple_records
		#@app.log_options[:sql] = true
		qualifier = Qualifier.format "title like '*'"
		sort      = SortOrdering.new('release')
		fetchspec = FetchSpec.new('cd', qualifier, [sort])
		fetchspec.limit = 10
		objects = @ec.fetch(fetchspec)

		# update the objects
		updated   = "test #{Time.now}"
		originals = []
		objects.each do |object|
			originals << object['title']
			object['title'] = updated
		end
		@ec.save

		# get and check update object
		qualifier = Qualifier.format("title like %@", [updated])
		fetchspec = FetchSpec.new('cd', qualifier, [sort])
		fetchspec.limit = 10
		objects   = @ec.fetch(fetchspec)
		objects.each_with_index do |object, index|
			assert_equal(updated, object['title'])

			# restore updated object
			object['title'] = originals[index]
		end
		@ec.save
	end

	def test_mapping
		time = Time.now
		date = Date.today
		timestamp = Timestamp.now
		test = {'string'=>"test_mapping<#{time.to_s}>", 'integer'=>timestamp.to_time.to_i, 
			'float'=>0.1, 'decimal'=>nil, 'date'=>date, 'time'=>time,
			'timestamp'=>timestamp, 'binary'=>nil}

		fetchspec = FetchSpec.new 'Mapping'
		fetchspec.limit = 1
		objects = @ec.fetch(fetchspec)
		object  = objects.first

		# record original
		original = {'string'=>object['string'], 'integer'=>object['integer'],
			'float'=>object['float'], 'decimal'=>object['decimal'], 'date'=>object['date'],
			'time'=>object['time'], 'timestamp'=>object['timestamp'],
			'binary'=>object['binary']}

		# update
		test.each do |key, value|
			object[key] = value
		end
		@ec.save

		# check
		fetchspec.refresh = true
		object = @ec.fetch(fetchspec).first
		test.each do |key, value|
			assert_equal(value, object[key])
		end

		# restore
		original.each do |key, value|
			object[key] = value
		end
		@ec.save
	end

	def test_update_relationships
		#@app.log_options[:sql] = true
		fetchspec = FetchSpec.new 'Book'
		fetchspec.limit = 1
		objects = @ec.fetch(fetchspec)
		object  = objects.first

		# update a relationship to a new object
		original = object['publisher']['name'].dup
		new_obj = @ec.create 'Publisher'
		test = "test_update_relationships<#{Time.now.to_s}>"
		new_obj['name'] = test
		object['publisher'] = new_obj
		@ec.save

		# check
		qualifier = Qualifier.format("publisher.name == %@", [test])
		fetchspec = FetchSpec.new('Book', qualifier)
		fetchspec.limit = 1
		fetchspec.refresh = true
		objects = @ec.fetch(fetchspec)
		object  = objects.first
		assert_equal(new_obj, object['publisher'])

		# restore (update a relationship to a existed object)
		qualifier = Qualifier.format("name == %@", [original])
		fetchspec = FetchSpec.new('Publisher', qualifier)
		fetchspec.limit = 1
		fetchspec.refresh = true
		origin_obj = @ec.fetch(fetchspec).first
		object['publisher'] = origin_obj
		@ec.save

		# check restored
		qualifier = Qualifier.format("publisher.name == %@", [original])
		fetchspec = FetchSpec.new('Book', qualifier)
		fetchspec.limit = 1
		#fetchspec.refresh = true
		objects = @app.create_editing_context.fetch(fetchspec)
		assert_equal(false, objects.empty?)
	end
end

