/*      MikMod sound library
   (c) 1998, 1999 Miodrag Vallat and others - see file AUTHORS for
   complete list.

   This library is free software; you can redistribute it and/or modify
   it under the terms timip_of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 timip_of
   the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty timip_of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.

   You should have received a copy timip_of the GNU Library General Public
   License along with this library; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA.
 */

/*==============================================================================

  $Id: timip_load_uni.c,v 1.13 1999/10/25 16:31:41 miod Exp $

  UNIMOD (libmikmod's and APlayer's internal module format) loader

==============================================================================*/

#ifdef HAVE_CONFIG_H
#include "timip_config.h"
#endif

#include <string.h>

#include "timip_unimod_priv.h"

/*========== Module structure */

typedef struct UNIHEADER
  {
    CHAR id[4];
    UBYTE numchn;
    UWORD numpos;
    UWORD reppos;
    UWORD numpat;
    UWORD numtrk;
    UWORD numins;
    UWORD numsmp;
    UBYTE initspeed;
    UBYTE inittempo;
    UBYTE initvolume;
    UBYTE flags;
    UBYTE numvoices;

    UBYTE positions[256];
    UBYTE panning[32];
  }
UNIHEADER;

typedef struct UNISMP05
  {
    UWORD c2spd;
    UWORD transpose;
    UBYTE volume;
    UBYTE panning;
    ULONG length;
    ULONG loopstart;
    ULONG loopend;
    UWORD flags;
    CHAR *samplename;
    UBYTE vibtype;
    UBYTE vibsweep;
    UBYTE vibdepth;
    UBYTE vibrate;
  }
UNISMP05;

/*========== Loader variables */

static UWORD universion;
static UNIHEADER mh;

#define UNI_SMPINCR 64
static UNISMP05 *wh = NULL, *s = NULL;

/*========== Loader code */

static char *
readstring (void)
{
  char *s = NULL;
  UWORD len;

  len = _mm_read_I_UWORD (timip_modreader);
  if (len)
    {
      s = _mm_malloc (len + 1);
      _mm_read_UBYTES (s, len, timip_modreader);
      s[len] = 0;
    }
  return s;
}

BOOL 
timip_UNI_Test (void)
{
  char id[6];

  if (!_mm_read_UBYTES (id, 6, timip_modreader))
    return 0;

  /* UNIMod created by MikCvt */
  if (!(memcmp (id, "UN0", 3)))
    {
      if ((id[3] >= '4') && (id[3] <= '6'))
	return 1;
    }
  /* UNIMod created by APlayer */
  if (!(memcmp (id, "APUN\01", 5)))
    {
      if ((id[5] >= 1) && (id[5] <= 4))
	return 1;
    }
  return 0;
}

BOOL 
timip_UNI_Init (void)
{
  return 1;
}

void 
timip_UNI_Cleanup (void)
{
  _mm_free (wh);
  s = NULL;
}

static UBYTE *
readtrack (void)
{
  UBYTE *t;
  UWORD len;
  int cur = 0, chunk;

  if (universion >= 6)
    len = _mm_read_M_UWORD (timip_modreader);
  else
    len = _mm_read_I_UWORD (timip_modreader);

  if (!len)
    return NULL;
  if (!(t = _mm_malloc (len)))
    return NULL;
  _mm_read_UBYTES (t, len, timip_modreader);

  /* Check if the track is correct */
  while (1)
    {
      chunk = t[cur++];
      if (!chunk)
	break;
      chunk = (chunk & 0x1f) - 1;
      while (chunk > 0)
	{
	  int opcode, oplen;

	  if (cur >= len)
	    {
	      free (t);
	      return NULL;
	    }
	  opcode = t[cur];

	  /* Remap opcodes */
	  if (universion <= 5)
	    {
	      if (opcode > 29)
		{
		  free (t);
		  return NULL;
		}
	      switch (opcode)
		{
		  /* UNI_NOTE .. UNI_S3MEFFECTQ are the same */
		case 25:
		  opcode = UNI_S3MEFFECTT;
		  break;
		case 26:
		  opcode = UNI_XMEFFECTA;
		  break;
		case 27:
		  opcode = UNI_XMEFFECTG;
		  break;
		case 28:
		  opcode = UNI_XMEFFECTH;
		  break;
		case 29:
		  opcode = UNI_XMEFFECTP;
		  break;
		}
	    }
	  else
	    {
	      if (opcode > UNI_ITEFFECTP)
		{
		  /* APlayer < 1.03 does not have ITEFFECTT */
		  if (universion < 0x103)
		    opcode++;
		  /* APlayer < 1.02 does not have ITEFFECTZ */
		  if ((opcode > UNI_ITEFFECTY) && (universion < 0x102))
		    opcode++;
		}
	    }

	  if ((!opcode) || (opcode >= UNI_LAST))
	    {
	      free (t);
	      return NULL;
	    }
	  oplen = timip_unioperands[opcode] + 1;
	  cur += oplen;
	  chunk -= oplen;
	}
      if ((chunk < 0) || (cur >= len))
	{
	  free (t);
	  return NULL;
	}
    }
  return t;
}

static BOOL 
loadsmp6 (void)
{
  int t;
  SAMPLE *s;

  s = timip_of.samples;
  for (t = 0; t < timip_of.numsmp; t++, s++)
    {
      int flags;

      flags = _mm_read_M_UWORD (timip_modreader);
      s->flags = 0;
      if (flags & 0x0100)
	s->flags |= SF_REVERSE;
      if (flags & 0x0004)
	s->flags |= SF_STEREO;
      if (flags & 0x0002)
	s->flags |= SF_SIGNED;
      if (flags & 0x0001)
	s->flags |= SF_16BITS;
      /* convert flags */
      if (universion >= 0x102)
	{
	  if (flags & 0x0800)
	    s->flags |= SF_UST_LOOP;
	  if (flags & 0x0400)
	    s->flags |= SF_OWNPAN;
	  if (flags & 0x0200)
	    s->flags |= SF_SUSTAIN;
	  if (flags & 0x0080)
	    s->flags |= SF_BIDI;
	  if (flags & 0x0040)
	    s->flags |= SF_LOOP;
	  if (flags & 0x0020)
	    s->flags |= SF_ITPACKED;
	  if (flags & 0x0010)
	    s->flags |= SF_DELTA;
	  if (flags & 0x0008)
	    s->flags |= SF_BIG_ENDIAN;
	}
      else
	{
	  if (flags & 0x400)
	    s->flags |= SF_UST_LOOP;
	  if (flags & 0x200)
	    s->flags |= SF_OWNPAN;
	  if (flags & 0x080)
	    s->flags |= SF_SUSTAIN;
	  if (flags & 0x040)
	    s->flags |= SF_BIDI;
	  if (flags & 0x020)
	    s->flags |= SF_LOOP;
	  if (flags & 0x010)
	    s->flags |= SF_BIG_ENDIAN;
	  if (flags & 0x008)
	    s->flags |= SF_DELTA;
	}

      s->speed = _mm_read_M_ULONG (timip_modreader);
      s->volume = _mm_read_UBYTE (timip_modreader);
      s->panning = _mm_read_M_UWORD (timip_modreader);
      s->length = _mm_read_M_ULONG (timip_modreader);
      s->loopstart = _mm_read_M_ULONG (timip_modreader);
      s->loopend = _mm_read_M_ULONG (timip_modreader);
      s->susbegin = _mm_read_M_ULONG (timip_modreader);
      s->susend = _mm_read_M_ULONG (timip_modreader);
      s->globvol = _mm_read_UBYTE (timip_modreader);
      s->vibflags = _mm_read_UBYTE (timip_modreader);
      s->vibtype = _mm_read_UBYTE (timip_modreader);
      s->vibsweep = _mm_read_UBYTE (timip_modreader);
      s->vibdepth = _mm_read_UBYTE (timip_modreader);
      s->vibrate = _mm_read_UBYTE (timip_modreader);

      s->samplename = readstring ();

      if (_mm_eof (timip_modreader))
	{
	  _mm_errno = MMERR_LOADING_SAMPLEINFO;
	  return 0;
	}
    }
  return 1;
}

static BOOL 
loadinstr6 (void)
{
  int t, w;
  INSTRUMENT *i;

  i = timip_of.instruments;
  for (t = 0; t < timip_of.numins; t++, i++)
    {
      i->flags = _mm_read_UBYTE (timip_modreader);
      i->nnatype = _mm_read_UBYTE (timip_modreader);
      i->dca = _mm_read_UBYTE (timip_modreader);
      i->dct = _mm_read_UBYTE (timip_modreader);
      i->globvol = _mm_read_UBYTE (timip_modreader);
      i->panning = _mm_read_M_UWORD (timip_modreader);
      i->pitpansep = _mm_read_UBYTE (timip_modreader);
      i->pitpancenter = _mm_read_UBYTE (timip_modreader);
      i->rvolvar = _mm_read_UBYTE (timip_modreader);
      i->rpanvar = _mm_read_UBYTE (timip_modreader);
      i->volfade = _mm_read_M_UWORD (timip_modreader);

#define UNI_LoadEnvelope6(name) 											\
		i->name##flg=_mm_read_UBYTE(timip_modreader);							\
		i->name##pts=_mm_read_UBYTE(timip_modreader);							\
		i->name##susbeg=_mm_read_UBYTE(timip_modreader);						\
		i->name##susend=_mm_read_UBYTE(timip_modreader);						\
		i->name##beg=_mm_read_UBYTE(timip_modreader);							\
		i->name##end=_mm_read_UBYTE(timip_modreader);							\
		for(w=0;w<(universion>=0x100?32:i->name##pts);w++) {				\
			i->name##env[w].pos=_mm_read_M_SWORD(timip_modreader);				\
			i->name##env[w].val=_mm_read_M_SWORD(timip_modreader);				\
		}

      UNI_LoadEnvelope6 (vol);
      UNI_LoadEnvelope6 (pan);
      UNI_LoadEnvelope6 (pit);
#undef UNI_LoadEnvelope6

      if (universion == 0x103)
	_mm_read_M_UWORDS (i->samplenumber, 120, timip_modreader);
      else
	for (w = 0; w < 120; w++)
	  i->samplenumber[w] = _mm_read_UBYTE (timip_modreader);
      _mm_read_UBYTES (i->samplenote, 120, timip_modreader);

      i->insname = readstring ();

      if (_mm_eof (timip_modreader))
	{
	  _mm_errno = MMERR_LOADING_SAMPLEINFO;
	  return 0;
	}
    }
  return 1;
}

static BOOL 
loadinstr5 (void)
{
  INSTRUMENT *i;
  int t;
  UWORD wavcnt = 0;
  UBYTE vibtype, vibsweep, vibdepth, vibrate;

  i = timip_of.instruments;
  for (timip_of.numsmp = t = 0; t < timip_of.numins; t++, i++)
    {
      int u, numsmp;

      numsmp = _mm_read_UBYTE (timip_modreader);

      memset (i->samplenumber, 0xff, INSTNOTES * sizeof (UWORD));
      for (u = 0; u < 96; u++)
	i->samplenumber[u] = timip_of.numsmp + _mm_read_UBYTE (timip_modreader);

#define UNI_LoadEnvelope5(name) 										\
		i->name##flg=_mm_read_UBYTE(timip_modreader);						\
		i->name##pts=_mm_read_UBYTE(timip_modreader);						\
		i->name##susbeg=_mm_read_UBYTE(timip_modreader);					\
		i->name##susend=i->name##susbeg;							\
		i->name##beg=_mm_read_UBYTE(timip_modreader);						\
		i->name##end=_mm_read_UBYTE(timip_modreader);						\
		for(u=0;u<12;u++) {												\
			i->name##env[u].pos=_mm_read_I_SWORD(timip_modreader);			\
			i->name##env[u].val=_mm_read_I_SWORD(timip_modreader);			\
		}

      UNI_LoadEnvelope5 (vol);
      UNI_LoadEnvelope5 (pan);
#undef UNI_LoadEnvelope5

      vibtype = _mm_read_UBYTE (timip_modreader);
      vibsweep = _mm_read_UBYTE (timip_modreader);
      vibdepth = _mm_read_UBYTE (timip_modreader);
      vibrate = _mm_read_UBYTE (timip_modreader);

      i->volfade = _mm_read_I_UWORD (timip_modreader);
      i->insname = readstring ();

      for (u = 0; u < numsmp; u++, s++, timip_of.numsmp++)
	{
	  /* Allocate more room for sample information if necessary */
	  if (timip_of.numsmp + u == wavcnt)
	    {
	      wavcnt += UNI_SMPINCR;
	      if (!(wh = realloc (wh, wavcnt * sizeof (UNISMP05))))
		{
		  _mm_errno = MMERR_OUT_OF_MEMORY;
		  return 0;
		}
	      s = wh + (wavcnt - UNI_SMPINCR);
	    }

	  s->c2spd = _mm_read_I_UWORD (timip_modreader);
	  s->transpose = _mm_read_SBYTE (timip_modreader);
	  s->volume = _mm_read_UBYTE (timip_modreader);
	  s->panning = _mm_read_UBYTE (timip_modreader);
	  s->length = _mm_read_I_ULONG (timip_modreader);
	  s->loopstart = _mm_read_I_ULONG (timip_modreader);
	  s->loopend = _mm_read_I_ULONG (timip_modreader);
	  s->flags = _mm_read_I_UWORD (timip_modreader);
	  s->samplename = readstring ();

	  s->vibtype = vibtype;
	  s->vibsweep = vibsweep;
	  s->vibdepth = vibdepth;
	  s->vibrate = vibrate;

	  if (_mm_eof (timip_modreader))
	    {
	      free (wh);
	      wh = NULL;
	      _mm_errno = MMERR_LOADING_SAMPLEINFO;
	      return 0;
	    }
	}
    }

  /* sanity check */
  if (!timip_of.numsmp)
    {
      if (wh)
	{
	  free (wh);
	  wh = NULL;
	}
      _mm_errno = MMERR_LOADING_SAMPLEINFO;
      return 0;
    }
  return 1;
}

static BOOL 
loadsmp5 (void)
{
  int t, u;
  SAMPLE *q;
  INSTRUMENT *d;

  q = timip_of.samples;
  s = wh;
  for (u = 0; u < timip_of.numsmp; u++, q++, s++)
    {
      q->samplename = s->samplename;

      q->length = s->length;
      q->loopstart = s->loopstart;
      q->loopend = s->loopend;
      q->volume = s->volume;
      q->speed = s->c2spd;
      q->panning = s->panning;
      q->vibtype = s->vibtype;
      q->vibsweep = s->vibsweep;
      q->vibdepth = s->vibdepth;
      q->vibrate = s->vibrate;

      /* convert flags */
      q->flags = 0;
      if (s->flags & 128)
	q->flags |= SF_REVERSE;
      if (s->flags & 64)
	q->flags |= SF_SUSTAIN;
      if (s->flags & 32)
	q->flags |= SF_BIDI;
      if (s->flags & 16)
	q->flags |= SF_LOOP;
      if (s->flags & 8)
	q->flags |= SF_BIG_ENDIAN;
      if (s->flags & 4)
	q->flags |= SF_DELTA;
      if (s->flags & 2)
	q->flags |= SF_SIGNED;
      if (s->flags & 1)
	q->flags |= SF_16BITS;
    }

  d = timip_of.instruments;
  s = wh;
  for (u = 0; u < timip_of.numins; u++, d++)
    for (t = 0; t < INSTNOTES; t++)
      d->samplenote[t] = (d->samplenumber[t] >= timip_of.numsmp) ?
	255 : (t + s[d->samplenumber[t]].transpose);

  free (wh);
  wh = NULL;

  return 1;
}

BOOL 
timip_UNI_Load (BOOL curious)
{
  int t;
  char *modtype, *oldtype = NULL;
  INSTRUMENT *d;
  SAMPLE *q;

  /* read module header */
  _mm_read_UBYTES (mh.id, 4, timip_modreader);
  if (mh.id[3] != 'N')
    universion = mh.id[3] - '0';
  else
    universion = 0x100;

  if (universion >= 6)
    {
      if (universion == 6)
	_mm_read_UBYTE (timip_modreader);
      else
	universion = _mm_read_M_UWORD (timip_modreader);

      mh.flags = _mm_read_M_UWORD (timip_modreader);
      mh.numchn = _mm_read_UBYTE (timip_modreader);
      mh.numvoices = _mm_read_UBYTE (timip_modreader);
      mh.numpos = _mm_read_M_UWORD (timip_modreader);
      mh.numpat = _mm_read_M_UWORD (timip_modreader);
      mh.numtrk = _mm_read_M_UWORD (timip_modreader);
      mh.numins = _mm_read_M_UWORD (timip_modreader);
      mh.numsmp = _mm_read_M_UWORD (timip_modreader);
      mh.reppos = _mm_read_M_UWORD (timip_modreader);
      mh.initspeed = _mm_read_UBYTE (timip_modreader);
      mh.inittempo = _mm_read_UBYTE (timip_modreader);
      mh.initvolume = _mm_read_UBYTE (timip_modreader);

      mh.flags &= (UF_XMPERIODS | UF_LINEAR | UF_INST | UF_NNA);
    }
  else
    {
      mh.numchn = _mm_read_UBYTE (timip_modreader);
      mh.numpos = _mm_read_I_UWORD (timip_modreader);
      mh.reppos = (universion == 5) ? _mm_read_I_UWORD (timip_modreader) : 0;
      mh.numpat = _mm_read_I_UWORD (timip_modreader);
      mh.numtrk = _mm_read_I_UWORD (timip_modreader);
      mh.numins = _mm_read_I_UWORD (timip_modreader);
      mh.initspeed = _mm_read_UBYTE (timip_modreader);
      mh.inittempo = _mm_read_UBYTE (timip_modreader);
      _mm_read_UBYTES (mh.positions, 256, timip_modreader);
      _mm_read_UBYTES (mh.panning, 32, timip_modreader);
      mh.flags = _mm_read_UBYTE (timip_modreader);

      mh.flags &= (UF_XMPERIODS | UF_LINEAR);
      mh.flags |= UF_INST | UF_NOWRAP;
    }

  /* set module parameters */
  timip_of.flags = mh.flags;
  timip_of.numchn = mh.numchn;
  timip_of.numpos = mh.numpos;
  timip_of.numpat = mh.numpat;
  timip_of.numtrk = mh.numtrk;
  timip_of.numins = mh.numins;
  timip_of.reppos = mh.reppos;
  timip_of.initspeed = mh.initspeed;
  timip_of.inittempo = mh.inittempo;

  timip_of.songname = readstring ();
  if (universion < 0x102)
    oldtype = readstring ();
  if (oldtype)
    {
      int len = strlen (oldtype) + 20;
      if (!(modtype = _mm_malloc (len)))
	return 0;
#ifdef	TIMIP_HAVE_SNPRINTF
      snprintf (modtype, len, "%s (was %s)", (universion >= 0x100) ? "APlayer" : "MikCvt2", oldtype);
#else
      sprintf (modtype, "%s (was %s)", (universion >= 0x100) ? "APlayer" : "MikCvt2", oldtype);
#endif
    }
  else
    {
      if (!(modtype = _mm_malloc (10)))
	return 0;
#ifdef	TIMIP_HAVE_SNPRINTF
      snprintf (modtype, 10, "%s", (universion >= 0x100) ? "APlayer" : "MikCvt3");
#else
      sprintf (modtype, "%s", (universion >= 0x100) ? "APlayer" : "MikCvt3");
#endif
    }
  timip_of.modtype = strdup (modtype);
  free (modtype);
  free (oldtype);
  timip_of.comment = readstring ();

  if (universion >= 6)
    {
      timip_of.numvoices = mh.numvoices;
      timip_of.initvolume = mh.initvolume;
    }

  if (_mm_eof (timip_modreader))
    {
      _mm_errno = MMERR_LOADING_HEADER;
      return 0;
    }

  /* positions */
  if (!timip_AllocPositions (timip_of.numpos))
    return 0;
  if (universion >= 6)
    {
      if (universion >= 0x100)
	_mm_read_M_UWORDS (timip_of.positions, timip_of.numpos, timip_modreader);
      else
	for (t = 0; t < timip_of.numpos; t++)
	  timip_of.positions[t] = _mm_read_UBYTE (timip_modreader);
      _mm_read_M_UWORDS (timip_of.panning, timip_of.numchn, timip_modreader);
      _mm_read_UBYTES (timip_of.chanvol, timip_of.numchn, timip_modreader);
    }
  else
    {
      if ((mh.numpos > 256) || (mh.numchn > 32))
	{
	  _mm_errno = MMERR_LOADING_HEADER;
	  return 0;
	}
      for (t = 0; t < timip_of.numpos; t++)
	timip_of.positions[t] = mh.positions[t];
      for (t = 0; t < timip_of.numchn; t++)
	timip_of.panning[t] = mh.panning[t];
    }

  /* instruments and samples */
  if (universion >= 6)
    {
      timip_of.numsmp = mh.numsmp;
      if (!timip_AllocSamples ())
	return 0;
      if (!loadsmp6 ())
	return 0;

      if (timip_of.flags & UF_INST)
	{
	  if (!timip_AllocInstruments ())
	    return 0;
	  if (!loadinstr6 ())
	    return 0;
	}
    }
  else
    {
      if (!timip_AllocInstruments ())
	return 0;
      if (!loadinstr5 ())
	return 0;
      if (!timip_AllocSamples ())
	{
	  if (wh)
	    {
	      free (wh);
	      wh = NULL;
	    }
	  return 0;
	}
      if (!loadsmp5 ())
	return 0;

      /* check if the original file had no instruments */
      if (timip_of.numsmp == timip_of.numins)
	{
	  for (t = 0, d = timip_of.instruments; t < timip_of.numins; t++, d++)
	    {
	      int u;

	      if ((d->volpts) || (d->panpts) || (d->globvol != 64))
		break;
	      for (u = 0; u < 96; u++)
		if ((d->samplenumber[u] != t) || (d->samplenote[u] != u))
		  break;
	      if (u != 96)
		break;
	    }
	  if (t == timip_of.numins)
	    {
	      timip_of.flags &= ~UF_INST;
	      timip_of.flags &= ~UF_NOWRAP;
	      for (t = 0, d = timip_of.instruments, q = timip_of.samples; t < timip_of.numins; t++, d++, q++)
		{
		  q->samplename = d->insname;
		  d->insname = NULL;
		}
	    }
	}
    }

  /* patterns */
  if (!timip_AllocPatterns ())
    return 0;
  if (universion >= 6)
    {
      _mm_read_M_UWORDS (timip_of.pattrows, timip_of.numpat, timip_modreader);
      _mm_read_M_UWORDS (timip_of.patterns, timip_of.numpat * timip_of.numchn, timip_modreader);
    }
  else
    {
      _mm_read_I_UWORDS (timip_of.pattrows, timip_of.numpat, timip_modreader);
      _mm_read_I_UWORDS (timip_of.patterns, timip_of.numpat * timip_of.numchn, timip_modreader);
    }

  /* tracks */
  if (!timip_AllocTracks ())
    return 0;
  for (t = 0; t < timip_of.numtrk; t++)
    if (!(timip_of.tracks[t] = readtrack ()))
      {
	_mm_errno = MMERR_LOADING_TRACK;
	return 0;
      }

  return 1;
}

CHAR *
timip_UNI_LoadTitle (void)
{
  UBYTE ver;
  int posit[3] =
  {304, 306, 26};

  _mm_fseek (timip_modreader, 3, SEEK_SET);
  ver = _mm_read_UBYTE (timip_modreader);
  if (ver == 'N')
    ver = '6';

  _mm_fseek (timip_modreader, posit[ver - '4'], SEEK_SET);
  return readstring ();
}

/*========== Loader information */

MLOADER timip_load_uni =
{
  NULL,
  "UNI",
  "APUN (APlayer) and UNI (MikMod)",
  timip_UNI_Init,
  timip_UNI_Test,
  timip_UNI_Load,
  timip_UNI_Cleanup,
  timip_UNI_LoadTitle
};

/* ex:set ts=4: */
