/*      MikMod sound library
   (c) 1998, 1999 Miodrag Vallat and others - see file AUTHORS for
   complete list.

   This library is free software; you can redistribute it and/or modify
   it under the terms timip_of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 timip_of
   the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty timip_of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.

   You should have received a copy timip_of the GNU Library General Public
   License along with this library; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA.
 */

/*==============================================================================

  $Id: timip_load_mtm.c,v 1.27 1999/10/25 16:31:41 miod Exp $

  MTM module loader

==============================================================================*/

#ifdef HAVE_CONFIG_H
#include "timip_config.h"
#endif

#include <string.h>

#include "timip_unimod_priv.h"

/*========== Module structure */

typedef struct MTMHEADER
  {
    UBYTE id[3];		/* MTM file marker */
    UBYTE version;		/* upper major, lower nibble minor version number */
    CHAR songname[20];		/* ASCIIZ songname */
    UWORD numtracks;		/* number timip_of tracks saved */
    UBYTE lastpattern;		/* last pattern number saved */
    UBYTE lastorder;		/* last order number to play (songlength-1) */
    UWORD commentsize;		/* length timip_of comment field */
    UBYTE numsamples;		/* number timip_of samples saved  */
    UBYTE attribute;		/* attribute byte (unused) */
    UBYTE beatspertrack;
    UBYTE numchannels;		/* number timip_of channels used  */
    UBYTE panpos[32];		/* timip_voice pan positions */
  }
MTMHEADER;

typedef struct MTMSAMPLE
  {
    CHAR samplename[22];
    ULONG length;
    ULONG reppos;
    ULONG repend;
    UBYTE timip_finetune;
    UBYTE volume;
    UBYTE attribute;
  }
MTMSAMPLE;

typedef struct MTMNOTE
  {
    UBYTE a, b, c;
  }
MTMNOTE;

/*========== Loader variables */

static MTMHEADER *mh = NULL;
static MTMNOTE *mtmtrk = NULL;
static UWORD pat[32];

static CHAR MTM_Version[] = "MTM";

/*========== Loader code */

BOOL 
timip_MTM_Test (void)
{
  UBYTE id[3];

  if (!_mm_read_UBYTES (id, 3, timip_modreader))
    return 0;
  if (!memcmp (id, "MTM", 3))
    return 1;
  return 0;
}

BOOL 
timip_MTM_Init (void)
{
  if (!(mtmtrk = (MTMNOTE *) _mm_calloc (64, sizeof (MTMNOTE))))
    return 0;
  if (!(mh = (MTMHEADER *) _mm_malloc (sizeof (MTMHEADER))))
    return 0;

  return 1;
}

void 
timip_MTM_Cleanup (void)
{
  _mm_free (mtmtrk);
  _mm_free (mh);
}

static UBYTE *
MTM_Convert (void)
{
  int t;
  UBYTE a, b, inst, note, eff, dat;

  timip_UniReset ();
  for (t = 0; t < 64; t++)
    {
      a = mtmtrk[t].a;
      b = mtmtrk[t].b;
      inst = ((a & 0x3) << 4) | (b >> 4);
      note = a >> 2;
      eff = b & 0xf;
      dat = mtmtrk[t].c;

      if (inst)
	UniInstrument (inst - 1);
      if (note)
	UniNote (note + 2 * OCTAVE);

      /* MTM bug workaround : when the effect is volslide, slide-up *always*
         overrides slide-down. */
      if (eff == 0xa && (dat & 0xf0))
	dat &= 0xf0;

      /* Convert pattern jump from Dec to Hex */
      if (eff == 0xd)
	dat = (((dat & 0xf0) >> 4) * 10) + (dat & 0xf);
      timip_UniPTEffect (eff, dat);
      timip_UniNewline ();
    }
  return timip_UniDup ();
}

BOOL 
timip_MTM_Load (BOOL curious)
{
  int t, u;
  MTMSAMPLE s;
  SAMPLE *q;

  /* try to read module header  */
  _mm_read_UBYTES (mh->id, 3, timip_modreader);
  mh->version = _mm_read_UBYTE (timip_modreader);
  _mm_read_string (mh->songname, 20, timip_modreader);
  mh->numtracks = _mm_read_I_UWORD (timip_modreader);
  mh->lastpattern = _mm_read_UBYTE (timip_modreader);
  mh->lastorder = _mm_read_UBYTE (timip_modreader);
  mh->commentsize = _mm_read_I_UWORD (timip_modreader);
  mh->numsamples = _mm_read_UBYTE (timip_modreader);
  mh->attribute = _mm_read_UBYTE (timip_modreader);
  mh->beatspertrack = _mm_read_UBYTE (timip_modreader);
  mh->numchannels = _mm_read_UBYTE (timip_modreader);
  _mm_read_UBYTES (mh->panpos, 32, timip_modreader);

  if (_mm_eof (timip_modreader))
    {
      _mm_errno = MMERR_LOADING_HEADER;
      return 0;
    }

  /* set module variables */
  timip_of.initspeed = 6;
  timip_of.inittempo = 125;
  timip_of.modtype = strdup (MTM_Version);
  timip_of.numchn = mh->numchannels;
  timip_of.numtrk = mh->numtracks + 1;	/* get number timip_of channels */
  timip_of.songname = timip_DupStr (mh->songname, 20, 1);	/* make a cstr timip_of songname */
  timip_of.numpos = mh->lastorder + 1;	/* copy the songlength */
  timip_of.numpat = mh->lastpattern + 1;
  timip_of.reppos = 0;
  for (t = 0; t < 32; t++)
    timip_of.panning[t] = mh->panpos[t] << 4;
  timip_of.numins = timip_of.numsmp = mh->numsamples;

  if (!timip_AllocSamples ())
    return 0;
  q = timip_of.samples;
  for (t = 0; t < timip_of.numins; t++)
    {
      /* try to read sample info */
      _mm_read_string (s.samplename, 22, timip_modreader);
      s.length = _mm_read_I_ULONG (timip_modreader);
      s.reppos = _mm_read_I_ULONG (timip_modreader);
      s.repend = _mm_read_I_ULONG (timip_modreader);
      s.timip_finetune = _mm_read_UBYTE (timip_modreader);
      s.volume = _mm_read_UBYTE (timip_modreader);
      s.attribute = _mm_read_UBYTE (timip_modreader);

      if (_mm_eof (timip_modreader))
	{
	  _mm_errno = MMERR_LOADING_SAMPLEINFO;
	  return 0;
	}

      q->samplename = timip_DupStr (s.samplename, 22, 1);
      q->seekpos = 0;
      q->speed = timip_finetune[s.timip_finetune];
      q->length = s.length;
      q->loopstart = s.reppos;
      q->loopend = s.repend;
      q->volume = s.volume;
      if ((s.repend - s.reppos) > 2)
	q->flags |= SF_LOOP;

      if (s.attribute & 1)
	{
	  /* If the sample is 16-bits, convert the length and replen
	     byte-values into sample-values */
	  q->flags |= SF_16BITS;
	  q->length >>= 1;
	  q->loopstart >>= 1;
	  q->loopend >>= 1;
	}

      q++;
    }

  if (!timip_AllocPositions (timip_of.numpos))
    return 0;
  for (t = 0; t < timip_of.numpos; t++)
    timip_of.positions[t] = _mm_read_UBYTE (timip_modreader);
  for (; t < 128; t++)
    _mm_read_UBYTE (timip_modreader);
  if (_mm_eof (timip_modreader))
    {
      _mm_errno = MMERR_LOADING_HEADER;
      return 0;
    }

  if (!timip_AllocTracks ())
    return 0;
  if (!timip_AllocPatterns ())
    return 0;

  timip_of.tracks[0] = MTM_Convert ();	/* track 0 is empty */
  for (t = 1; t < timip_of.numtrk; t++)
    {
      int s;

      for (s = 0; s < 64; s++)
	{
	  mtmtrk[s].a = _mm_read_UBYTE (timip_modreader);
	  mtmtrk[s].b = _mm_read_UBYTE (timip_modreader);
	  mtmtrk[s].c = _mm_read_UBYTE (timip_modreader);
	}

      if (_mm_eof (timip_modreader))
	{
	  _mm_errno = MMERR_LOADING_TRACK;
	  return 0;
	}

      if (!(timip_of.tracks[t] = MTM_Convert ()))
	return 0;
    }

  for (t = 0; t < timip_of.numpat; t++)
    {
      _mm_read_I_UWORDS (pat, 32, timip_modreader);
      for (u = 0; u < timip_of.numchn; u++)
	timip_of.patterns[((long) t * timip_of.numchn) + u] = pat[u];
    }

  /* read comment field */
  if (mh->commentsize)
    if (!timip_ReadLinedComment (mh->commentsize / 40, 40))
      return 0;

  return 1;
}

CHAR *
timip_MTM_LoadTitle (void)
{
  CHAR s[20];

  _mm_fseek (timip_modreader, 4, SEEK_SET);
  if (!_mm_read_UBYTES (s, 20, timip_modreader))
    return NULL;

  return (timip_DupStr (s, 20, 1));
}

/*========== Loader information */

MLOADER timip_load_mtm =
{
  NULL,
  "MTM",
  "MTM (MultiTracker Module editor)",
  timip_MTM_Init,
  timip_MTM_Test,
  timip_MTM_Load,
  timip_MTM_Cleanup,
  timip_MTM_LoadTitle
};

/* ex:set ts=4: */
