package jp.snowgoose.treno.context;

import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Map;

import jp.snowgoose.treno.component.Addon;
import jp.snowgoose.treno.metadata.BindDescriptor;
import jp.snowgoose.treno.util.Maps;
import jp.snowgoose.treno.util.StringUtils;

/**
 * @author snowgoose
 */
public interface ParameterConverter extends Addon {

    <T> T convert(Object value, BindDescriptor bindDesc);

    @SuppressWarnings("unchecked")
    public static class AutoParameterConverter implements ParameterConverter {

        private static final Map<ConverterKey, Converter> converterMap = Maps.newLinkedHashMap();

        static {
            converterMap.put(ConverterKey.get(String.class, Boolean.class), new BooleanConverter());
            converterMap.put(ConverterKey.get(String.class, Character.class),
                    new CharacterConverter());
            converterMap.put(ConverterKey.get(String.class, Float.class), new FloatConverter());
            converterMap.put(ConverterKey.get(String.class, Short.class), new ShortConverter());
            converterMap.put(ConverterKey.get(String.class, Integer.class), new IntConverter());
            converterMap.put(ConverterKey.get(String.class, Long.class), new LongConverter());
            converterMap.put(ConverterKey.get(String.class, BigDecimal.class),
                    new DecimalConverter());
            converterMap.put(ConverterKey.get(String.class, Date.class), new DateConverter());
        }

        public <T> T convert(Object param, BindDescriptor bindDesc) {
            if (param == null) {
                return null;
            }
            Converter converter = getConverter(param.getClass(), bindDesc);
            return (T) converter.convert(bindDesc, param);
        }

        private Converter<?, ?> getConverter(Class<?> fromClass, BindDescriptor bindDesc) {
            ConverterKey key = ConverterKey.get(fromClass, bindDesc.getParameterType());
            if (converterMap.containsKey(key)) {
                return converterMap.get(key);
            } else {
                return new NoneConverter();
            }
        }

    }

    class ConverterKey {

        private Class<?> from;
        private Class<?> to;

        static ConverterKey get(Class<?> from, Class<?> to) {
            return new ConverterKey(from, to);
        }

        ConverterKey(Class<?> from, Class<?> to) {
            this.from = from;
            this.to = to;
        }

        @Override
        public boolean equals(Object other) {
            if (other instanceof ConverterKey) {
                ConverterKey otherKey = (ConverterKey) other;
                return (otherKey.from.equals(from) && otherKey.to.equals(to));
            } else {
                return false;
            }
        }

        @Override
        public int hashCode() {
            return from.hashCode() & to.hashCode();
        }

        public String toString() {
            return from.getCanonicalName() + "_to_" + to.getCanonicalName();
        }
    }

    public interface Converter<F, T> {
        T convert(BindDescriptor bindDesc, F from);
    }

    public class NoneConverter<F, T> implements Converter<F, T> {
        @SuppressWarnings("unchecked")
        public T convert(BindDescriptor bindDesc, F from) {
            return (T) from;
        }
    }

    public abstract class StringToConverter<T> extends NoneConverter<String, T> {
    }

    public class BooleanConverter extends StringToConverter<Boolean> {
        @Override
        public Boolean convert(BindDescriptor bindDesc, String from) {
            return Boolean.valueOf(from);
        }
    }

    public class ShortConverter extends StringToConverter<Short> {
        @Override
        public Short convert(BindDescriptor bindDesc, String from) {
            return Short.valueOf(from);
        }
    }

    public class FloatConverter extends StringToConverter<Float> {
        @Override
        public Float convert(BindDescriptor bindDesc, String from) {
            return Float.valueOf(from);
        }
    }

    public class CharacterConverter extends StringToConverter<Character> {
        @Override
        public Character convert(BindDescriptor bindDesc, String from) {
            if (StringUtils.isEmpty(from)) {
                return null;
            } else {
                return Character.valueOf(from.toCharArray()[0]);
            }
        }
    }

    public class IntConverter extends StringToConverter<Integer> {
        @Override
        public Integer convert(BindDescriptor bindDesc, String from) {
            return Integer.valueOf(from);
        }
    }

    public class LongConverter extends StringToConverter<Long> {
        @Override
        public Long convert(BindDescriptor bindDesc, String from) {
            return Long.valueOf(from);
        }
    }

    public class DecimalConverter extends StringToConverter<BigDecimal> {
        @Override
        public BigDecimal convert(BindDescriptor bindDesc, String from) {
            String format;
            if (StringUtils.isNotEmpty(format = bindDesc.getFormat())) {
                DecimalFormat formatter = new DecimalFormat(format);
                try {
                    return BigDecimal.valueOf((Long) formatter.parse(from));
                } catch (ParseException e) {
                    return new BigDecimal(from);
                }
            } else {
                return new BigDecimal(from);
            }
        }
    }

    public class DateConverter extends StringToConverter<Date> {
        @Override
        public Date convert(BindDescriptor bindDesc, String from) {
            try {
                // TODO replace utility
                return new SimpleDateFormat(bindDesc.getFormat()).parse(from);
            } catch (ParseException e) {
                // TODO log error.
                return null;
            }
        }
    }

}
