package jp.snowgoose.treno.util;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Enumeration;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author snowgoose
 */
public class ResourceUtils {

    private static final Logger log = LoggerFactory.getLogger(ResourceUtils.class);

    private static final List<ResourceFinder> DEFAULT_STRATEGIES = Arrays
            .asList(new ResourceFinder[] { FindResourceStrategies.CLASSPATH,
                    FindResourceStrategies.URL, FindResourceStrategies.FILE });

    private ResourceUtils() {
        // can not instanticate this class.
    }

    public static URL findResource(String name) {
        if (StringUtils.isEmpty(name)) {
            log.debug("resource name is empty.");
            return null;
        }
        if (log.isDebugEnabled()) {
            log.debug("resource name : " + name);
        }
        return findResource(name, DEFAULT_STRATEGIES);
    }

    public static URL findResource(String name, List<ResourceFinder> strategies) {
        URL found;
        for (ResourceFinder strategy : strategies) {
            found = strategy.find(name);
            if (found != null) {
                return found;
            }
        }
        return null;
    }

    public interface ResourceFinder {
        URL find(String name);
    }

    public interface FindResourceStrategies {
        static final ResourceFinder FILE = new ResourceFinder() {
            public URL find(String filename) {
                try {
                    char sep = File.separator.charAt(0);
                    String file = filename.replace(sep, '/');
                    if (file.charAt(0) != '/') {
                        String dir = System.getProperty("user.dir");
                        dir = dir.replace(sep, '/') + '/';
                        if (dir.charAt(0) != '/') {
                            dir = "/" + dir;
                        }
                    }
                    URL fileURL = new URL("file", null, file);
                    log.debug("file resource : " + fileURL);
                    // if (fileURL.openConnection().getContent() == null) {
                    if (new File(file).exists() == false) {
                        log.debug("not avilable file resource.");
                        return null;
                    }
                    if (log.isDebugEnabled()) {
                        log.debug("file resource found. " + fileURL);
                    }
                    return fileURL;
                } catch (MalformedURLException e) {
                    log.debug("can't convert from filename");
                }
                return null;
            }
        };
        static final ResourceFinder CLASSPATH = new ResourceFinder() {
            public URL find(String name) {
                URL resource = Thread.currentThread().getContextClassLoader().getResource(name);
                if (resource != null) {
                    log.debug("classpath resource found." + resource.getPath());
                    return resource;
                } else {
                    log.debug("classpath resource not found.");
                    return null;
                }
            }

        };
        static final ResourceFinder URL = new ResourceFinder() {
            public URL find(String name) {
                try {
                    URL url = new URL(name);
                    log.debug("url resource found." + url.getPath());
                    return url;
                } catch (MalformedURLException e) {
                    return null;
                }
            }
        };
    }

    public static List<URL> findResources(String name) {
        List<URL> found = new ArrayList<URL>();
        URL resource = findResource(name, Arrays.asList(FindResourceStrategies.FILE,
                FindResourceStrategies.URL));
        if (resource != null) {
            found.add(resource);
        }
        try {
            Enumeration<URL> urls = Thread.currentThread().getContextClassLoader().getResources(
                    name);
            while (urls.hasMoreElements()) {
                if (urls != null) {
                    found.add(urls.nextElement());
                }
            }
        } catch (IOException e) {
            return Collections.emptyList();
        }
        return found;
    }
}
