/**********************************************************************
 * conn.h                                                   August 2005
 *
 * KSSLD: An implementation of SSL/TLS in the Linux Kernel
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This file based in part on code from LVS www.linuxvirtualserver.org
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 **********************************************************************/

#ifndef _KSSL_CONN_H
#define _KSSL_CONN_H

#include <linux/net.h>
#include <linux/list.h>
#include <linux/crypto.h>

#include <net/tcp.h>

#include "conn_list.h"
#include "daemon.h"
#include "log.h"
#include "session.h"

#include "types/security_parameters_t.h"
#include "types/record_t.h"
#include "types/ssl2_record_t.h"
#include "types/handshake_t.h"
#include "types/alert_t.h"

#define KSSL_CONN_PT_CLOSE     0x00000001
#define KSSL_CONN_SSL_CLOSE    0x00000002
#define KSSL_CONN_ALL_CLOSE    (KSSL_CONN_PT_CLOSE|KSSL_CONN_SSL_CLOSE)

#define KSSL_CONN_PT           0x00000001
#define KSSL_CONN_SSL          0x00000002

typedef struct kssl_conn kssl_conn_t;

typedef struct {
	content_type_t type;
	union {
		handshake_t handshake;
		change_cipher_spec_t css;
		alert_t alert;
		opaque_t *app_data;
	} data;
} kssl_message_t;

typedef struct {
	struct iovec *iov;
	struct sk_buff **skbv;
	size_t iov_len;
	size_t iov_alloc;
	size_t total_len;
	size_t content_len;
	record_t record;
	ssl2_head_t ssl2_head;
	kssl_conn_t *conn;
	kssl_message_t *msg;
	u32 offset;
	u32 state;
	u32 conn_flag;
	u64 seq;
	struct list_head list;
} kssl_record_t;

typedef struct {
	kssl_session_id_t id;
	certificate_t cert;
	cipher_suite_t cs;
	compression_method_t cm;
} kssl_session_state_t;

typedef enum {
	kssl_aa_null = 0,
	kssl_aa_rsa  = 1,
	kssl_aa_dss  = 2
} kssl_auth_alorithm_t;

typedef enum {
	kssl_kea_anon = 0,
	kssl_kea_rsa  = 1,
	kssl_kea_dh   = 2, /* DH */
	kssl_kea_dhe  = 3, /* Ephemeral DH */
	kssl_kea_dha  = 4  /* Anonymous DH */
} kssl_key_exchange_alorithm_t;

typedef enum {
	kssl_sa_null=0,
	kssl_sa_rsa=1,
	kssl_sa_dss=2
} kssl_signature_alorithm_t;

typedef struct {
	kssl_auth_alorithm_t aa;
	kssl_key_exchange_alorithm_t kea;
	kssl_signature_alorithm_t sa;
	size_t key_limit;
} key_exchange_suite_t;

typedef struct {
	/* opaque_t client_random[32]; In Security Parameters */
        /* opaque_t server_random[32]; In Security Parameters */
	opaque_t *client_mac_secret;
	opaque_t *server_mac_secret;
	opaque_t *client_key;
	opaque_t *server_key;
	uint64_t in_seq;
	uint64_t out_seq;

	/* Extra Stuff */
	protocol_version_t version;
	key_exchange_suite_t kes;
	cipher_suite_t cs;
	opaque_t *client_iv;
	opaque_t *server_iv;
	struct crypto_tfm *hs_digest_sha1;
	struct crypto_tfm *hs_digest_md5;
	struct crypto_tfm *dec;
	struct crypto_tfm *enc;
	struct crypto_tfm *mac;
} kssl_connection_state_t;

struct kssl_conn {
	struct socket *ssl_sock;
	struct socket *pt_sock;
	spinlock_t sock_lock;
	kssl_record_t *ssl_record;
	kssl_record_t *pt_record;
	unsigned int ssl_in_bytes;
	unsigned int ssl_out_bytes;
	unsigned int pt_in_bytes;
	unsigned int pt_out_bytes;
	kssl_daemon_t *daemon;
	security_parameters_t *sec_param_in_pend;
	security_parameters_t *sec_param_in_act;
	security_parameters_t *sec_param_out_pend;
	security_parameters_t *sec_param_out_act;
	kssl_session_state_t sess_state;
	kssl_connection_state_t conn_state;
	u32 flag;
	u32 msg_mask;
	u32 users;
	struct list_head list;
};

kssl_conn_t * kssl_conn_create(kssl_daemon_t *daemon, connection_end_t entity);

int kssl_conn_activate_sec_param_in(kssl_conn_t *conn);

int kssl_conn_activate_sec_param_out(kssl_conn_t *conn);

void __kssl_conn_destroy(kssl_conn_t *conn);

int kssl_conn_cpy_digest(kssl_conn_t *conn, struct crypto_tfm **hs_digest_md5,
		                struct crypto_tfm **hs_digest_sha1);

void __kssl_conn_start_close(kssl_conn_t *conn, unsigned int flag);

#define kssl_conn_destroy(conn) \
do { \
	KSSL_DEBUG(12, "%s(%s:%d): kssl_conn_destroy: %p\n",  \
			__FUNCTION__, __FILE__, __LINE__, conn); \
	__kssl_conn_destroy(conn); \
} while(0)

#define kssl_conn_close(conn, flag) \
do { \
	KSSL_DEBUG(12, "%s(%s:%d): kssl_conn_close: %p 0x%02x\n",  \
			__FUNCTION__, __FILE__, __LINE__, conn, flag); \
	__kssl_conn_start_close(conn, flag); \
} while(0)

#define kssl_conn_get(conn) \
do { \
	conn->users++; \
	KSSL_DEBUG(12, "%s(%s:%d): kssl_conn_get: %u\n", \
			__FUNCTION__, __FILE__, __LINE__, conn->users); \
} while(0)

#define kssl_conn_put(conn) \
do { \
	conn->users--; \
	KSSL_DEBUG(12, "%s(%s:%d): kssl_conn_put: %u\n", \
			__FUNCTION__, __FILE__, __LINE__, conn->users); \
} while(0)

int kssl_conn_accept(kssl_daemon_t *daemon, kssl_conn_t **conn);

int kssl_conn_open_pt(kssl_conn_t *conn);

int kssl_conn_recv(kssl_conn_t *conn);

int kssl_conn_send(kssl_conn_t *conn, struct msghdr *msg, int size);

int kssl_conn_set_cipher_pending(kssl_conn_t *conn, random_t *client_random,
		cipher_suite_t *cs, compression_method_t cm);

int kssl_conn_keying_material_init(kssl_conn_t *conn, u8 *secret, 
		u32 secret_len);

int kssl_conn_keying_material_init_from_master_secret(kssl_conn_t *conn, 
		u8 *master_secret);

#endif /* _KSSL_CONN_H */

