/**********************************************************************
 * core.c                                                   August 2005
 *
 * KSSLD: An implementation of SSL/TLS in the Linux Kernel
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This file based in part on code from LVS www.linuxvirtualserver.org
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 **********************************************************************/

#define __KERNEL_SYSCALLS__

#include <linux/config.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/sched.h>
#include <linux/wait.h>
#include <linux/unistd.h>
#include <linux/errno.h>

#include <asm/uaccess.h>

#include "daemon.h"
#include "conn.h"
#include "conn_list.h"
#include "message.h"
#include "record_list.h"
#include "record.h"
#include "socket.h"
#include "sockopt.h"
#include "kssl_asym.h"
#include "kssl_alloc.h"
#include "kssl_proc.h"
#include "log.h"
#include "util.h"
#include "session.h"

#define KSSL_DAEMON_NAME "kssld"

#ifndef VERSION
#define "unknown"
#endif

static int errno;

static DECLARE_WAIT_QUEUE_HEAD(kssl_wait);
static DECLARE_WAIT_QUEUE_HEAD(kssl_stop_wait);
static pid_t kssl_pid = 0;
static int kssl_stop = 0;


static int
kssl_daemon_accept(kssl_daemon_t *daemon, void *data)
{
	int used = 0;
	int status;

	/* Handle listening socket */
	switch (kssl_daemon_get_mode(daemon)) {
		case kssl_daemon_mode_running:
			if (daemon->sock)
				break;
			kssl_daemon_get_write(daemon);
			status = kssl_socket_inet_open_listen(daemon->vaddr, 
					daemon->vport, &(daemon->sock));
			kssl_daemon_put_write(daemon);
			if (status < 0) {
			KSSL_DEBUG(6, "kssl_daemon_accept: "
					"kssl_socket_inet_open_listen\n");
				KSSL_DEBUG(3, "Could not bind to "
						"socket\n");
				return status;
			}
			break;
		case kssl_daemon_mode_stopped:
		case kssl_daemon_mode_quiescent:
			if (!daemon->sock)
				break;
			kssl_socket_close(daemon->sock);
			kssl_daemon_get_write(daemon);
			daemon->sock = NULL;
			kssl_daemon_put_write(daemon);
			break;
		case kssl_daemon_mode_none:
		case kssl_daemon_mode_unknown:
			break;
	}

	if (kssl_daemon_get_mode(daemon) == kssl_daemon_mode_running) {
		status = kssl_conn_list_accept(daemon, &kssl_conn_list);
		if (status < 0) {
			KSSL_DEBUG(6, "kssl_daemon_accept: "
					"kssl_conn_list_accept\n");
			return status;
		}
		used += status;
	}

	*(int *)data += used;

	return 1;
}


static int 
kssl_main_loop(void)
{
	int status = 0;
	int used;
	struct list_head *c_list;
	struct list_head *t_list;
	kssl_conn_t *conn;
	kssl_record_t *cr;
	u32 pause;

	INIT_LIST_HEAD(&kssl_record_in_list);
	INIT_LIST_HEAD(&kssl_message_in_list);
	INIT_LIST_HEAD(&kssl_record_out_list);
	INIT_LIST_HEAD(&kssl_conn_list);
	INIT_LIST_HEAD(&kssl_conn_close_list);

	used = 0;
	pause = 1;
	while (!kssl_stop) {
		/* There is probably a much better way to handle this */
		if (used) {
			if (!in_softirq())
				cond_resched();
			pause = 1;
			used=0;
		}
		else  {
			if(!(pause & 0xffffc000)) {
				if (!in_softirq())
					cond_resched();
			}
			else {
				kssl_sleep(0, pause);
			}
			if(!(pause & 0x800000))
				pause++;
		}

		status = kssl_daemon_list_for_each(kssl_daemon_accept, &used);
		if (status < 0) {
			KSSL_DEBUG(6, "kssl_main_loop: "
					"kssl_daemon_accept\n");
			break;
		}

		list_for_each_safe(c_list, t_list, &kssl_conn_list) {
			conn = list_entry(c_list, kssl_conn_t, list);
			if (kssl_daemon_get_mode(conn->daemon) == 
					kssl_daemon_mode_stopped) {
				kssl_conn_close(conn, KSSL_CONN_ALL_CLOSE);
				continue;
			}
			status = kssl_conn_recv(conn);
			if (status <= 0) {
				if (status == -EAGAIN)
					continue;
				KSSL_DEBUG(9, "kssl_main_loop: "
						"kssl_conn_recv: %d\n", 
						status);
			}
			used++;
			if (!in_softirq())
				cond_resched();
		}

		list_for_each_safe(c_list, t_list, &kssl_record_in_list) {
			cr = list_entry(c_list, kssl_record_t, list);
			if (kssl_daemon_get_mode(cr->conn->daemon) == 
					kssl_daemon_mode_stopped) {
				kssl_record_destroy(cr);
				continue;
			}
			status = kssl_record_process_in(cr);
			KSSL_DEBUG(12, "core: processed in record: %d\n",
					status);
			used++;
			if (!in_softirq())
				cond_resched();
		}

		list_for_each_safe(c_list, t_list, &kssl_message_in_list) {
			cr = list_entry(c_list, kssl_record_t, list);
			if (kssl_daemon_get_mode(cr->conn->daemon) == 
					kssl_daemon_mode_stopped) {
				kssl_record_destroy(cr);
				continue;
			}
			status = kssl_message_process_in(cr);
			KSSL_DEBUG(12, "core: processed in message: %d\n",
						status);
			used++;
			if (!in_softirq())
				cond_resched();
		}

		list_for_each_safe(c_list, t_list, &kssl_record_out_list) {
			cr = list_entry(c_list, kssl_record_t, list);
			if (kssl_daemon_get_mode(cr->conn->daemon) == 
					kssl_daemon_mode_stopped) {
				kssl_record_destroy(cr);
				continue;
			}
			status = kssl_record_process_out(cr);
			KSSL_DEBUG(12, "core: processed out record: %d\n",
					status);
			if (status == -EAGAIN)
				/* This needs to be handled on
				 * a per-connection basis */
				break;
			used++;
			if (!in_softirq())
				cond_resched();
			break;
		}

		kssl_conn_list_close(&kssl_conn_close_list);
	}

	kssl_record_list_clear(&kssl_message_in_list);
	kssl_record_list_clear(&kssl_record_out_list);
	kssl_record_list_clear(&kssl_record_in_list);
	kssl_conn_list_close(&kssl_conn_close_list);
	kssl_conn_list_close(&kssl_conn_list);

	kssl_alloc_dump();
	kssl_alloc_cleanup();

	return status > 0 ? 0 : status;
}


static int 
kssl_thread(void *startup)
{
	int status = 0;

	DECLARE_WAITQUEUE(wait, current);
	mm_segment_t mm;
	
	/* MOD_INC_USE_COUNT; */
	daemonize();

	mm = get_fs();
	set_fs(KERNEL_DS);

	sprintf(current->comm, KSSL_DAEMON_NAME);

//	spin_lock_irq(&current->sighand->siglock);
	spin_lock_irq(&current->sigmask_lock);
	siginitsetinv(&current->blocked, 0);
//	recalc_sigpending();
	recalc_sigpending(current);
	spin_unlock_irq(&current->sigmask_lock);

	add_wait_queue(&kssl_wait, &wait);

	kssl_pid = current->pid;
	complete((struct completion *)startup);

	/* Run some code here */
	status = kssl_main_loop();

	remove_wait_queue(&kssl_wait, &wait);

	kssl_pid = 0;

	set_fs(mm);
	/* MOD_DEC_USE_COUNT; */

	/* kssl_stop = 0; */
	wake_up(&kssl_stop_wait);

	return status;
}


static int 
kssl_daemon_thread(void *startup)
{
	int status;

	status = kernel_thread(kssl_thread, startup, 0);
	if (status < 0)
		return status;

	return 0;
}


static int 
kssl_start_thread(void)
{
	DECLARE_COMPLETION(startup);
	pid_t pid;
	int status;

	if (kssl_pid)
		return -EEXIST;

	pid = kernel_thread(kssl_daemon_thread, &startup, 0);
	if (pid < 0)
		return pid;

	status = waitpid(pid, NULL, __WCLONE);
	if (status != pid)
		return pid<0?pid:-EINVAL;

	wait_for_completion(&startup);

	return 0;
}


static int 
kssl_stop_thread(void)
{
	DECLARE_WAITQUEUE(wait, current);

	if (!kssl_pid)
		return -ESRCH;

	__set_current_state(TASK_UNINTERRUPTIBLE);
	add_wait_queue(&kssl_stop_wait, &wait);

	/* Set some flags here so the thread will stop itself */
	kssl_stop = 1;

	wake_up(&kssl_wait);
	schedule();
	__set_current_state(TASK_RUNNING);
	remove_wait_queue(&kssl_stop_wait, &wait);

	return 0;
}


static int __init 
kssl_init(void)
{
	int status;

	printk(KERN_NOTICE "kssld: (C) 2005 NTT COMWARE Corporation "
			"Version %s. Build %s %s\n", VERSION, 
			__DATE__, __TIME__);

	status = kssl_daemon_init();
	if (status < 0) {
		KSSL_DEBUG(6, "kssl_init: kssl_daemon_init\n");
		return status;
	}

	status = kssl_asym_init();
	if (status < 0) {
		KSSL_DEBUG(6, "kssl_init: kssl_asym_init\n");
		goto error_asym;
	}

	status = kssl_ctl_init();
	if (status < 0) {
		KSSL_DEBUG(6, "kssl_init: kssl_ctl_init\n");
		goto error_ctl;
	}

	status = kssl_proc_init();
	if (status < 0) {
		KSSL_DEBUG(6, "kssl_init: kssl_proc_init\n");
		goto error_proc;
	}

	status = kssl_session_init();
	if (status < 0) {
		KSSL_DEBUG(6, "kssl_init: kssl_start_thread\n");
		goto error_session;
	}

	status = kssl_start_thread();
	if (status < 0) {
		KSSL_DEBUG(6, "kssl_init: kssl_start_thread\n");
		goto error_thread;
	}

	return 0;

error_thread:
	kssl_session_cleanup();
error_session:
	 kssl_proc_destroy();
error_proc:
	 kssl_ctl_cleanup();
error_ctl:
	 kssl_asym_cleanup();
error_asym:
	 kssl_daemon_cleanup();
	 
	 return status;
}


static void 
__exit kssl_cleanup(void)
{
	kssl_stop_thread();

	kssl_session_cleanup();
	kssl_proc_destroy();
	kssl_ctl_cleanup();
	kssl_asym_cleanup();
	kssl_daemon_cleanup();
}


module_init(kssl_init);
module_exit(kssl_cleanup);
MODULE_LICENSE("GPL");
MODULE_AUTHOR("NTT COMWARE Corporation");
MODULE_DESCRIPTION("SSL to plain-text proxy");
