/**********************************************************************
 * base64.c                                                 August 2005
 *
 * ASYM: An implementation of Asymetric Cryptography in the Linux Kernel
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This file based in part on code from LVS www.linuxvirtualserver.org
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 **********************************************************************/

#include "base64.h"

#ifdef __KERNEL__
#include <linux/slab.h>
#else
#include "compat.h"
#endif


/**
 * code characters for values 0..63 
 */ 
char base64_fwdtab[64+1+1] = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=";
char *base64_revtab=NULL;

/** 
 * returns an array of base64-encoded characters to represent the 
 * passed data array. 
 * 
 * @param data the array of chars to encode 
 */ 
void base64_encode(char *out, const char *data, size_t inLen) { 
  size_t i, index, val;
  int quad, trip;

  base64_init();

  /* 
   * 3 chars encode to 4 chars. Output is always an even 
   * multiple of 4 characters. 
   */ 
  for (i=0,index=0; i<inLen; i+=3,index+=4) { 
    quad = 0;
    trip = 0;
  
    val = (0xff & data[i]); 
    val <<= 8; 
    if ((i+1) < inLen) { 
      val |= (0Xff & data[i+1]); 
      trip = 1;
    } 
    val <<= 8; 
    if ((i+2) < inLen) { 
      val |= (0xff & data[i+2]); 
      quad = 1;
    } 
    out[index+3] = base64_fwdtab[(quad? (val & 0x3f): 64)]; 
    val >>= 6; 
    out[index+2] = base64_fwdtab[(trip? (val & 0x3f): 64)]; 
    val >>= 6; 
    out[index+1] = base64_fwdtab[val & 0x3f]; 
    val >>= 6; 
    out[index+0] = base64_fwdtab[val & 0x3f]; 
  }
  out[index] = 0;

  #ifdef BASE64_SECURE
    i = index = val = quad = trip = 0;
  #endif
} 
  
/** 
 * Returns an array of chars which were encoded in the passed 
 * character array. 
 * 
 * @param data the array of base64-encoded characters 
 */ 
void base64_decode(char *out, size_t *outLen, const char *data, size_t inLen) {
  int shift = 0; /* # of excess bits stored in accum  */
  int index = 0; 
  size_t ix;
  int accum = 0; /* excess bits */
  char value;

  base64_init();

  for (ix=0; ix<inLen; ix++) { 
    value = base64_revtab[ data[ix] & 0xff ];
    if (value != (char)0xff) { /* skip over non-code */
      accum <<= 6; /* bits shift up by 6 each time thru */
      accum |= value; /* at the bottom. */
      shift += 6; /* loop, with new bits being put in */
      if ( shift >= 8 ) { /* whenever there are 8 or more shifted in, */
        shift -= 8; /* write them out (from the top, leaving any */
        out[index++] = (char) ((accum >> shift) & 0xff); /* excess at the bottom for next iteration. */
      }
    }
  }

  *outLen = index;
  
  #ifdef BASE64_SECURE
    shift = index = ix = accum = value = 0;
  #endif
} 
  
/* 
 * lookup table for converting base64 characters to value in range 0..63 
 */
static int __base64_init(void) {
  int i;

  base64_revtab = (char*) kmalloc(256, GFP_KERNEL);
  if(!base64_revtab)
    return(-ENOMEM);

  memset(base64_revtab, (char)0xff, 256);
  for (i='A'; i<='Z'; i++) base64_revtab[i] = (char)(     i - 'A'); 
  for (i='a'; i<='z'; i++) base64_revtab[i] = (char)(26 + i - 'a'); 
  for (i='0'; i<='9'; i++) base64_revtab[i] = (char)(52 + i - '0'); 
  base64_revtab['+'] = 62; 
  base64_revtab['/'] = 63; 

  return(0);
} 


int base64_init(void) {
  if(base64_revtab)
    return(0);

  return(__base64_init());
} 
