/**********************************************************************
 * unsx_e.c                                                 August 2005
 *
 * ASYM: An implementation of Asymetric Cryptography in the Linux Kernel
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This file based in part on code from LVS www.linuxvirtualserver.org
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 **********************************************************************/

#include <linux/kernel.h>

#ifdef __KERNEL__
#include <linux/types.h>
#include <linux/random.h>
#include <linux/errno.h>
#include <linux/slab.h>
#else
#include "compat.h"
#endif

#include "unsx.h"

static void printUnsx(unsx *a, int len) {
      	int i;
	char *str;
	char *p;

	str = (char *)kmalloc(len * 9 + 1, GFP_KERNEL);
	if(!str) {
		return;
	}

	p = str;
	for (i=0; i<unsx_countArr(a,len); i++) {
		sprintf(p, "%.8x ", a[i]);
		p += 9;
	}
	*(p-1) = '\n';
	*(p) = '\0';
	
	printk(KERN_DEBUG "%s", str);
	kfree(str);
}



static void testInt(const char *msg, int x, int y) {
  int r;
  printk(KERN_DEBUG "%s: %u == %u  [%s]\n", msg, x, y, 
		  (r=x==y) ? "PASSED" : "FAILED");
}

static void testHex(const char *msg, long long x, long long y) {
  int r;
  printk(KERN_DEBUG "%s: %.8llx == %.8llx  [%s]\n", msg, x, y, 
		  (r=x==y) ? "PASSED" : "FAILED");
}

static void testUNSXlow(const char *msg, unsx l, unsx *a, int len) {
  int r;
  printk(KERN_DEBUG "%s: {%u}  [%s]\n", msg, l, 
		  (r=unsx_isLow(a,l,len)) ? "PASSED" : "FAILED");
}

static int unsx_testsuite(int bitLength) {
  unsx a,b,c,d;

  testInt("sizeof(uns8)", sizeof(uns8), 1);
  testInt("sizeof(uns16)", sizeof(uns16), 2);
  testInt("sizeof(uns32)", sizeof(uns32), 4);
  testInt("sizeof(uns64)", sizeof(uns64), 8);
  testInt("sizeof(unsx)", sizeof(unsx), 4);
  testInt("sizeof(uns2x)", sizeof(uns2x), 8);
  
  testHex("uns8_Max", 0xff, uns8_Max);
  testHex("uns16_Max", 0xffff, uns16_Max);
  testHex("uns32_Max", 0xffffffff, uns32_Max);
  testHex("uns64_Max", 0xffffffffffffffff, uns64_Max);
  testHex("unsx_Max", 0xffffffff, unsx_Max);
  testHex("uns2x_Max", 0xffffffffffffffff, uns2x_Max);
  
  testInt("uns8_log", 3, uns8_log);
  testInt("uns16_log", 4, uns16_log);
  testInt("uns32_log", 5, uns32_log);
  testInt("uns64_log", 6, uns64_log);
  testInt("unsx_log", 5, unsx_log);
  testInt("uns2x_log", 6, uns2x_log);
  
  testInt("unsx_Bit_Length", 32, unsx_Bit_Length);

  testInt("MAX(1,0)", 1, MAX(1,0));
  testInt("MAX(0,1)", 1, MAX(0,1));
  testInt("MAX(2,0)", 2, MAX(2,0));
  testInt("MAX(0,2)", 2, MAX(0,2));
  
  testInt("UNSX_LENGTH", bitLength/32, UNSX_LENGTH);

#ifdef UNSX_SECURE
  testInt("UNSX_SECURE", 1, 1);
#else
  testInt("UNSX_SECURE", 0, 0);
#endif
  
/*
#ifdef UNSX_MALLOC
  testInt("UNSX_MALLOC", 1, 1);
#else
  testInt("UNSX_MALLOC", 1, 0);
#endif
//*/
  
#ifdef UNSX_MEMLIB
  testInt("UNSX_MEMLIB", 1, 1);
#else
  testInt("UNSX_MEMLIB", 1, 0);
#endif

  testInt("unsx_Trial_Prime_size", 2*3*5*7*11, unsx_Trial_Prime_size);
  testInt("unsx_Trial_Prime_Sets", 80, unsx_Trial_Prime_Sets);
  testInt("unsx_Trial_Prime_Sets_Start", 11, unsx_Trial_Prime_Sets_Start);

  testUNSXlow("unsx_ZERO", 0, unsx_ZERO, UNSX_LENGTH);
  testUNSXlow("unsx_ONE", 1, unsx_ONE, UNSX_LENGTH);
  testUNSXlow("unsx_TWO", 2, unsx_TWO, UNSX_LENGTH);
  testUNSXlow("unsx_THREE", 3, unsx_THREE, UNSX_LENGTH);
  testUNSXlow("unsx_FIVE", 5, unsx_FIVE, UNSX_LENGTH);
  testUNSXlow("unsx_SEVEN", 7, unsx_SEVEN, UNSX_LENGTH);
  testUNSXlow("unsx_ELEVEN", 11, unsx_ELEVEN, UNSX_LENGTH);
  testUNSXlow("unsx_THIRTEEN", 13, unsx_THIRTEEN, UNSX_LENGTH);
  testUNSXlow("unsx_TWENTYTHREE", 23, unsx_TWENTYTHREE, UNSX_LENGTH);
  testUNSXlow("unsx_1662803", 1662803, unsx_1662803, UNSX_LENGTH);
 
  testInt("UNSX_TESTBIT", 1, UNSX_TESTBIT(unsx_TWO, 1));
  UNSX_SETBIT(unsx_TWO, 100);
  testInt("UNSX_TESTBIT", 1, UNSX_TESTBIT(unsx_TWO, 100));
  UNSX_CLEARBIT(unsx_TWO, 100);
  testInt("UNSX_TESTBIT", 0, UNSX_TESTBIT(unsx_TWO, 100));

  a = 0x80000000;
  b = 0x80000000;

  /* {c,d} == a * b */
  UNSX_DOUBLEMULT(c,d, a,b);
  testInt("UNSX_DOUBLEMULT", 0x40000000, c);
  testInt("UNSX_DOUBLEMULT", 0x00000000, d);

  UNSX_DOUBLEDIVI(a,c,d,b);
  testInt("UNSX_DOUBLEDIVI", 0x80000000, a);

  testInt("UNSX_ISODD", 1, UNSX_ISODD(unsx_THREE, UNSX_LENGTH));
  testInt("UNSX_ISODD", 0, UNSX_ISODD(unsx_TWO, UNSX_LENGTH));
  testInt("UNSX_ISEVEN", 0, UNSX_ISEVEN(unsx_THREE, UNSX_LENGTH));
  testInt("UNSX_ISEVEN", 1, UNSX_ISEVEN(unsx_TWO, UNSX_LENGTH));
  
  return(0);
}


int unsx_test(int bits)
{
	int i, j;
	int status = -ENOMEM;
	int chunks;
	UNSX_NEW(a, bits);

	chunks = bits/unsx_Bit_Length;

	if(!a)
		goto leave;

  	printk(KERN_DEBUG "size=%u\n", chunks);
	unsx_init(bits);
	unsx_testsuite(bits);
	unsx_setLow(a, 9, chunks);
	
	for (i=0; i <= 2*chunks; i++) {
		for (j=0; j < i/2; j++) {
			get_random_bytes(&(a[j]), sizeof(a[j]));
			a[j] *= 0x01010101;
		}
		printk(KERN_DEBUG "finding %u-bit prime\n", j*unsx_Bit_Length);
		unsx_nextPrime(a, chunks);
		if ( !unsx_isPrimePRPn(a, unsx_TWO, chunks) ) {
			printk(KERN_DEBUG "a is not prime!  "
					"unsx_nextPrime failed!\n");
			goto leave;
		}
		printUnsx(a, chunks);
		yield();
	}

	status = 0;
leave:
	UNSX_FREE(a, chunks);
	return(status);
}

