/**********************************************************************
 * ssl3mac.c                                                August 2005
 *
 * KSSLD: An implementation of SSL/TLS in the Linux Kernel
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This file based in part on code from LVS www.linuxvirtualserver.org
 * SSL3MAC: Mac used in SSL3 http://wp.netscape.com/eng/ssl3/draft302.txt
 *
 * Based heavily on ssl3mac.c
 * Copyright (c) 2002 James Morris <jmorris@intercode.com.au>
 *
 * Which in turn has its ssl3mac implementation is derived from USAGI.
 * Copyright (c) 2002 Kazunori Miyazawa <miyazawa@linux-ipv6.org> / USAGI
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 **********************************************************************/

#include <linux/crypto.h>
#include <linux/mm.h>
#include <linux/highmem.h>
#include <linux/slab.h>
#include <asm/scatterlist.h>

static
int crypto_alloc_ssl3mac_block(struct crypto_tfm *tfm)
{
	int ret = 0;

	BUG_ON(!crypto_tfm_alg_blocksize(tfm));
	
	tfm->crt_digest.dit_hmac_block = kmalloc(crypto_tfm_alg_blocksize(tfm),
	                                         GFP_KERNEL);
	if (tfm->crt_digest.dit_hmac_block == NULL)
		ret = -ENOMEM;

	return ret;
		
}

void crypto_free_ssl3mac_block(struct crypto_tfm *tfm)
{
	if (tfm->crt_digest.dit_hmac_block) {
		kfree(tfm->crt_digest.dit_hmac_block);
		tfm->crt_digest.dit_hmac_block = NULL;
	}
}

/* The SSL3 specification is a bit broken in
 * that it specifies padding of only 40 bytes for
 * sha1. 64-20 = 44 would be much more logical. */
static inline unsigned int crypto_ssl3mac_padsize(struct crypto_tfm *tfm)
{
	if (!strcmp(crypto_tfm_alg_name(tfm), "sha1"))
		return 40;
	
	/* Not needed as this is the same
	 * as crypto_tfm_alg_blocksize() - crypto_tfm_alg_digestsize()
	if(!strcmp(crypto_tfm_alg_name(tfm), "md5")) {
		return(48);
	}
	*/

	return crypto_tfm_alg_blocksize(tfm) - 
			crypto_tfm_alg_digestsize(tfm);
}

void crypto_ssl3mac_init(struct crypto_tfm *tfm, u8 *key, unsigned int *keylen)
{
	struct scatterlist tmp;
	char *ipad = tfm->crt_digest.dit_hmac_block;
	unsigned int padsize;

	crypto_digest_init(tfm);


	if (*keylen) {
		tmp.page = virt_to_page(key);
		tmp.offset = ((long)key & ~PAGE_MASK);
		tmp.length = *keylen;
		crypto_digest_update(tfm, &tmp, 1);
	}

	padsize = crypto_ssl3mac_padsize(tfm);
	memset(ipad, 0x36, padsize);
	tmp.page = virt_to_page(ipad);
	tmp.offset = ((long)ipad & ~PAGE_MASK);
	tmp.length = padsize;
	crypto_digest_update(tfm, &tmp, 1);
}

void crypto_ssl3mac_update(struct crypto_tfm *tfm,
                        struct scatterlist *sg, unsigned int nsg)
{
	crypto_digest_update(tfm, sg, nsg);
}

void crypto_ssl3mac_final(struct crypto_tfm *tfm, u8 *key,
                       unsigned int *keylen, u8 *out)
{
	struct scatterlist tmp;
	char *opad = tfm->crt_digest.dit_hmac_block;
	unsigned int padsize;

	crypto_digest_final(tfm, out);
	crypto_digest_init(tfm);

	if (*keylen) {
		tmp.page = virt_to_page(key);
		tmp.offset = ((long)key & ~PAGE_MASK);
		tmp.length = *keylen;
		crypto_digest_update(tfm, &tmp, 1);
	}

	padsize = crypto_ssl3mac_padsize(tfm);
	memset(opad, 0x5c, padsize);
	tmp.page = virt_to_page(opad);
	tmp.offset = ((long)opad & ~PAGE_MASK);
	tmp.length = padsize;
	crypto_digest_update(tfm, &tmp, 1);
	
	tmp.page = virt_to_page(out);
	tmp.offset = ((long)out & ~PAGE_MASK);
	tmp.length = crypto_tfm_alg_digestsize(tfm);
	crypto_digest_update(tfm, &tmp, 1);

	crypto_digest_final(tfm, out);
}

void crypto_ssl3mac(struct crypto_tfm *tfm, u8 *key, unsigned int *keylen,
                 struct scatterlist *sg, unsigned int nsg, u8 *out)
{
	crypto_ssl3mac_init(tfm, key, keylen);
	crypto_ssl3mac_update(tfm, sg, nsg);
	crypto_ssl3mac_final(tfm, key, keylen, out);
}

static int __init init(void)
{
	return(0);
}

static void __exit fini(void)
{
	return;
}

module_init(init);
module_exit(fini);

MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("SSL3 MAC Algorithm");
MODULE_AUTHOR("NTT COMWARE Corporation");

EXPORT_SYMBOL_GPL(crypto_ssl3mac_init);
EXPORT_SYMBOL_GPL(crypto_ssl3mac_update);
EXPORT_SYMBOL_GPL(crypto_ssl3mac_final);
EXPORT_SYMBOL_GPL(crypto_ssl3mac);

