/**********************************************************************
 * record_t.h                                               August 2005
 *
 * KSSLD: An implementation of SSL/TLS in the Linux Kernel
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This file based in part on code from LVS www.linuxvirtualserver.org
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 * Contribution from other Authors:
 * 
 * Kfishd activity patch - The Rasterman <raster@rasterman.com>
 *
 **********************************************************************/

#ifndef _RECORD_T_H
#define _RECORD_T_H

#ifdef __KERNEL__
#include <linux/kernel.h>
#include <linux/string.h>
#else 
#include <string.h>
#endif

#include "types/base_t.h"


/**********************************************************************
 * Specialised Types
 * Used in record (and other) messages
 **********************************************************************/

typedef struct {
	u8 major;
	u8 minor;
} protocol_version_t;

#define PROTOCOL_VERSION_NLEN 2

/* buf must be at least 2 bytes long */
static inline void protocol_version_from_buf(protocol_version_t *p, 
		const u8 *buf)
{
	p->major = *(buf);
	p->minor = *(buf+1);
}

/* buf must be at least 2 bytes long */
static inline void protocol_version_to_buf(const protocol_version_t *p, 
		u8 *buf)
{
	*(buf) = p->major;
	*(buf+1) = p->minor;
}


typedef struct {
	opaque_t *content;
	opaque_t *mac;
} generic_stream_cipher_t;


typedef struct {
	opaque_t *content;
	opaque_t *mac;
	u8 *padding;
	u8 padding_length;
} generic_block_cipher_t;


/**********************************************************************
 * Record
 **********************************************************************/

typedef enum {
	ct_change_cipher_spec=20,
	ct_alert=21,
	ct_handshake=22,
	ct_application_data=23,
        ct_ssl2=254, /* Fudge for SSL2 headers, not sent over the wire */
	ct_last=255
} content_type_t;



typedef struct {
	content_type_t type;
	protocol_version_t version;
	u16 len;
} tls_head_t;


/* Length of the header on the wire */
#define TLS_HEAD_NLEN 5

/* Maximum ammount of plaintext */
#define TLS_MAX_DATA 16383 /* 2^14 -1 */


/* buf must be at least 5 bytes long */
static inline void tls_head_from_buf(tls_head_t *head, const u8 *buf)
{
	u16 len;

	head->type = *buf;
	protocol_version_from_buf(&(head->version), buf+1);
	memcpy(&len, buf+3, sizeof(u16));
	head->len = ntohs(len);
}


/* buf must be at least 5 bytes long */
static inline void tls_head_to_buf(const tls_head_t *head, u8 *buf)
{
	u16 len;

	*buf = head->type;
	protocol_version_to_buf(&(head->version), buf+1);
	len = htons(head->len);
	memcpy(buf+3, &len, sizeof(u16));
}


typedef struct {
	opaque_t *fragment;
} tls_plain_text_body_t;


typedef tls_plain_text_body_t tls_compressed_body_t;


typedef union {
	generic_stream_cipher_t stream;
	generic_block_cipher_t block;
} tls_cipher_text_body_t;


typedef struct {
	tls_head_t head;
	union {
		tls_plain_text_body_t plain_text;
		tls_compressed_body_t compressed;
		tls_cipher_text_body_t cipher_text;
	} body;
} record_t;

#define RECORD_NLEN(record) ( (record)->head.len + TLS_HEAD_NLEN )

#endif /* _RECORD_T_H */

